package ru.yandex.passport.familypay.backend;

import java.util.List;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxFamilyInfo;
import ru.yandex.blackbox.BlackboxFamilyInfoRequest;
import ru.yandex.blackbox.BlackboxFamilyMember;
import ru.yandex.blackbox.BlackboxNotFoundException;
import ru.yandex.blackbox.BlackboxPhoneAttributeType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.client.pg.SqlQuery;

public abstract class FamilyMemberChangeEvent extends FamilyChangeEventBase {
    private static final SqlQuery DELETE_FAMILY_MEMBERS =
        new SqlQuery("delete-family-members.sql", FamilyChangeHandler.class);
    private static final SqlQuery UPDATE_FAMILY_MEMBERS =
        new SqlQuery("update-family-members.sql", FamilyChangeHandler.class);

    protected FamilyMemberChangeEvent(final String familyId, final long uid) {
        super(familyId, uid);
    }

    @Override
    protected void familyNotFound(
        final FamilypayCallback<? super Boolean> callback,
        final String eventId)
    {
        BlackboxClient blackboxClient =
            callback.context().server().blackboxClient().adjust(
                callback.context().session().context());
        blackboxClient.familyInfo(
            new BlackboxFamilyInfoRequest(familyId),
            callback.context().session().listener().createContextGeneratorFor(
                blackboxClient),
            new FamilyInfoCallback(callback, eventId));
    }

    @Override
    protected void familyFound(
        final FamilypayCallback<? super Boolean> callback,
        final String eventId,
        final Family family)
    {
        BlackboxClient blackboxClient =
            callback.context().server().blackboxClient().adjust(
                callback.context().session().context());
        blackboxClient.familyInfo(
            new BlackboxFamilyInfoRequest(familyId),
            callback.context().session().listener().createContextGeneratorFor(
                blackboxClient),
            new FamilypayFamilyInfoCallback(callback, eventId, family));
    }

    // All three following methods called only for families without familypay

    // Family wasn't found in blackbox
    protected abstract void blackboxFamilyNotFound(
        FamilypayCallback<? super Boolean> callback,
        String eventId);

    // Family was found in blackbox, but uid is not longer a family member
    protected abstract void blackboxFamilyFoundWithoutMember(
        FamilypayCallback<? super Boolean> callback,
        String eventId,
        long adminUid);

    // Family was found in blackbox and uid is a family member
    protected abstract void blackboxFamilyFoundWithMember(
        FamilypayCallback<? super Boolean> callback,
        String eventId,
        long adminUid);

    private class FamilyInfoCallback
        extends AbstractFilterFamilypayCallback<BlackboxFamilyInfo, Boolean>
    {
        private final String eventId;

        FamilyInfoCallback(
            final FamilypayCallback<? super Boolean> callback,
            final String eventId)
        {
            super(callback);
            this.eventId = eventId;
        }

        @Override
        public void completed(final BlackboxFamilyInfo familyInfo) {
            List<BlackboxFamilyMember> members = familyInfo.members();
            int size = members.size();
            for (int i = 0; i < size; ++i) {
                if (uid == members.get(i).uid()) {
                    blackboxFamilyFoundWithMember(
                        callback,
                        eventId,
                        familyInfo.adminUid());
                    return;
                }
            }
            blackboxFamilyFoundWithoutMember(
                callback,
                eventId,
                familyInfo.adminUid());
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof BlackboxNotFoundException) {
                blackboxFamilyNotFound(callback, eventId);
            } else {
                super.failed(e);
            }
        }
    }

    private static class FamilypayFamilyInfoCallback
        extends AbstractFilterFamilypayCallback<BlackboxFamilyInfo, Boolean>
    {
        private final String eventId;
        private final Family family;

        FamilypayFamilyInfoCallback(
            final FamilypayCallback<? super Boolean> callback,
            final String eventId,
            final Family family)
        {
            super(callback);
            this.eventId = eventId;
            this.family = family;
        }

        @Override
        public void completed(final BlackboxFamilyInfo familyInfo) {
            List<BlackboxFamilyMember> members = familyInfo.members();
            int size = members.size();
            Long[] uids = new Long[size];
            for (int i = 0; i < size; ++i) {
                uids[i] = members.get(i).uid();
            }
            Tuple tuple = Tuple.tuple();
            tuple.addArrayOfLong(uids);
            tuple.addString(family.familyInfo().familyId());
            callback.context().server().pgClient().executeOnMaster(
                DELETE_FAMILY_MEMBERS,
                tuple,
                callback.context().session().listener(),
                new DeleteFamilyMembersCallback(
                    callback,
                    eventId,
                    family,
                    familyInfo));
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof BlackboxNotFoundException) {
                // Race condition, family already deleted in Blackbox, but not
                // in Familypay backend, skip this event
                // Maybe in later events there will be family delete
                callback.context().session().logger().warning(
                    "Family " + family.familyInfo().familyId() + " has gone");
                callback.completed(Boolean.FALSE);
            } else {
                super.failed(e);
            }
        }
    }

    private static class DeleteFamilyMembersCallback
        extends AbstractFilterFamilypayCallback<RowSet<Row>, Boolean>
    {
        private final String eventId;
        private final Family family;
        private final BlackboxFamilyInfo familyInfo;

        DeleteFamilyMembersCallback(
            final FamilypayCallback<? super Boolean> callback,
            final String eventId,
            final Family family,
            final BlackboxFamilyInfo familyInfo)
        {
            super(callback);
            this.eventId = eventId;
            this.family = family;
            this.familyInfo = familyInfo;
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            callback.context().session().logger().info(
                "Rows affected by members delete: " + rowSet.rowCount());
            List<BlackboxFamilyMember> members = familyInfo.members();
            int size = members.size();
            long[] uids = new long[size - 1];
            for (int i = 1; i < size; ++i) {
                uids[i - 1] = members.get(i).uid();
            }
            BlackboxClient blackboxClient =
                callback.context().server().blackboxClient().adjust(
                    callback.context().session().context());
            blackboxClient.userinfo(
                new BlackboxUserinfoRequest(members.get(0).uid(), uids)
                    .sid("smtp")
                    .phoneAttributes(BlackboxPhoneAttributeType.IS_SECURED),
                callback.context().session().listener()
                    .createContextGeneratorFor(blackboxClient),
                new UserinfosCallback(
                    callback,
                    eventId,
                    family));
        }
    }

    private static class UserinfosCallback
        extends AbstractFilterFamilypayCallback
            <List<BlackboxUserinfo>, Boolean>
    {
        private final String eventId;
        private final Family family;

        UserinfosCallback(
            final FamilypayCallback<? super Boolean> callback,
            final String eventId,
            final Family family)
        {
            super(callback);
            this.eventId = eventId;
            this.family = family;
        }

        @Override
        public void completed(final List<BlackboxUserinfo> userinfos) {
            int size = userinfos.size();
            Long[] uids = new Long[size];
            Boolean[] usersPhoneStatus = new Boolean[size];
            for (int i = 0; i < size; ++i) {
                BlackboxUserinfo userinfo = userinfos.get(i);
                uids[i] = userinfo.uid();
                usersPhoneStatus[i] = userinfo.hasSecurePhone();
            }
            Tuple tuple = Tuple.tuple();
            tuple.addString(family.familyInfo().familyId());
            tuple.addArrayOfLong(uids);
            tuple.addArrayOfBoolean(usersPhoneStatus);
            callback.context().server().pgClient().executeOnMaster(
                UPDATE_FAMILY_MEMBERS,
                tuple,
                callback.context().session().listener(),
                new UpdateCallback(
                    callback,
                    eventId,
                    family));
        }
    }

    private static class UpdateCallback
        extends AbstractFilterFamilypayCallback<RowSet<Row>, Boolean>
    {
        private final String eventId;
        private final Family family;

        UpdateCallback(
            final FamilypayCallback<? super Boolean> callback,
            final String eventId,
            final Family family)
        {
            super(callback);
            this.eventId = eventId;
            this.family = family;
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            callback.context().session().logger().info(
                "Rows affected by members update: " + rowSet.rowCount());
            callback.completed(Boolean.TRUE);
            FamilyHandler.findFamily(
                family.familyInfo().familyId(),
                new FamilyChangeNotifyCallback(
                    callback.context(),
                    family,
                    eventId));
        }
    }
}

