package ru.yandex.passport.familypay.backend;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpException;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxFamilyInfo;
import ru.yandex.blackbox.BlackboxFamilyInfoRequest;
import ru.yandex.blackbox.BlackboxFamilyMember;
import ru.yandex.blackbox.BlackboxNotFoundException;
import ru.yandex.blackbox.BlackboxPhoneAttributeType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.client.pg.SqlQuery;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ForbiddenException;
import ru.yandex.http.util.UnauthorizedException;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.passport.familypay.backend.config.ImmutableServiceConfig;

public class FamilyStartHandler extends FamilyHandlerBase {
    private static final SqlQuery ADD_FAMILY =
        new SqlQuery("add-family.sql", FamilyStartHandler.class);

    public FamilyStartHandler(final FamilypayBackend server) {
        super(server, false);
    }

    @Override
    protected void handle(
        final String familyId,
        final JsonObject payload,
        final RequestContext context)
        throws HttpException, JsonException
    {
        handle(
            familyId,
            payload,
            context,
            false);
    }

    protected LimitsInfo parseLimits(final JsonObject payload)
        throws JsonException
    {
        return LimitsInfo.fromJson(
            Objects.requireNonNullElse(
                payload.get("defaultLimits").asMapOrNull(),
                JsonMap.EMPTY));
    }

    @SuppressWarnings("ReferenceEquality")
    protected void handle(
        final String familyId,
        final JsonObject payload,
        final RequestContext context,
        final boolean singleLimit)
        throws HttpException, JsonException
    {
        long uid = context.session().params().get(
            "uid",
            PositiveLongValidator.INSTANCE);
        String origin = context.session().params().getString("origin", "");
        ImmutableServiceConfig serviceConfig =
            server.servicesConfigs().get(origin);
        Set<String> allowedServices = null;
        LimitsInfo limitsInfo;
        boolean userTicketRequired;
        boolean sendPush;
        if (serviceConfig == null) {
            if (!origin.isEmpty()) {
                throw new BadRequestException("Unknown origin: " + origin);
            }
            userTicketRequired = true;
            sendPush = true;
            limitsInfo = parseLimits(payload);
            JsonList defaultAllowedServices =
                payload.get("defaultAllowedServices").asListOrNull();
            Boolean allowAllServices =
                payload.get("defaultAllowAllServices").asBooleanOrNull();
            if (defaultAllowedServices == null) {
                if (allowAllServices != Boolean.TRUE) {
                    throw new BadRequestException(
                        "Neither defaultAllowedServices nor "
                        + "defaultAllowAllServices is set");
                }
            } else {
                if (allowAllServices == Boolean.TRUE) {
                    throw new BadRequestException(
                        "allowAllServices is set while "
                        + "defaultAllowedServices also set");
                }
                allowedServices =
                    convertAllowedServices(defaultAllowedServices);
            }
        } else {
            userTicketRequired = serviceConfig.userTicketRequired();
            sendPush = serviceConfig.sendPushOnFamilyStart();
            limitsInfo = new LimitsInfo(
                serviceConfig.dayLimit(),
                Long.MAX_VALUE,
                Long.MAX_VALUE,
                Long.MAX_VALUE);
            Set<String> enabledServices = serviceConfig.enabledServices();
            if (!enabledServices.isEmpty()) {
                allowedServices = new TreeSet<>(enabledServices);
            }
        }
        Long ticketUid =
            (Long) context.session().context().getAttribute(
                HttpServer.TVM_USER_UID);
        if (ticketUid == null) {
            if (userTicketRequired) {
                throw new UnauthorizedException("User ticket not provided");
            }
        } else if (!Long.valueOf(uid).equals(ticketUid)) {
            throw new ForbiddenException(
                "User ticked uid mismatch: " + uid + " != " + ticketUid);
        }

        JsonMap cardInfoMap = payload.get("cardInfo").asMap();
        CardInfo cardInfo =
            CardInfo.fromJson(
                cardInfoMap,
                // At family start card already good and bound
                // if not specified otherwise
                cardInfoMap.getBoolean("bound", true));

        Boolean unlim = payload.get("defaultUnlim").asBooleanOrNull();
        if (unlim == null) {
            unlim = limitsInfo.unlim();
        }
        FamilyInfo familyInfo =
            new FamilyInfo(
                familyId,
                uid,
                cardInfo,
                origin,
                unlim.booleanValue(),
                limitsInfo,
                allowedServices);
        context.tskvLogger().log(
            context.tskvRecord(
                TskvFields.Stage.INTERMEDIATE,
                "Family start requested for " + familyInfo));

        BlackboxClient blackboxClient =
            context.server().blackboxClient().adjust(
                context.session().context());
        blackboxClient.familyInfo(
            new BlackboxFamilyInfoRequest(familyId),
            context.session().listener().createContextGeneratorFor(
                blackboxClient),
            new BlackboxCallback(
                context,
                familyInfo,
                sendPush && familyInfo.cardInfo().bound(),
                singleLimit));
    }

    private static class BlackboxCallback
        extends AbstractFamilypayCallback<BlackboxFamilyInfo>
    {
        private final FamilyInfo familyInfo;
        private final boolean sendPush;
        private final boolean singleLimit;

        BlackboxCallback(
            final RequestContext context,
            final FamilyInfo familyInfo,
            final boolean sendPush,
            final boolean singleLimit)
        {
            super(context);
            this.familyInfo = familyInfo;
            this.sendPush = sendPush;
            this.singleLimit = singleLimit;
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof BlackboxNotFoundException) {
                failed(ErrorType.FAMILY_NOT_FOUND, null, e);
            } else {
                failed(ErrorType.BLACKBOX_FAILED, null, e);
            }
        }

        @Override
        public void completed(final BlackboxFamilyInfo blackboxFamilyInfo) {
            context.session().logger().info(
                "Blackbox family info: " + blackboxFamilyInfo);
            context.tskvLogger().log(
                context.tskvRecord(
                    TskvFields.Stage.INTERMEDIATE,
                    "Blackbox family info: " + blackboxFamilyInfo));
            if (familyInfo.adminUid() == blackboxFamilyInfo.adminUid()) {
                List<BlackboxFamilyMember> members =
                    blackboxFamilyInfo.members();
                int size = members.size();
                long[] uids = new long[size - 1];
                for (int i = 1; i < size; ++i) {
                    uids[i - 1] = members.get(i).uid();
                }
                ProxySession session = context.session();
                BlackboxClient blackboxClient =
                    context.server().blackboxClient().adjust(
                        session.context());
                blackboxClient.userinfo(
                    new BlackboxUserinfoRequest(members.get(0).uid(), uids)
                        .sid("smtp")
                        .phoneAttributes(
                            BlackboxPhoneAttributeType.IS_SECURED),
                    session.listener()
                        .createContextGeneratorFor(blackboxClient),
                    new UserinfosCallback(
                        context,
                        familyInfo,
                        sendPush,
                        singleLimit));
            } else {
                failed(
                    ErrorType.ADMIN_MISMATCH,
                    "actual family admin uid is "
                    + blackboxFamilyInfo.adminUid()
                    + ", tried to create family with admin uid "
                    + familyInfo.adminUid(),
                    null);
            }
        }
    }

    private static class UserinfosCallback
        extends AbstractFamilypayCallback<List<BlackboxUserinfo>>
    {
        private final FamilyInfo familyInfo;
        private final boolean sendPush;
        private final boolean singleLimit;

        UserinfosCallback(
            final RequestContext context,
            final FamilyInfo familyInfo,
            final boolean sendPush,
            final boolean singleLimit)
        {
            super(context);
            this.familyInfo = familyInfo;
            this.sendPush = sendPush;
            this.singleLimit = singleLimit;
        }

        @Override
        public void failed(final Exception e) {
            failed(ErrorType.BLACKBOX_FAILED, "Can't retrieve userinfos", e);
        }

        @Override
        public void completed(final List<BlackboxUserinfo> userinfos) {
            Tuple tuple = Tuple.tuple();
            familyInfo.toTuple(tuple);
            int size = userinfos.size();
            Long[] uids = new Long[size];
            Boolean[] hasSecurePhone = new Boolean[size];
            for (int i = 0; i < size; ++i) {
                BlackboxUserinfo userinfo = userinfos.get(i);
                uids[i] = userinfo.uid();
                hasSecurePhone[i] = userinfo.hasSecurePhone();
            }
            tuple.addArrayOfLong(uids);
            tuple.addArrayOfBoolean(hasSecurePhone);

            context.server().pgClient().executeOnMaster(
                ADD_FAMILY,
                tuple,
                context.session().listener(),
                new InsertCallback(
                    context,
                    familyInfo.familyId(),
                    sendPush,
                    singleLimit));
        }
    }

    private static class InsertCallback
        extends AbstractFamilypayCallback<RowSet<Row>>
    {
        private final String familyId;
        private final boolean sendPush;
        private final boolean singleLimit;

        InsertCallback(
            final RequestContext context,
            final String familyId,
            final boolean sendPush,
            final boolean singleLimit)
        {
            super(context);
            this.familyId = familyId;
            this.sendPush = sendPush;
            this.singleLimit = singleLimit;
        }

        @Override
        public void failed(final Exception e) {
            failed(ErrorType.INTERNAL_ERROR, "Failed to insert users", e);
        }

        @Override
        @SuppressWarnings("ReferenceEquality")
        public void completed(final RowSet<Row> rowSet) {
            int rowCount = rowSet.rowCount();
            context.session().logger().info(
                "Family inserted, row count = " + rowCount);
            if (rowCount == 0) {
                failed(
                    ErrorType.FAMILY_MISMATCH,
                    "Mismatching family settings already present in database",
                    null);
            } else {
                Row row = rowSet.iterator().next();
                Boolean isNew = row.getBoolean("is_new");
                FamilypayCallback<Family> callback =
                    new FamilyHandler.FamilyCallback(context, singleLimit);
                if (sendPush && isNew == Boolean.TRUE) {
                    callback = new NotifyCallback(callback);
                }
                FamilyHandler.findFamily(
                    familyId,
                    callback);
            }
        }
    }

    private static class NotifyCallback
        extends AbstractFilterFamilypayCallback<Family, Family>
    {
        NotifyCallback(final FamilypayCallback<? super Family> callback) {
            super(callback);
        }

        @Override
        public void completed(final Family family) {
            context().server().pusher().notifyOnFamilyChange(
                context(),
                null,
                family);
            callback.completed(family);
        }
    }
}

