package ru.yandex.passport.familypay.backend;

import java.util.Set;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpException;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;

public class FamilyUpdateHandler extends FamilyHandlerBase {
    private static final SqlQuery UPDATE_FAMILY =
        new SqlQuery("update-family.sql", FamilyUpdateHandler.class);

    public FamilyUpdateHandler(final FamilypayBackend server) {
        super(server, false);
    }

    @Override
    protected void handle(
        final String familyId,
        final JsonObject payload,
        final RequestContext context)
        throws HttpException, JsonException
    {
        JsonMap limitsMap = payload.get("defaultLimits").asMapOrNull();
        LimitsInfo limitsInfo;
        if (limitsMap == null) {
            limitsInfo = null;
        } else {
            limitsInfo = LimitsInfo.fromJson(limitsMap);
        }
        handle(
            familyId,
            payload,
            context,
            limitsInfo);
    }

    @SuppressWarnings("ReferenceEquality")
    protected void handle(
        final String familyId,
        final JsonObject payload,
        final RequestContext context,
        final LimitsInfo limitsInfo)
        throws HttpException, JsonException
    {
        JsonMap cardMap = payload.get("cardInfo").asMapOrNull();
        CardInfo cardInfo;
        if (cardMap == null) {
            cardInfo = null;
        } else {
            cardInfo = CardInfo.fromJson(cardMap);
        }

        Set<String> allowedServices;
        JsonList defaultAllowedServices =
            payload.get("defaultAllowedServices").asListOrNull();
        if (defaultAllowedServices == null) {
            allowedServices = null;
        } else {
            allowedServices = convertAllowedServices(defaultAllowedServices);
        }
        Boolean allowAllServices =
            payload.get("defaultAllowAllServices").asBooleanOrNull();
        if (allowAllServices == Boolean.TRUE && allowedServices != null) {
            throw new BadRequestException(
                "defaultAllowAllServices is set while "
                + "defaultAllowedServices also set");
        }

        context.tskvLogger().log(
            context.tskvRecord(
                TskvFields.Stage.INTERMEDIATE,
                "Family update requested. New limits: " + limitsInfo
                + ", new card info: " + cardInfo
                + ", allowed services: " + allowedServices
                + ", allow all services: " + allowAllServices));

        Long adminUid =
            (Long) context.session().context().getAttribute(
                HttpServer.TVM_USER_UID);
        Tuple tuple = Tuple.tuple();
        tuple.addString(familyId);
        tuple.addLong(adminUid);
        if (cardInfo == null) {
            tuple.addValue(null);
            tuple.addValue(null);
            tuple.addValue(null);
            tuple.addValue(null);
            tuple.addValue(null);
        } else {
            cardInfo.toTuple(tuple);
        }
        Boolean unlim = payload.get("defaultUnlim").asBooleanOrNull();
        if (unlim == null && limitsInfo != null) {
            unlim = limitsInfo.unlim();
        }
        tuple.addBoolean(unlim);
        if (limitsInfo == null) {
            tuple.addValue(null);
            tuple.addValue(null);
            tuple.addValue(null);
            tuple.addValue(null);
        } else {
            limitsInfo.toTuple(tuple);
        }
        if (allowedServices == null) {
            tuple.addValue(null);
        } else {
            tuple.addArrayOfString(
                allowedServices.toArray(new String[allowedServices.size()]));
        }
        tuple.addBoolean(allowAllServices);
        FamilyCallback callback = new FamilyCallback(context, tuple);
        FamilyHandler.findFamily(familyId, callback);
    }

    private static class FamilyCallback
        extends AbstractFamilypayCallback<Family>
    {
        private final Tuple tuple;

        FamilyCallback(final RequestContext context, final Tuple tuple) {
            super(context);
            this.tuple = tuple;
        }

        @Override
        public void completed(final Family family) {
            context.server().pgClient().executeOnMaster(
                UPDATE_FAMILY,
                tuple,
                context.session().listener(),
                new UpdateCallback(context, family));
        }
    }

    private static class UpdateCallback
        extends AbstractFamilypayCallback<RowSet<Row>>
    {
        private final Family oldFamily;

        UpdateCallback(final RequestContext context, final Family oldFamily) {
            super(context);
            this.oldFamily = oldFamily;
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            if (rowSet.rowCount() <= 0) {
                failed(ErrorType.FAMILY_NOT_FOUND, null, null);
            } else {
                emptyResponse();

                FamilyHandler.findFamily(
                    oldFamily.familyInfo().familyId(),
                    new FamilyChangeNotifyCallback(context, oldFamily));
            }
        }
    }
}

