package ru.yandex.passport.familypay.backend;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpException;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.string.PositiveLongValidator;

public class FamilyUpdateUsersHandler extends FamilyHandlerBase {
    private static final SqlQuery UPDATE_USER =
        new SqlQuery("update-users.sql", FamilyUpdateUsersHandler.class);

    public FamilyUpdateUsersHandler(final FamilypayBackend server) {
        super(server, false);
    }

    protected LimitsInfo parseLimits(final JsonMap userInfo)
        throws JsonException
    {
        JsonMap limitsMap = userInfo.get("limits").asMapOrNull();
        if (limitsMap == null) {
            return null;
        } else {
            return LimitsInfo.fromJson(limitsMap);
        }
    }

    @Override
    @SuppressWarnings("ReferenceEquality")
    protected void handle(
        final String familyId,
        final JsonObject payload,
        final RequestContext context)
        throws HttpException, JsonException
    {
        Long adminUid =
            (Long) context.session().context().getAttribute(
                HttpServer.TVM_USER_UID);
        JsonList list = payload.asList();
        int size = list.size();
        List<Tuple> tuples = new ArrayList<>(size);
        Set<Long> uids = new TreeSet<>();
        for (int i = 0; i < size; ++i) {
            JsonMap map = list.get(i).asMap();

            long uid = map.get("uid", PositiveLongValidator.INSTANCE);
            uids.add(uid);
            Tuple tuple = Tuple.tuple();
            tuple.addString(familyId);
            tuple.addLong(adminUid);
            tuple.addLong(uid);

            LimitsInfo limitsInfo = parseLimits(map);
            Boolean unlim = map.get("unlim").asBooleanOrNull();
            if (unlim == null && limitsInfo != null) {
                unlim = limitsInfo.unlim();
            }
            tuple.addBoolean(unlim);
            tuple.addString(map.get("limitCurrency").asStringOrNull());

            if (limitsInfo == null) {
                tuple.addValue(null);
                tuple.addValue(null);
                tuple.addValue(null);
                tuple.addValue(null);
            } else {
                limitsInfo.toTuple(tuple);
            }

            Boolean allowAllServices =
                map.get("allowAllServices").asBooleanOrNull();
            JsonList allowedServicesList =
                map.get("allowedServices").asListOrNull();
            if (allowedServicesList == null) {
                tuple.addValue(null);
            } else {
                if (allowAllServices == Boolean.TRUE) {
                    throw new BadRequestException(
                        "allowAllServices is set while "
                        + "allowedServices also set for uid " + uid);
                }
                Set<String> allowedServices =
                    convertAllowedServices(allowedServicesList);
                tuple.addArrayOfString(
                    allowedServices.toArray(
                        new String[allowedServices.size()]));
            }
            Boolean enabled = map.getBoolean("enabled", null);
            tuple.addBoolean(enabled);
            tuple.addBoolean(allowAllServices);
            tuples.add(tuple);

            context.tskvLogger().log(
                context.tskvRecord(
                    TskvFields.Stage.INTERMEDIATE,
                    "Family user " + uid
                    + " update requested. New limits: " + limitsInfo
                    + ", allowed services: "
                    + JsonType.NORMAL.toString(allowedServicesList)
                    + ", familypay enabled: " + enabled
                    + ", allow all services: " + allowAllServices));
        }
        FamilyCallback callback = new FamilyCallback(context, uids, tuples);
        FamilyHandler.findFamily(familyId, callback);
    }

    private static class FamilyCallback
        extends AbstractFamilypayCallback<Family>
    {
        private final Set<Long> uids;
        private final List<Tuple> tuples;

        FamilyCallback(
            final RequestContext context,
            final Set<Long> uids,
            final List<Tuple> tuples)
        {
            super(context);
            this.uids = uids;
            this.tuples = tuples;
        }

        @Override
        public void completed(final Family family) {
            context.server().pgClient().executeBatchOnMaster(
                UPDATE_USER,
                tuples,
                context.session().listener(),
                new UpdateCallback(context, uids, family));
        }
    }

    private static class UpdateCallback
        extends AbstractFamilypayCallback<RowSet<Row>>
    {
        private final Set<Long> uids;
        private final Family oldFamily;

        UpdateCallback(
            final RequestContext context,
            final Set<Long> uids,
            final Family oldFamily)
        {
            super(context);
            this.uids = uids;
            this.oldFamily = oldFamily;
        }

        @Override
        public void completed(RowSet<Row> rowSet) {
            Set<Long> updatedUids = new TreeSet<>();
            try {
                while (rowSet != null) {
                    for (Row row: rowSet) {
                        updatedUids.add(row.getLong("uid"));
                    }
                    rowSet = rowSet.next();
                }
            } catch (RuntimeException e) {
                failed(
                    ErrorType.INTERNAL_ERROR,
                    "Unable to get updated uids list",
                    e);
                return;
            }
            JsonList updated = new JsonList(
                BasicContainerFactory.INSTANCE,
                updatedUids.size());
            for (Long updatedUid: updatedUids) {
                updated.add(new JsonLong(updatedUid.longValue()));
            }
            JsonList skipped = new JsonList(
                BasicContainerFactory.INSTANCE,
                uids.size() - updated.size());
            for (Long uid: uids) {
                if (!updatedUids.contains(uid)) {
                    skipped.add(new JsonLong(uid.longValue()));
                }
            }
            JsonMap map = new JsonMap(BasicContainerFactory.INSTANCE, 3);
            map.put("updated", updated);
            map.put("skipped", skipped);
            context.tskvLogger().log(
                context.tskvRecord(
                    TskvFields.Stage.INTERMEDIATE,
                    "Family update result: " + JsonType.NORMAL.toString(map)));
            sendResponse(map);

            FamilyHandler.findFamily(
                oldFamily.familyInfo().familyId(),
                new FamilyChangeNotifyCallback(context, oldFamily));
        }
    }
}

