package ru.yandex.passport.familypay.backend;

import java.io.IOException;
import java.util.Map;

import io.vertx.core.Vertx;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.client.pg.PgClient;
import ru.yandex.collection.Pattern;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.ByteArrayProcessableWithContentType;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestPatternParser;
import ru.yandex.http.util.server.ExternalDataSubscriber;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.passport.familypay.backend.config.ImmutableFamilyPaybackConfig;
import ru.yandex.passport.familypay.backend.config.ImmutableServiceConfig;
import ru.yandex.passport.familypay.services.FamilypayServices;
import ru.yandex.tskv.BasicPrefixedTskvLogger;

public class FamilypayBackend
    extends HttpProxy<ImmutableFamilyPaybackConfig>
    implements ExternalDataSubscriber
{
    private final Map<String, ImmutableServiceConfig> servicesConfigs;
    private final PgClient pgClient;
    private final BlackboxClient blackboxClient;
    private final AsyncClient antifraudClient;
    private final HttpHost antifraudHost;
    private final BasicPrefixedTskvLogger tskvLogger;
    private final Pusher pusher;
    private volatile FamilypayServices services;

    public FamilypayBackend(final ImmutableFamilyPaybackConfig config)
        throws ConfigException,
            HttpException,
            IOException,
            InterruptedException,
            JsonException
    {
        super(config);
        servicesConfigs = config.servicesConfigs();

        Vertx vertx = Vertx.vertx();
        closeChain.add(vertx::close);
        pgClient = new PgClient(vertx, config.pgClientConfig(), logger);
        closeChain.add(pgClient);

        blackboxClient =
            registerClient(
                "Blackbox",
                new BlackboxClient(reactor, config.blackboxConfig()),
                config.blackboxConfig());

        ImmutableHttpHostConfig antifraudConfig = config.antifraudConfig();
        antifraudClient = client("Antifraud", antifraudConfig);
        antifraudHost = antifraudConfig.host();

        tskvLogger =
            new BasicPrefixedTskvLogger(
                config.tskvLoggerConfig().build(
                    config.loggers().handlersManager()));
        registerLoggerForLogrotate(tskvLogger.logger());

        pusher = new Pusher(this);

        subscribeForExternalDataUpdates("bunker", this);

        register(
            RequestPatternParser.INSTANCE.apply(
                "/family/*{path_regex:/family/f?[0-9]+}"),
            new FamilyHandler(this));
        register(
            RequestPatternParser.INSTANCE.apply(
                "/family/*{path_regex:/family/f?[0-9]+/single_limit}"),
            new SingleLimitFamilyHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/family/*{path_regex:/family/f?[0-9]+/start}"),
            new FamilyStartHandler(this));
        register(
            RequestPatternParser.INSTANCE.apply(
                "/family/*{path_regex:/family/f?[0-9]+/start/single_limit}"),
            new SingleLimitFamilyStartHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/family/*{path_regex:/family/f?[0-9]+/stop}"),
            new FamilyStopHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/family/*{path_regex:/family/f?[0-9]+/update}"),
            new FamilyUpdateHandler(this));
        register(
            RequestPatternParser.INSTANCE.apply(
                "/family/*{path_regex:/family/f?[0-9]+/update/single_limit}"),
            new SingleLimitFamilyUpdateHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/family/*{path_regex:/family/f?[0-9]+/update/users}"),
            new FamilyUpdateUsersHandler(this));
        register(
            RequestPatternParser.INSTANCE.apply(
                "/family/*{path_regex:/family/f?[0-9]+/update/users"
                + "/single_limit}"),
            new SingleLimitFamilyUpdateUsersHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/family/*{path_regex:/family/f?[0-9]+/bind_card}"),
            new FamilyBindCardHandler(this));

        register(
            new Pattern<>("/payment", false),
            new PaymentHandler(this));
        register(
            new Pattern<>("/payment/create", false),
            new CreatePaymentHandler(this));
        register(
            new Pattern<>("/payment/complete", false),
            new CompletePaymentHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/user/*{path_regex:/user/[0-9]+/info}"),
            new UserInfoHandler(this));
        register(
            RequestPatternParser.INSTANCE.apply(
                "/user/*{path_regex:/user/[0-9]+/info/single_limit}"),
            new SingleLimitUserInfoHandler(this));
        register(
            RequestPatternParser.INSTANCE.apply(
                "/user/*{path_regex:/user/[0-9]+/settings}"),
            new UserSettingsHandler(this));
        register(
            RequestPatternParser.INSTANCE.apply(
                "/user/*{path_regex:/user/[0-9]+/reset_expenses}"),
            new ResetUserExpensesHandler(this));

        register(
            new Pattern<>("/update-family", false),
            new FamilyChangeHandler(this));

        register(
            new Pattern<>("/update-card-info", false),
            new CardInfoChangeHandler(this));

        register(
            new Pattern<>("/update-phone-info", false),
            new PhoneInfoChangeHandler(this));
    }

    public Map<String, ImmutableServiceConfig> servicesConfigs() {
        return servicesConfigs;
    }

    public PgClient pgClient() {
        return pgClient;
    }

    public BlackboxClient blackboxClient() {
        return blackboxClient;
    }

    public AsyncClient antifraudClient() {
        return antifraudClient;
    }

    public HttpHost antifraudHost() {
        return antifraudHost;
    }

    public BasicPrefixedTskvLogger tskvLogger() {
        return tskvLogger;
    }

    public Pusher pusher() {
        return pusher;
    }

    public FamilypayServices services() {
        return services;
    }

    @Override
    public void updateExternalData(
        final ByteArrayProcessableWithContentType data)
        throws HttpException, IOException
    {
        services = new FamilypayServices(data);
    }
}

