package ru.yandex.passport.familypay.backend;

import java.util.Objects;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.Tuple;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.string.NonNegativeLongValidator;

public class LimitsInfo extends AbstractStringBuilderable {
    private static final long UNLIM_VALUE = 1_000_000_000L;

    private final Long day;
    private final Long week;
    private final Long month;
    private final Long total;

    public LimitsInfo(
        final Long day,
        final Long week,
        final Long month,
        final Long total)
    {
        this.day = day;
        this.week = week;
        this.month = month;
        this.total = total;
    }

    public static LimitsInfo fromRow(
        final Row row,
        final String prefix,
        final String suffix)
    {
        return new LimitsInfo(
            Objects.requireNonNullElse(
                row.getLong(prefix + "day" + suffix),
                0L),
            Objects.requireNonNullElse(
                row.getLong(prefix + "week" + suffix),
                0L),
            Objects.requireNonNullElse(
                row.getLong(prefix + "month" + suffix),
                0L),
            Objects.requireNonNullElse(
                row.getLong(prefix + "total" + suffix),
                0L));
    }

    public boolean unlim() {
        return day >= UNLIM_VALUE
            && week >= UNLIM_VALUE
            && month >= UNLIM_VALUE
            && total >= UNLIM_VALUE;
    }

    public void toTuple(final Tuple tuple) {
        tuple.addLong(day);
        tuple.addLong(week);
        tuple.addLong(month);
        tuple.addLong(total);
    }

    public static LimitsInfo fromJson(final JsonMap map) throws JsonException {
        return new LimitsInfo(
            map.get("day", null, NonNegativeLongValidator.INSTANCE),
            map.get("week", null, NonNegativeLongValidator.INSTANCE),
            map.get("month", null, NonNegativeLongValidator.INSTANCE),
            map.get("total", null, NonNegativeLongValidator.INSTANCE));
    }

    public static LimitsInfo fromJsonSingle(final JsonMap map)
        throws JsonException
    {
        LimitMode limitMode = map.getEnum(LimitMode.class, "limitMode");
        return limitMode.createLimitsInfo(
            map.get("value", NonNegativeLongValidator.INSTANCE));
    }

    public JsonMap toJson(
        final ContainerFactory containerFactory,
        final boolean singleLimit,
        final String suffix)
    {
        if (singleLimit) {
            JsonMap result = new JsonMap(containerFactory, 3);
            LimitMode mode = LimitMode.detectLimitMode(this);
            result.put("limitMode", new JsonString(mode.name()));
            result.put("value", new JsonLong(mode.limitValue(this)));
            return result;
        } else {
            JsonMap result = new JsonMap(containerFactory, 6);
            result.put("day" + suffix, new JsonLong(day));
            result.put("week" + suffix, new JsonLong(week));
            result.put("month" + suffix, new JsonLong(month));
            result.put("total" + suffix, new JsonLong(total));
            return result;
        }
    }

    public Long day() {
        return day;
    }

    public Long week() {
        return week;
    }

    public Long month() {
        return month;
    }

    public Long total() {
        return total;
    }

    @Override
    public int hashCode() {
        return Objects.hash(day, week, month, total);
    }

    @Override
    public boolean equals(final Object o) {
        if (o instanceof LimitsInfo) {
            LimitsInfo other = (LimitsInfo) o;
            return Objects.equals(day, other.day)
                && Objects.equals(week, other.week)
                && Objects.equals(month, other.month)
                && Objects.equals(total, other.total);
        }
        return false;
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append("LimitsInfo(");
        sb.append(day);
        sb.append(',');
        sb.append(week);
        sb.append(',');
        sb.append(month);
        sb.append(',');
        sb.append(total);
        sb.append(')');
    }
}

