package ru.yandex.passport.familypay.backend;

import java.time.OffsetDateTime;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxPhoneAttributeType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.blackbox.SingleUserinfoBlackboxCallback;
import ru.yandex.client.pg.SqlQuery;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.FakeAsyncConsumer;

public class PhoneInfoChangeHandler
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final SqlQuery SET_HAS_SECURE_PHONE =
        new SqlQuery(
            "set-has-secure-phone.sql",
            PhoneInfoChangeHandler.class);

    protected final FamilypayBackend server;

    public PhoneInfoChangeHandler(final FamilypayBackend server) {
        this.server = server;
    }

    @Override
    public FakeAsyncConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new FakeAsyncConsumer<>(request);
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(server, exchange, context);
        long uid = session.params().getLong("uid");
        RequestContext requestContext =
            new RequestContext(
                server,
                session,
                server.tskvLogger().record()
                    .append(TskvFields.HANDLER, getClass().getSimpleName())
                    .append(TskvFields.UNDERGOER_UID, uid),
                "",
                false);
        FamilyCallback callback = new FamilyCallback(requestContext, uid);
        FamilyHandler.findFamilyByMemberUid(uid, callback);
    }

    private static class FamilyCallback
        extends AbstractFamilypayCallback<Family>
    {
        private final long uid;

        FamilyCallback(
            final RequestContext context,
            final long uid)
        {
            super(context);
            this.uid = uid;
        }

        @Override
        public void failed(
            final ErrorType errorType,
            final String message,
            final Throwable cause)
        {
            if (errorType == ErrorType.FAMILY_NOT_FOUND) {
                completed(null);
            } else {
                super.failed(errorType, message, cause);
            }
        }

        @Override
        public void completed(final Family family) {
            if (family == null) {
                context.session().logger().info("Family not found");
                emptyResponse();
            } else {
                ProxySession session = context.session();
                BlackboxClient blackboxClient =
                    context.server().blackboxClient().adjust(
                        session.context());
                blackboxClient.userinfo(
                    new BlackboxUserinfoRequest(uid)
                        .sid("smtp")
                        .phoneAttributes(
                            BlackboxPhoneAttributeType.IS_SECURED),
                    session.listener()
                        .createContextGeneratorFor(blackboxClient),
                    new SingleUserinfoBlackboxCallback(
                        new BlackboxCallback(
                            context,
                            uid,
                            family,
                            OffsetDateTime.now())));
            }
        }
    }

    private static class BlackboxCallback
        extends AbstractFamilypayCallback<BlackboxUserinfo>
    {
        private final long uid;
        private final Family family;
        private final OffsetDateTime timestamp;

        BlackboxCallback(
            final RequestContext context,
            final long uid,
            final Family family,
            final OffsetDateTime timestamp)
        {
            super(context);
            this.uid = uid;
            this.family = family;
            this.timestamp = timestamp;
        }

        @Override
        public void completed(final BlackboxUserinfo userinfo) {
            Tuple tuple = Tuple.tuple();
            tuple.addBoolean(userinfo.hasSecurePhone());
            tuple.addLong(uid);
            tuple.addOffsetDateTime(timestamp);
            context.server().pgClient().executeOnMaster(
                SET_HAS_SECURE_PHONE,
                tuple,
                context.session().listener(),
                new Callback(context, family));
        }
    }

    private static class Callback
        extends AbstractFamilypayCallback<RowSet<Row>>
    {
        private final Family oldFamily;

        Callback(final RequestContext context, final Family oldFamily) {
            super(context);
            this.oldFamily = oldFamily;
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            if (rowSet.rowCount() <= 0) {
                failed(ErrorType.CONCURRENT_PHONE_STATUS_CHANGE, null, null);
            } else {
                emptyResponse();

                FamilyHandler.findFamily(
                    oldFamily.familyInfo().familyId(),
                    new FamilyChangeNotifyCallback(context, oldFamily));
            }
        }
    }
}

