package ru.yandex.passport.familypay.backend;

import java.io.IOException;
import java.text.NumberFormat;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.LoggingFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.uri.PctEncoder;
import ru.yandex.parser.uri.PctEncodingRule;
import ru.yandex.passport.familypay.backend.config.ImmutableServiceConfig;
import ru.yandex.util.timesource.TimeSource;

public class Pusher {
    private static final AmountFormatter ROOT_AMOUNT_FORMATTER =
        new AmountFormatter(Locale.ROOT);
    private static final Map<String, AmountFormatter> AMOUNT_FORMATTERS =
        Map.of(
            "AMD", new AmountFormatter(Locale.forLanguageTag("hy-AM")),
            "BYN", new AmountFormatter(Locale.forLanguageTag("be-BY")),
            "EUR", new AmountFormatter(Locale.forLanguageTag("de-DE")),
            "GBP", new AmountFormatter(Locale.forLanguageTag("en-GB")),
            "ILS", new AmountFormatter(Locale.forLanguageTag("he-IL")),
            "KZT", new AmountFormatter(Locale.forLanguageTag("kk-Cyrl-KZ")),
            "RUB", new AmountFormatter(Locale.forLanguageTag("ru-RU")),
            "USD", new AmountFormatter(Locale.forLanguageTag("en-US")));

    private final FamilypayBackend server;

    public Pusher(final FamilypayBackend server) {
        this.server = server;
    }

    private static String nonUniqueId() {
        // Some request doesn't have unique ids, like family start/stop
        // So, we generate some non-unique id, so we can deduplicate retries
        // but still send push for family restart
        return Long.toString(TimeSource.INSTANCE.currentTimeMillis() / 20000L);
    }

    private static String formatAmount(
        final long amount,
        final String currency)
    {
        AmountFormatter formatter =
            AMOUNT_FORMATTERS.getOrDefault(currency, ROOT_AMOUNT_FORMATTER);
        return formatter.format(amount);
    }

    private void sendPush(
        final RequestContext context,
        final long recipientUid,
        final String pushRequest,
        final Map<String, Object> requestParams,
        final String eventName,
        final String templateId,
        final Map<String, Object> indexes,
        final Map<String, Object> freeParams,
        final String pushId)
    {
        String uidString = Long.toString(recipientUid);
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
            writer.startObject();
            writer.key("recipient_uid");
            writer.value(uidString);
            writer.key("push_request");
            writer.value(pushRequest);
            writer.key("push_id");
            StringBuilder sb = new StringBuilder(pushId);
            sb.append('_');
            sb.append(recipientUid);
            sb.append('_');
            sb.append(templateId);
            writer.value(new String(sb));
            writer.key("request_params");
            writer.startObject();
            for (Map.Entry<String, Object> entry: requestParams.entrySet()) {
                writer.key(entry.getKey());
                writer.value(entry.getValue().toString());
            }
            writer.endObject();
            writer.key("event_name");
            writer.value(eventName);
            for (Map.Entry<String, Object> entry: freeParams.entrySet()) {
                writer.key(entry.getKey());
                writer.value(entry.getValue().toString());
            }
            writer.key("template_id");
            writer.value(templateId);
            writer.key("template_parameters");
            writer.startObject();
            for (Map.Entry<String, Object> entry: indexes.entrySet()) {
                writer.key(entry.getKey());
                writer.value(entry.getValue().toString());
            }
            writer.endObject();
            writer.endObject();
        } catch (IOException e) {
            // can't be
        }
        ProxySession session = context.session();
        String body = sbw.toString();
        session.logger().info("Sending push request " + body);
        String uri =
            "/execute?app=family-push&uid=" + uidString + "&session_id="
            + session.context().getAttribute(HttpServer.SESSION_ID);
        AsyncClient client =
            server.antifraudClient().adjust(session.context());
        client.execute(
            server.antifraudHost(),
            new BasicAsyncRequestProducerGenerator(
                uri,
                body,
                ContentType.APPLICATION_JSON),
            EmptyAsyncConsumerFactory.ANY_GOOD,
            session.listener().createContextGeneratorFor(client),
            new PushResultFutureCallback<>(
                new LoggingFutureCallback<>(
                    EmptyFutureCallback.INSTANCE,
                    session.logger()),
                context,
                uri,
                body));
    }

    private void notifyAdminOnFamilyStart(
        final RequestContext context,
        final long adminUid,
        final String pushId)
    {
        sendPush(
            context,
            adminUid,
            "/am/push/family",
            Collections.emptyMap(),
            "family_change",
            "add-family-card-for-admin",
            Collections.emptyMap(),
            Collections.emptyMap(),
            pushId);
    }

    private void notifyAdminOnFamilyStop(
        final RequestContext context,
        final long adminUid,
        final String pushId)
    {
        sendPush(
            context,
            adminUid,
            "/am/push/family",
            Collections.emptyMap(),
            "family_change",
            "off-family-card-for-admin",
            Collections.emptyMap(),
            Collections.emptyMap(),
            pushId);
    }

    public void notifyOnFamilyChange(
        final RequestContext context,
        final Family oldFamily,
        final Family newFamily)
    {
        String pushId;
        if (oldFamily == null) {
            pushId = "family-start-" + nonUniqueId();
        } else if (newFamily == null) {
            pushId = "family-stop-" + nonUniqueId();
        } else {
            pushId = "family-change-" + nonUniqueId();
        }
        notifyOnFamilyChange(context, oldFamily, newFamily, pushId);
    }

    public void notifyOnFamilyChange(
        final RequestContext context,
        final Family oldFamily,
        final Family newFamily,
        final String pushId)
    {
        String origin;
        if (oldFamily == null) {
            origin = newFamily.familyInfo().origin();
        } else {
            origin = oldFamily.familyInfo().origin();
        }
        ImmutableServiceConfig serviceConfig =
            context.server().servicesConfigs().get(origin);
        boolean sendPush = true;
        if (serviceConfig != null) {
            sendPush = serviceConfig.sendPushOnFamilyChange();
        }
        ProxySession session = context.session();
        session.logger()
            .info("For origin " + origin + ", sendPush = " + sendPush);
        if (!sendPush) {
            return;
        }
        if (oldFamily == null) {
            // Family start
            long adminUid = newFamily.familyInfo().adminUid();
            notifyAdminOnFamilyStart(
                context.addPrefix(
                    TskvFields.RECIPIENT_UID,
                    Long.toString(adminUid)),
                adminUid,
                pushId);
            for (FamilyMember familyMember
                : newFamily.familyMembers().values())
            {
                long uid = familyMember.uid();
                if (uid != adminUid) {
                    notifyMemberOnFamilyChange(
                        context.addPrefix(
                            TskvFields.RECIPIENT_UID,
                            Long.toString(familyMember.uid())),
                        uid,
                        null,
                        newFamily,
                        null,
                        familyMember,
                        false,
                        pushId);
                }
            }
        } else if (newFamily == null) {
            // Family stop or family drop
            long adminUid = oldFamily.familyInfo().adminUid();
            notifyAdminOnFamilyStop(
                context.addPrefix(
                    TskvFields.RECIPIENT_UID,
                    Long.toString(adminUid)),
                adminUid,
                pushId);
            for (FamilyMember familyMember
                : oldFamily.familyMembers().values())
            {
                long uid = familyMember.uid();
                if (familyMember.uid() != adminUid) {
                    notifyMemberOnFamilyChange(
                        context.addPrefix(
                            TskvFields.RECIPIENT_UID,
                            Long.toString(familyMember.uid())),
                        uid,
                        oldFamily,
                        null,
                        familyMember,
                        null,
                        false,
                        pushId);
                }
            }
        } else {
            // Family change
            long adminUid = oldFamily.familyInfo().adminUid();
            for (Map.Entry<Long, FamilyMember> entry
                : oldFamily.familyMembers().entrySet())
            {
                long uid = entry.getKey();
                if (uid != adminUid) {
                    notifyMemberOnFamilyChange(
                        context.addPrefix(
                            TskvFields.RECIPIENT_UID,
                            Long.toString(uid)),
                        uid,
                        oldFamily,
                        newFamily,
                        entry.getValue(),
                        newFamily.familyMembers().get(uid),
                        true,
                        pushId);
                }
            }
            for (Map.Entry<Long, FamilyMember> entry
                : newFamily.familyMembers().entrySet())
            {
                long uid = entry.getKey();
                if (uid != adminUid) {
                    if (oldFamily.familyMembers().get(uid) == null) {
                        notifyMemberOnFamilyChange(
                            context.addPrefix(
                                TskvFields.RECIPIENT_UID,
                                Long.toString(uid)),
                            uid,
                            oldFamily,
                            newFamily,
                            null,
                            entry.getValue(),
                            true,
                            pushId);
                    }
                }
            }
        }
    }

    public void notifyAdminOnFamilyDelete(
        final RequestContext context,
        final long adminUid,
        final String pushId)
    {
        sendPush(
            context,
            adminUid,
            "/am/push/family",
            Collections.emptyMap(),
            "family_change",
            "admin-leave-family-for-admin",
            Collections.emptyMap(),
            Collections.emptyMap(),
            pushId);
    }

    public void notifyMemberOnFamilyDelete(
        final RequestContext context,
        final long uid,
        final String pushId)
    {
        sendPush(
            context,
            uid,
            "/am/push/family",
            Collections.emptyMap(),
            "family_change",
            "admin-leave-family-for-close",
            Collections.emptyMap(),
            Collections.emptyMap(),
            pushId);
    }

    public void notifyOnMemberAdd(
        final RequestContext context,
        final long adminUid,
        final long uid,
        final String pushId)
    {
        sendPush(
            context,
            adminUid,
            "/am/push/family",
            Map.of("member_uid", uid),
            "family_change",
            "add-close-to-family-for-admin",
            Map.of("action", "accept"),
            Map.of("uid", uid),
            pushId);
    }

    public void notifyOnMemberDelete(
        final RequestContext context,
        final long adminUid,
        final long uid,
        final String pushId)
    {
        sendPush(
            context,
            adminUid,
            "/am/push/family",
            Map.of("member_uid", uid),
            "family_change",
            "member-leave-family-for-admin",
            Map.of("action", "leave"),
            Map.of("uid", uid),
            pushId);
        sendPush(
            context,
            uid,
            "/am/push/family",
            Collections.emptyMap(),
            "family_change",
            "admin-remove-member-for-close",
            Collections.emptyMap(),
            Collections.emptyMap(),
            pushId);
    }

    private void notifyMemberOnFamilyChange(
        final RequestContext context,
        final long uid,
        final Family oldFamily,
        final Family newFamily,
        final FamilyMember oldMemberInfo,
        final FamilyMember newMemberInfo,
        final boolean notifyAdmin,
        final String pushId)
    {
        ProxySession session = context.session();
        boolean oldEnabled =
            Family.familypayEnabled(oldFamily)
            && FamilyMember.familypayEnabled(oldMemberInfo);
        boolean newEnabled =
            Family.familypayEnabled(newFamily)
            && FamilyMember.familypayEnabled(newMemberInfo);
        if (oldEnabled) {
            if (newEnabled) {
                LimitsInfo oldLimits = oldMemberInfo.limits();
                LimitsInfo newLimits = newMemberInfo.limits();
                if (newLimits.equals(oldLimits)
                    && newMemberInfo.expenses().equals(
                        oldMemberInfo.expenses())
                    && newMemberInfo.limitCurrency().equals(
                        oldMemberInfo.limitCurrency()))
                {
                    session.logger().info("Nothing changed for " + uid);
                } else {
                    FamilyInfo familyInfo = newFamily.familyInfo();
                    long adminUid = familyInfo.adminUid();
                    String currency = newMemberInfo.limitCurrency();
                    session.logger().info(
                        "Something changed for " + uid
                        + " from " + oldMemberInfo + " to " + newMemberInfo
                        + ", family info: " + familyInfo);
                    LimitMode limitMode =
                        LimitMode.detectLimitMode(newLimits);
                    long limitValue = limitMode.limitValue(newLimits);
                    long expenses =
                        limitMode.limitValue(newMemberInfo.expenses());
                    // XXX
                    // long balance = Math.max(0L, limitValue - expenses);
                    long balance = limitValue - expenses;
                    if (newLimits.unlim() || newMemberInfo.unlim()) {
                        if (!newLimits.equals(oldLimits)) {
                            if (notifyAdmin) {
                                sendPush(
                                    context,
                                    adminUid,
                                    "/am/push/family-limits",
                                    Map.of("member_uid", uid),
                                    "family_change",
                                    "limit-change-to-unlimit-for-admin",
                                    Collections.emptyMap(),
                                    Map.of("uid", uid),
                                    pushId);
                            }
                            sendPush(
                                context,
                                uid,
                                "/am/push/family-limits-member",
                                Collections.emptyMap(),
                                "family_change",
                                "limit-change-to-unlimit-for-close",
                                Collections.emptyMap(),
                                Collections.emptyMap(),
                                pushId);
                        }
                    } else if (limitMode == LimitMode.TOTAL) {
                        if (notifyAdmin) {
                            sendPush(
                                context,
                                adminUid,
                                "/am/push/family-limits",
                                Map.of("member_uid", uid),
                                "family_change",
                                "limit-change-without-range-for-admin",
                                Map.of(
                                    "balance",
                                    formatAmount(balance, currency)),
                                Map.of("uid", uid),
                                pushId + '_' + balance);
                        }
                        sendPush(
                            context,
                            uid,
                            "/am/push/family-limits-member",
                            Collections.emptyMap(),
                            "family_change",
                            "limit-change-to-max-for-close",
                            Map.of(
                                "balance",
                                formatAmount(balance, currency)),
                            Collections.emptyMap(),
                            pushId + '_' + balance);
                    } else {
                        String range =
                            limitMode.toString().toLowerCase(Locale.ROOT);
                        if (notifyAdmin) {
                            sendPush(
                                context,
                                adminUid,
                                "/am/push/family-limits",
                                Map.of("member_uid", uid),
                                "family_change",
                                "limit-change-with-range-for-admin",
                                Map.of(
                                    "limit",
                                    formatAmount(limitValue, currency),
                                    "balance",
                                    formatAmount(balance, currency),
                                    "range", range),
                                Map.of("uid", uid),
                                pushId + '_' + limitValue);
                        }
                        sendPush(
                            context,
                            uid,
                            "/am/push/family-limits-member",
                            Collections.emptyMap(),
                            "family_change",
                            "limit-change-for-close",
                            Map.of(
                                "limit",
                                formatAmount(limitValue, currency),
                                "balance",
                                formatAmount(balance, currency),
                                "range", range),
                            Collections.emptyMap(),
                            pushId + '_' + limitValue);
                    }
                }
            } else if (newFamily == null) {
                FamilyInfo familyInfo = oldFamily.familyInfo();
                session.logger().info(
                    "Familypay dropped for " + uid
                    + ", family info was: " + familyInfo);
                long adminUid = familyInfo.adminUid();
                if (notifyAdmin) {
                    notifyAdminOnFamilyStop(context, adminUid, pushId);
                }
                sendPush(
                    context,
                    uid,
                    "/am/push/family",
                    Collections.emptyMap(),
                    "family_change",
                    "off-family-card-for-close",
                    Map.of("action", "disable"),
                    Map.of("uid", adminUid),
                    pushId);
            } else {
                FamilyInfo familyInfo = oldFamily.familyInfo();
                session.logger().info(
                    "Familypay disabled for " + uid
                    + ", family info was: " + familyInfo);
                long adminUid = familyInfo.adminUid();
                if (notifyAdmin) {
                    sendPush(
                        context,
                        adminUid,
                        "/am/push/family",
                        Collections.emptyMap(),
                        "family_change",
                        "lost-card-access-for-admin",
                        Collections.emptyMap(),
                        Map.of("uid", uid),
                        pushId);
                }
                sendPush(
                    context,
                    uid,
                    "/am/push/family",
                    Collections.emptyMap(),
                    "family_change",
                    "lost-card-access-for-close",
                    Map.of("action", "restrict"),
                    Map.of("uid", adminUid),
                    pushId);
            }
        } else if (newEnabled) {
            FamilyInfo familyInfo = newFamily.familyInfo();
            long adminUid = familyInfo.adminUid();
            String currency = newMemberInfo.limitCurrency();
            session.logger().info(
                "Familypay enabled for " + uid
                + ", family info: " + familyInfo);
            LimitsInfo newLimits = newMemberInfo.limits();
            LimitMode limitMode =
                LimitMode.detectLimitMode(newLimits);
            long limitValue = limitMode.limitValue(newLimits);
            if (newLimits.unlim() || newMemberInfo.unlim()) {
                if (notifyAdmin) {
                    sendPush(
                        context,
                        adminUid,
                        "/am/push/family-limits",
                        Map.of("member_uid", uid),
                        "family_change",
                        "get-card-access-unlimit-for-admin",
                        Map.of("action", "get"),
                        Map.of("uid", uid),
                        pushId);
                }
                sendPush(
                    context,
                    uid,
                    "/am/push/family-limits-member",
                    Collections.emptyMap(),
                    "family_change",
                    "get-card-access-unlimit-for-close",
                    Map.of("action", "share"),
                    Map.of("uid", adminUid),
                    pushId);
            } else if (limitMode == LimitMode.TOTAL) {
                if (notifyAdmin) {
                    sendPush(
                        context,
                        adminUid,
                        "/am/push/family-limits",
                        Map.of("member_uid", uid),
                        "family_change",
                        "get-card-access-without-range-for-admin",
                        Map.of(
                            "limit", formatAmount(limitValue, currency),
                            "action", "get"),
                        Map.of("uid", uid),
                        pushId + '_' + limitValue);
                }
                sendPush(
                    context,
                    uid,
                    "/am/push/family-limits-member",
                    Collections.emptyMap(),
                    "family_change",
                    "get-card-access-without-range-for-close",
                    Map.of(
                        "limit", formatAmount(limitValue, currency),
                        "action", "share"),
                    Map.of("uid", adminUid),
                    pushId + '_' + limitValue);
            } else {
                if (notifyAdmin) {
                    sendPush(
                        context,
                        adminUid,
                        "/am/push/family-limits",
                        Map.of("member_uid", uid),
                        "family_change",
                        "get-card-access-for-admin",
                        Map.of(
                            "limit", formatAmount(limitValue, currency),
                            "action", "get",
                            "range",
                            limitMode.toString().toLowerCase(Locale.ROOT)),
                        Map.of("uid", uid),
                        pushId + '_' + limitValue);
                }
                sendPush(
                    context,
                    uid,
                    "/am/push/family-limits-member",
                    Collections.emptyMap(),
                    "family_change",
                    "get-card-access-for-close",
                    Map.of(
                        "limit", formatAmount(limitValue, currency),
                        "action", "share",
                        "range",
                        limitMode.toString().toLowerCase(Locale.ROOT)),
                    Map.of("uid", adminUid),
                    pushId + '_' + limitValue);
            }
        } else {
            session.logger().info(
                "Family pay still unavailable for " + uid);
            context.tskvLogger().log(
                context.tskvRecord(
                    TskvFields.Stage.INTERMEDIATE,
                    "Familypay still not available for user"));
            if (newMemberInfo == null) {
                // Member with disabled familypay left family with familypay
                notifyOnMemberDelete(
                    context,
                    oldFamily.familyInfo().adminUid(),
                    oldMemberInfo.uid(),
                    pushId);
            } else if (oldMemberInfo == null) {
                // Member with disabled familypay joined family with familypay
                notifyOnMemberAdd(
                    context,
                    newFamily.familyInfo().adminUid(),
                    newMemberInfo.uid(),
                    pushId);
            }
        }
    }

    public void paymentCompleted(
        final RequestContext context,
        final UserInfo userInfo,
        final String paymentId,
        final long amount,
        final long serviceId)
    {
        ProxySession session = context.session();
        String uri;
        try {
            PctEncoder encoder = new PctEncoder(PctEncodingRule.PATH);
            encoder.process(paymentId.toCharArray());
            uri = "/order-history/order/" + encoder;
        } catch (Exception e) {
            session.logger().log(
                Level.WARNING,
                "Failed to construct push uri for payment " + paymentId
                + " and user "
                + JsonType.NORMAL.toString(
                    userInfo.toJson(BasicContainerFactory.INSTANCE, false)),
                e);
            return;
        }
        FamilyMember memberInfo = userInfo.memberInfo();
        String currency = memberInfo.limitCurrency();
        Map<String, Object> requestParams =
            Map.of(
                "amount", amount / 100d,
                "service_id", serviceId,
                "currency", currency,
                "cardMask", userInfo.cardInfo().cardMask());
        LimitsInfo limits = memberInfo.limits();
        if (limits.unlim() || memberInfo.unlim()) {
            String amountString = formatAmount(amount, currency);
            sendPush(
                context.addPrefix(
                    TskvFields.RECIPIENT_UID,
                    Long.toString(userInfo.familyAdminUid())),
                userInfo.familyAdminUid(),
                uri,
                requestParams,
                "pay_notification",
                "success-pay-unlimit-for-admin",
                Map.of("amount", amountString),
                Map.of(
                    "uid", memberInfo.uid(),
                    "service_id", serviceId),
                paymentId);
            sendPush(
                context.addPrefix(
                    TskvFields.RECIPIENT_UID,
                    Long.toString(memberInfo.uid())),
                memberInfo.uid(),
                uri,
                requestParams,
                "pay_notification",
                "success-pay-unlimit-for-close",
                Map.of("amount", amountString),
                Map.of("service_id", serviceId),
                paymentId);
        } else {
            LimitMode limitMode = LimitMode.detectLimitMode(limits);
            long limitValue = limitMode.limitValue(limits);
            long expenses = limitMode.limitValue(memberInfo.expenses());
            long balance = limitValue - expenses;
            sendPush(
                context.addPrefix(
                    TskvFields.RECIPIENT_UID,
                    Long.toString(userInfo.familyAdminUid())),
                userInfo.familyAdminUid(),
                uri,
                requestParams,
                "pay_notification",
                "success-pay-for-admin",
                Map.of(
                    "amount", formatAmount(amount, currency),
                    "balance", formatAmount(balance, currency)),
                Map.of(
                    "uid", memberInfo.uid(),
                    "service_id", serviceId),
                paymentId);
            sendPush(
                context.addPrefix(
                    TskvFields.RECIPIENT_UID,
                    Long.toString(memberInfo.uid())),
                memberInfo.uid(),
                uri,
                requestParams,
                "pay_notification",
                "success-pay-for-close",
                Map.of(
                    "amount", formatAmount(amount, currency),
                    "balance", formatAmount(balance, currency)),
                Map.of("service_id", serviceId),
                paymentId);
            if (balance < limitValue / 10L) {
                sendPush(
                    context.addPrefix(
                        TskvFields.RECIPIENT_UID,
                        Long.toString(userInfo.familyAdminUid())),
                    userInfo.familyAdminUid(),
                    "/am/push/family-limits",
                    Map.of("member_uid", memberInfo.uid()),
                    "pay_notification",
                    "low-balance-for-admin",
                    Collections.emptyMap(),
                    Map.of("uid", memberInfo.uid()),
                    paymentId);
            }
        }
    }

    public void paymentRejected(
        final RequestContext context,
        final UserInfo userInfo,
        final String paymentId,
        final long amount,
        final long serviceId)
    {
        LimitMode limitMode =
            LimitMode.detectLimitMode(userInfo.memberInfo().limits());
        long limitValue = limitMode.limitValue(userInfo.memberInfo().limits());
        long expenses =
            limitMode.limitValue(userInfo.memberInfo().expenses());
        long balance = limitValue - expenses;
        String currency = userInfo.memberInfo().limitCurrency();
        sendPush(
            context.addPrefix(
                TskvFields.RECIPIENT_UID,
                Long.toString(userInfo.familyAdminUid())),
            userInfo.familyAdminUid(),
            "/am/push/family-limits",
            Map.of("member_uid", userInfo.memberInfo().uid()),
            "pay_notification",
            "fail-pay-for-admin",
            Map.of(
                "amount", formatAmount(amount, currency),
                "balance", formatAmount(balance, currency)),
            Map.of(
                "uid", userInfo.memberInfo().uid(),
                "service_id", serviceId),
            paymentId);
        sendPush(
            context.addPrefix(
                TskvFields.RECIPIENT_UID,
                Long.toString(userInfo.memberInfo().uid())),
            userInfo.memberInfo().uid(),
            "/am/push/family-limits-member",
            Collections.emptyMap(),
            "pay_notification",
            "fail-pay-for-close",
            Map.of(
                "amount", formatAmount(amount, currency),
                "balance", formatAmount(balance, currency)),
            Map.of("service_id", serviceId),
            paymentId);
    }

    public void paymentFailed(
        final RequestContext context,
        final UserInfo userInfo,
        final String paymentId,
        final long amount,
        final long serviceId)
    {
        String currency = userInfo.memberInfo().limitCurrency();
        sendPush(
            context.addPrefix(
                TskvFields.RECIPIENT_UID,
                Long.toString(userInfo.familyAdminUid())),
            userInfo.familyAdminUid(),
            "/am/push/family",
            Collections.emptyMap(),
            "pay_notification",
            "card-trouble-for-admin",
            Map.of("amount", formatAmount(amount, currency)),
            Map.of(
                "uid", userInfo.memberInfo().uid(),
                "service_id", serviceId),
            paymentId);
        sendPush(
            context.addPrefix(
                TskvFields.RECIPIENT_UID,
                Long.toString(userInfo.memberInfo().uid())),
            userInfo.memberInfo().uid(),
            "/am/push/family",
            Collections.emptyMap(),
            "pay_notification",
            "card-trouble-for-close",
            Map.of("amount", formatAmount(amount, currency)),
            Map.of("service_id", serviceId),
            paymentId);
    }

    private static class AmountFormatter {
        private final NumberFormat kopeksFormat;
        private final NumberFormat roundNumberFormat;

        AmountFormatter(final Locale locale) {
            kopeksFormat = NumberFormat.getCurrencyInstance(locale);
            roundNumberFormat = (NumberFormat) kopeksFormat.clone();
            roundNumberFormat.setMinimumFractionDigits(0);
        }

        public String format(final long amount) {
            if ((amount % 100L) == 0L) {
                return roundNumberFormat.format(amount / 100L);
            } else {
                return kopeksFormat.format(amount / 100d);
            }
        }
    }

    private static class PushResultFutureCallback<T>
        implements FutureCallback<T>
    {
        private final FutureCallback<? super T> callback;
        private final RequestContext context;
        private final String uri;
        private final String body;

        PushResultFutureCallback(
            final FutureCallback<? super T> callback,
            final RequestContext context,
            final String uri,
            final String body)
        {
            this.callback = callback;
            this.context = context;
            this.uri = uri;
            this.body = body;
        }

        @Override
        public void cancelled() {
            context.tskvLogger().log(
                context.tskvRecord(
                    TskvFields.Stage.INTERMEDIATE,
                    "Push request cancelled: " + uri + ' ' + body));
            callback.cancelled();
        }

        @Override
        public void completed(final T result) {
            context.tskvLogger().log(
                context.tskvRecord(
                    TskvFields.Stage.INTERMEDIATE,
                    "Push request completed: " + uri + ' ' + body));
            callback.completed(result);
        }

        @Override
        public void failed(final Exception e) {
            context.tskvLogger().log(
                context.tskvRecord(
                    TskvFields.Stage.INTERMEDIATE,
                    "Push request failed: " + uri + ' ' + body
                    + ':' + ' ' + e));
            callback.failed(e);
        }
    }
}

