package ru.yandex.passport.familypay.backend;

import java.util.Objects;

import org.apache.http.protocol.HttpCoreContext;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.http.proxy.PrefixedProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.http.util.server.LoggingServerConnection;
import ru.yandex.http.util.server.SessionContext;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.tskv.BasicPrefixedTskvLogger;
import ru.yandex.tskv.TskvString;
import ru.yandex.util.timesource.TimeSource;

public class RequestContext {
    private static final DateTimeFormatter TIMESTAMP_FORMATTER =
        DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZZ")
            .withZone(DateTimeZone.forID("Europe/Moscow"));

    private final FamilypayBackend server;
    private final ProxySession session;
    private final boolean quiet;
    private final long sessionStartTimestamp;
    private final JsonType jsonType;
    private final BasicPrefixedTskvLogger tskvLogger;

    public RequestContext(
        final FamilypayBackend server,
        final ProxySession session,
        final TskvString tskvPrefix,
        final String input,
        final boolean quiet)
        throws BadRequestException
    {
        this.server = server;
        this.session = session;
        this.quiet = quiet;
        sessionStartTimestamp = TimeSource.INSTANCE.currentTimeMillis();
        jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        tskvPrefix.append(TskvFields.HOSTNAME, SessionContext.HOSTNAME);
        tskvPrefix.append(
            TskvFields.SESSION_START_TIMESTAMP_MS,
            TimeSource.INSTANCE.currentTimeMillis());
        tskvPrefix.append(
            TskvFields.TVM_SRC_ID,
            Objects.toString(
                session.context().getAttribute(HttpServer.TVM_SRC_ID)));
        tskvPrefix.append(
            TskvFields.TVM_USER_UID,
            Objects.toString(
                session.context().getAttribute(HttpServer.TVM_USER_UID)));
        tskvPrefix.append(
            TskvFields.SESSION_ID,
            Objects.toString(
                session.context().getAttribute(HttpServer.SESSION_ID)));
        tskvPrefix.append(
            TskvFields.X_REQUEST_ID,
            session.headers().getString(YandexHeaders.X_REQUEST_ID, ""));
        tskvPrefix.append(
            TskvFields.REQUEST_URI,
            session.request().getRequestLine().getUri());

        LoggingServerConnection conn = (LoggingServerConnection)
            session.context().getAttribute(HttpCoreContext.HTTP_CONNECTION);
        tskvPrefix.append(
            TskvFields.REMOTE_IP,
            Objects.toString(conn.getRemoteAddress()));

        this.tskvLogger = server.tskvLogger().prefix(tskvPrefix);
        if (!quiet) {
            this.tskvLogger.log(tskvRecord(TskvFields.Stage.INITIAL, input));
        }
    }

    private RequestContext(
        final RequestContext sample,
        final String key,
        final String value)
    {
        server = sample.server;
        session = new PrefixedProxySession(sample.session, value);
        quiet = sample.quiet;
        sessionStartTimestamp = sample.sessionStartTimestamp;
        jsonType = sample.jsonType;
        tskvLogger =
            sample.tskvLogger.prefix(
                sample.tskvLogger.record().append(key, value));
    }

    public RequestContext addPrefix(final String key, final String value) {
        return new RequestContext(this, key, value);
    }

    public TskvString tskvRecord(
        final TskvFields.Stage stage,
        final String message)
    {
        long timestamp = TimeSource.INSTANCE.currentTimeMillis();
        return tskvLogger.record()
            .append(TskvFields.UNIXTIME, timestamp / 1000L)
            .append(TskvFields.TIMESTAMP_MS, timestamp)
            .append(
                TskvFields.SESSION_LENGTH_MS,
                timestamp - sessionStartTimestamp)
            .append(TskvFields.STAGE, stage.toString())
            .append(
                TskvFields.TIMESTAMP,
                TIMESTAMP_FORMATTER.print(timestamp))
            .append(TskvFields.MESSAGE, message);
    }

    public FamilypayBackend server() {
        return server;
    }

    public ProxySession session() {
        return session;
    }

    public boolean quiet() {
        return quiet;
    }

    public JsonType jsonType() {
        return jsonType;
    }

    public BasicPrefixedTskvLogger tskvLogger() {
        return tskvLogger;
    }
}

