package ru.yandex.passport.familypay.backend;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonMap;

public class UserInfoHandler extends UserHandlerBase {
    private static final SqlQuery USER_INFO =
        new SqlQuery("user-info.sql", UserInfoHandler.class);

    public UserInfoHandler(final FamilypayBackend server) {
        super(server, true);
    }

    @Override
    protected void handle(final long uid, final RequestContext context)
        throws BadRequestException
    {
        handle(uid, context, false);
    }

    protected void handle(
        final long uid,
        final RequestContext context,
        final boolean singleLimit)
        throws BadRequestException
    {
        findUser(
            uid,
            context.session().params().getLong("service_id"),
            new UserCallback(context, singleLimit));
    }

    public static void findUser(
        final long uid,
        final long serviceId,
        final FamilypayCallback<UserInfo> callback)
    {
        String service =
            callback.context().server().services().serviceName(
                serviceId,
                uid);
        callback.context().session().logger().info(
            "Service resolved from service id " + serviceId
            + ':' + ' ' + service);
        if (service == null) {
            callback.failed(
                ErrorType.SERVICE_ID_NOT_FOUND,
                Long.toString(serviceId),
                null);
            return;
        }
        Tuple tuple = Tuple.tuple();
        tuple.addLong(uid);
        tuple.addString(service);
        callback.context().server().pgClient().executeOnMaster(
            USER_INFO,
            tuple,
            callback.context().session().listener(),
            new Callback(callback, uid));
    }

    private static class Callback
        extends AbstractFilterFamilypayCallback<RowSet<Row>, UserInfo>
    {
        private final long uid;

        Callback(final FamilypayCallback<UserInfo> callback, final long uid) {
            super(callback);
            this.uid = uid;
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            try {
                if (rowSet.rowCount() <= 0) {
                    failed(ErrorType.USER_NOT_FOUND, null, null);
                } else {
                    callback.completed(
                        UserInfo.fromRow(rowSet.iterator().next(), uid, true));
                }
            } catch (RuntimeException e) {
                failed(
                    ErrorType.INTERNAL_ERROR,
                    "Failed to parse database response",
                    e);
            }
        }
    }

    private static class UserCallback
        extends AbstractFamilypayCallback<UserInfo>
    {
        private final boolean singleLimit;

        UserCallback(final RequestContext context, final boolean singleLimit) {
            super(context);
            this.singleLimit = singleLimit;
        }

        @Override
        public void completed(final UserInfo userInfo) {
            JsonMap result =
                userInfo.toJson(BasicContainerFactory.INSTANCE, singleLimit);
            sendResponse(result);
        }
    }
}

