#include "coroutine_helper.h"

#include <library/cpp/coroutine/engine/network.h>

#include <util/generic/vector.h>
#include <util/generic/xrange.h>
#include <util/network/socket.h>

static void AddFds(const fd_set& set, int type, int maxfd, TVector<int>& what, TVector<SOCKET>& fds) {
#ifdef WIN32
    (void)maxfd;
    for (size_t i : xrange(set.fd_count)) {
        fds.push_back(set.fd_array[i]);
        what.push_back(type);
    }
#else
    for (size_t i : xrange(maxfd)) {
        if (FD_ISSET(i, &set)) {
            fds.push_back(i);
            what.push_back(type);
        }
    }
#endif
}

bool CoroutineWait(fd_set& readers, fd_set& writers, int nfds, size_t count, const TInstant& deadline, TCont* cont) {
    TVector<int> what(Reserve(count));
    TVector<SOCKET> fds(Reserve(count));

    AddFds(readers, CONT_POLL_READ, nfds, what, fds);
    AddFds(writers, CONT_POLL_WRITE, nfds, what, fds);
    auto result = NCoro::SelectD(cont, fds.data(), what.data(), fds.size(), nullptr, deadline);

    if (result != 0)
        ythrow TSystemError(result);
    return result == 0;
}

int CoroutineSelect(fd_set& readers, fd_set& writers, fd_set& excep, int nfds, const TDuration& timeout, TCont* cont) {
    if(nfds == 0) {
        cont->SleepT(TDuration::MilliSeconds(100));
        return 0;
    }

    TVector<int> what(Reserve(nfds));
    TVector<SOCKET> fds(Reserve(nfds));

    for (int i : xrange(nfds)) {
        int w = 0;
        if(FD_ISSET(i, &readers))
            w |= POLLIN;
        if(FD_ISSET(i, &writers))
            w |= POLLOUT;
        if(FD_ISSET(i, &excep))
            w |= POLLERR;

        if(w) {
            fds.emplace_back(i);
            what.emplace_back(w);
        }
    }

    return NCoro::SelectT(cont, fds.data(), what.data(), fds.size(), nullptr, timeout);
}
