#pragma once

#include <util/generic/ptr.h>
#include <util/generic/maybe.h>
#include <util/generic/deque.h>
#include <util/generic/string.h>
#include <util/generic/hash_set.h>
#include <util/stream/null.h>
#include <util/stream/str.h>
#include <library/cpp/http/misc/httpcodes.h>
#include <util/system/fs.h>
#include <util/datetime/base.h>
#include <util/system/mutex.h>
#include <mail/so/libs/pool/pool.h>

typedef void CURL;
typedef void CURLSH;
template <>
void TDelete::Destroy<struct curl_slist>(curl_slist* t) noexcept;

namespace NConfig{
    struct TDict;
}

namespace NCurl {

    struct TRequestContext {
        TRequestContext& SetHost(TString v);
        TRequestContext& SetPort(ui16 v);
        TRequestContext& SetRequestTimeout(TDuration v);
        TRequestContext& SetConnectTimeout(TDuration v);
        TRequestContext& SetPostData(TString v);

        TRequestContext& SetContentType(TStringBuf contentType);
        template <class TContainer>
        TRequestContext& AddHeaders(TContainer&& v) {
            for (const auto& h : v)
                AddHeader(h);
            return *this;
        }
        TRequestContext& AddHeader(const TString& v);

        TRequestContext() noexcept = default;
        TRequestContext(TRequestContext&&) noexcept = default;
        TRequestContext& operator=(TRequestContext&&) noexcept = default;
        TRequestContext(const TRequestContext&) = delete;
        TRequestContext& operator=(const TRequestContext&) = delete;

        TMaybe<TString> url;
        TMaybe<ui16> port;
        TMaybe<TDuration> requestTimeout, connectTimeout;
        TMaybe<TString> data;
        THolder<curl_slist> curlHeaders;
    };

    struct TTimings {
        explicit TTimings(CURL* curl);

        friend IOutputStream& operator<<(IOutputStream& stream, const TTimings& timings);

        TDuration Total = TDuration::Max();
        TDuration NsLookup = TDuration::Max();
        TDuration Connect = TDuration::Max();
        TDuration PreTransfer = TDuration::Max();
        TDuration Redirect = TDuration::Max();
        TDuration AppConnect = TDuration::Max();
        TDuration StartTransfer = TDuration::Max();
    };

    struct TArtifacts {
        using THeaders = TDeque<TString>;
        using TBodyOutput = IOutputStream;

        TString firstLine;
        THeaders headers;
        HttpCodes code = HTTP_OK;
        TBodyOutput& bodyWriter;
        TVector<TTimings> Timings;

        explicit TArtifacts(IOutputStream& bodyWriter)
            : bodyWriter(bodyWriter) {
        }
        virtual ~TArtifacts() = default;
    };

    struct TSimpleArtifacts : TArtifacts {
        TStringStream body;
        explicit TSimpleArtifacts()
            : TArtifacts(body) {
        }
    };

    struct TSSL {
        TSSL() = default;
        explicit TSSL(const NConfig::TDict& config);

        bool Empty() const {
            return !certFile && !passPhrase && !keyFile && !caCertFile;
        }

        TSSL& SetCertFile(TString v) noexcept {
            NFs::EnsureExists(v);
            certFile = std::move(v);
            return *this;
        }
        TSSL& SetPassphrase(TString v) noexcept {
            passPhrase = std::move(v);
            return *this;
        }
        TSSL& SetKeyFile(TString v) noexcept {
            NFs::EnsureExists(v);
            keyFile = std::move(v);
            return *this;
        }
        TSSL& SetCACertFile(TString v) noexcept {
            NFs::EnsureExists(v);
            caCertFile = std::move(v);
            return *this;
        }
        TString certFile;
        TString passPhrase;
        TString keyFile;
        TString caCertFile;
    };

    struct TRetries {
        TRetries() = default;
        TRetries(size_t count, TDuration timeToWaitBeforeRetry, const THashSet<HttpCodes>& unretriableCodes={}) noexcept : Count(count),  TimeToWaitBeforeRetry(timeToWaitBeforeRetry), UnretriableCodes(unretriableCodes) {}
        explicit TRetries(const NConfig::TDict& config);
        size_t Count = 0;
        TDuration TimeToWaitBeforeRetry = TDuration::Zero();
        THashSet<HttpCodes> UnretriableCodes;
    };

    struct TOptions {
        TOptions() = default;
        explicit TOptions(const NConfig::TDict& config);

        TOptions& SetMaxAliveConnections(size_t count);
        TOptions& SetParseHeaders(bool v);
        TOptions& SetParseBody(bool v);

        TMaybe<size_t> maxAliveConnections;
        bool parseHeaders = true;
        bool parseBody = true;
    };

    struct TShare : public TNonCopyable {
        CURLSH* Get() {
            return share.Get();
        };
        const CURLSH* Get() const {
            return share.Get();
        };

        explicit TShare(bool multithread = true);

        struct TDestroyer {
            static void Destroy(CURLSH* c);
        };
        TVector<TMutex> locks;
        THolder<CURLSH, TDestroyer> share;
    };

    struct TError : public yexception {
        friend IOutputStream& operator<<(IOutputStream& stream, const TError& error) {
            return stream << error.AsStrBuf();
        }
        using yexception::yexception;
    };

    struct TCurlBase {
        TCurlBase() noexcept;

        static void Prepare(CURL* easy, TArtifacts::TBodyOutput& bodyWriter, TArtifacts::THeaders& headers);
        static void PrepareAndStoreArtifact(CURL* easy, TArtifacts& artifacts);

        [[nodiscard]] static TMaybe<TError> ProcessResult(CURL* easy, TArtifacts& artifacts, ui32 code);
        [[nodiscard]] static TMaybe<TError> LoadArtifactAndProcessResult(CURL* easy, ui32 code);
    };

    class TCurl : private TCurlBase {
    public:
        [[nodiscard]] TMaybe<TError> Perform(TArtifacts& artifacts);
        TCurl& Reset();

        TCurl& Setup(const TOptions& options);
        TCurl& Setup(TRequestContext context) noexcept;
        TCurl& Setup(TMaybe<TRetries> retryConfig);
        TCurl& Setup(const TSSL& ssl);
        TCurl& SetShare(TAtomicSharedPtr<TShare> share);

        CURL* Native() const noexcept;

        explicit TCurl(const TOptions& options = {});
        TCurl(TCurl&& other) noexcept = default;
        TCurl& operator=(TCurl&& other) noexcept = default;
        TCurl(const TCurl&) = delete;
        TCurl& operator=(const TCurl&) = delete;

    public:
        TStringBuf GetEffectiveURL() const noexcept;
        TDuration GetTotalTime() const noexcept;
        TDuration GetConnectTime() const noexcept;

    private:
        TMaybe<TError> PerformWithRetries(TArtifacts& artifacts);

    private:
        void SetMaxConnections(size_t connections);
        TRequestContext requestContext;

    private:
        TAtomicSharedPtr<TShare> share;
        struct TDestroyer {
            static void Destroy(CURL* c);
        };
        THolder<CURL, TDestroyer> context;
        TMaybe<TRetries> retryOpts;
    };

    class TPool : public ::TPool<TCurl> {
        using TBase = ::TPool<TCurl>;
    public:
        using TBase::TBase;
    };

    using TPoolItemHolder = TPool::TItemHolder;

    class TPoolTraits : public TPool::TPoolItemTraits {
    public:
        THolder<TCurl> create() const override;

        static TCurl& extract(TPool::TItemHolder& holder);

        explicit TPoolTraits() = default;
        explicit TPoolTraits(const NConfig::TDict& config);
        explicit TPoolTraits(TOptions options, TMaybe<TSSL> ssl = {});
        explicit TPoolTraits(TMaybe<TSSL> ssl, TMaybe<TRetries> retries);
        TPoolTraits(TPoolTraits&&) noexcept = default;
        ~TPoolTraits() override = default;

    private:
        TOptions options;
        TMaybe<TSSL> ssl;
        TMaybe<TRetries> retries;
    };
} //  namespace NCurl
