#include <util/generic/xrange.h>
#include "coroutine_helper.h"
#include <contrib/libs/curl/include/curl/curl.h>
#include <library/cpp/coroutine/engine/impl.h>
#include <library/cpp/iterator/zip.h>
#include <util/string/builder.h>

#include "multi.h"

namespace NCurl {
    TMultiCurl::TMultiCurl()
        : context(curl_multi_init()) {
        Y_ENSURE(context);
    }

    void TMultiCurl::TDestroyer::Destroy(CURLM* context) {
        curl_multi_cleanup(context);
    }

    TMultiCurl::TLock::TLock(CURL* easy, CURLM* multi)
        : easy(easy)
        , multi(multi) {
        curl_multi_add_handle(multi, easy);
    }

    TMultiCurl::TLock::~TLock() {
        curl_multi_remove_handle(multi.Get(), easy.Get());
    }

    [[nodiscard]] static TMaybe<TError> CheckCURLMcode(CURLMcode code) {
        if (code != CURLM_OK)
            return TError{} << curl_multi_strerror(code);
        return Nothing();
    }

    TVector<TMaybe<TError>> TMultiCurl::Perform(const TInstant& deadline, const TVector<TCurlArtPair>& curlsArts, TCont* cont) {
        TVector<TLock> locks(Reserve(curlsArts.size()));

        for (auto [curlRef, artRef] : curlsArts) {
            TCurl& curl = curlRef;
            TArtifacts& art = artRef;
            PrepareAndStoreArtifact(curl.Native(), art);
            locks.emplace_back(curl.Native(), context.Get());
        }

        int running = 0;
        if(auto error = CheckCURLMcode(curl_multi_perform(context.Get(), &running)))
            return {std::move(error)};

        const TDuration timeout = Min([this] {
            long timeo;
            curl_multi_timeout(context.Get(), &timeo);
            return timeo < 0 ? TDuration::MilliSeconds(5) : TDuration::MilliSeconds(timeo);
        }(), Now() - deadline);

        while (running) {
            int rc = Select(timeout, cont);
            switch (rc) {
                case -1:
                    return {TError{} << "select error " << LastSystemErrorText(errno)};
                case 0:
                    if(deadline < Now())
                        return {TError{} << "timeout exceed"};
                default:
                    if(auto error = CheckCURLMcode(curl_multi_perform(context.Get(), &running)))
                        return {std::move(error)};
                    break;
            }
        }

        TVector<TMaybe<TError>> errors(curlsArts.size());

        int msgs_left = 0;
        while (CURLMsg* msg = curl_multi_info_read(context.Get(), &msgs_left)) {
            if (msg->msg == CURLMSG_DONE) {

                auto it = FindIf(curlsArts, [handle = msg->easy_handle](const auto& curlArt){
                    return ((const TCurl&)curlArt.first).Native() == handle;
                });

                Y_VERIFY(it != curlsArts.end());

                errors[std::distance(curlsArts.begin(), it)] = LoadArtifactAndProcessResult(msg->easy_handle, msg->data.result);
            }
        }

        return errors;
    }

    int TMultiCurl::Select(const TDuration& timeout, TCont* cont) {
        return (cont == nullptr) ? SimpleSelect(timeout) : CoroSelect(timeout, cont);
    }

    int TMultiCurl::SimpleSelect(const TDuration& timeout) {
        int timeout_real = static_cast<int>(std::min<ui64>(std::numeric_limits<int>::max(), timeout.MilliSeconds()));
        return curl_multi_wait(context.Get(), nullptr, 0, timeout_real, nullptr);
    }

    int TMultiCurl::CoroSelect(const TDuration& timeout, TCont* cont) {
        fd_set fdread = {};
        FD_ZERO(&fdread);
        fd_set fdwrite = {};
        FD_ZERO(&fdwrite);
        fd_set fdexcep = {};
        FD_ZERO(&fdexcep);

        int maxfd = -1;
        if(auto r = curl_multi_fdset(context.Get(), &fdread, &fdwrite, &fdexcep, &maxfd); r != CURLM_OK)
            return r;
        return ::CoroutineSelect(fdread, fdwrite, fdexcep, maxfd + 1, timeout, cont);
    }

    THolder<TMultiCurl> TMultiPoolTraits::create() const {
        return MakeHolder<TMultiCurl>();
    }
} // namespace NCurl
