#include <util/generic/xrange.h>
#include "storage.h"

namespace NCurl {
    static const NCurl::TMultiPoolTraits multiTraits;
    TStorage::TStorage(const TPoolParams& pool)
        : TStorage(pool, pool)
    { }

    TStorage::TStorage(const TPoolParams& easy, const TPoolParams& multi)
        : pool(easy)
        , multi(multi)
    { }

    static NCurl::TSimpleArtifacts ProcessSingle(NCurl::TRequestContext request, NCurl::TPool& pool, const NCurl::TPoolTraits& traits, const TInstant& deadline) {
        NCurl::TSimpleArtifacts artifact;
        if(auto error = pool.get(traits, deadline).Get().Setup(std::move(request)).Perform(artifact))
            ythrow TWithBackTrace<yexception>() << *error;
        return artifact;
    }

    static TVector<NCurl::TSimpleArtifacts> ProcessMulti(TVector<NCurl::TPool::TItemHolder>& curls, NCurl::TMultiPool& multi, const TInstant& deadline, TCont* cont) {
        TVector<NCurl::TSimpleArtifacts> artifacts(curls.size());
        auto m = multi.get(multiTraits, deadline, cont);

        TVector<NCurl::TCurlArtPair> pairs(Reserve(curls.size()));
        for(size_t i : xrange(curls.size())) {
            pairs.emplace_back(std::ref(curls[i].Get()), std::ref(artifacts[i]));
        }

        for(const auto& error : m.Get().Perform(deadline, pairs, cont)) {
            if (error) {
                ythrow TWithBackTrace<yexception>() << *error;
            }
        }
        return artifacts;
    }

    NCurl::TSimpleArtifacts TStorage::Process(NCurl::TRequestContext request, const NCurl::TPoolTraits& traits, const TInstant& deadline, TCont* cont) {
        if (cont == nullptr) {
            return ProcessSingle(std::move(request), pool, traits, deadline);
        }

        TVector<NCurl::TPool::TItemHolder> curls;
        curls.emplace_back(pool.get(traits, deadline, cont)).Get().Setup(std::move(request));
        return std::move(ProcessMulti(curls, multi, deadline, cont).front());
    }

    TVector<NCurl::TSimpleArtifacts> TStorage::Process(TVector<NCurl::TRequestContext> requests, const NCurl::TPoolTraits& traits, const TInstant& deadline, TCont* cont) {
        if (requests.empty()) {
            return {};
        }

        if (requests.size() == 1 && cont == nullptr) {
            return{ 1, ProcessSingle(std::move(requests.front()), pool, traits, deadline) };
        }

        TVector<NCurl::TPool::TItemHolder> curls(Reserve(requests.size()));
        for (auto& item : requests) {
            curls.emplace_back(pool.get(traits, deadline, cont)).Get().Setup(std::move(item));
        }

        return ProcessMulti(curls, multi, deadline, cont);
    }
}
