#include "wrapper.h"

#include <mail/so/libs/jniwrapper_base/jniwrapper_base.h>

#include <mlp/mail/text_deobfuscator/lib/text_deobfuscator.h>

#include <library/cpp/yconf/conf.h>

#include <util/generic/ptr.h>
#include <util/generic/string.h>
#include <util/generic/yexception.h>

namespace NDeobfuscatorWrapper {
    BEGIN_CONFIG(TDeobfuscatorConfig)
        BEGIN_TOPSECTION(Deobfuscator)
        END_SECTION()
    END_CONFIG()

    NTextDeobfuscate::TTextDeobfuscator* CreateDeobfuscator(TYandexConfig& config) {
        auto deobfuscator = MakeHolder<NTextDeobfuscate::TTextDeobfuscator>(true, true);
        bool empty = true;
        TYandexConfig::Section* section = config.GetFirstChild("Deobfuscator");
        while (section) {
            empty = false;
            const TYandexConfig::Directives& directives = section->GetDirectives();
            TString remapPath;
            if (!directives.GetValue("RemapPath", remapPath)) {
                ythrow TWithBackTrace<yexception>() << "'RemapPath' not found in config";
            }
            TString triePath;
            if (!directives.GetValue("TriePath", triePath)) {
                ythrow TWithBackTrace<yexception>() << "'TriePath' not found in config";
            }
            deobfuscator->AddWordDeobfuscator(remapPath, triePath);
            section = section->Next;
            while (section) {
                if (section->Parsed() && stricmp(section->Name, "Deobfuscator") == 0) {
                    break;
                }
                section = section->Next;
            }
        }
        if (empty) {
            return nullptr;
        } else {
            deobfuscator->Initialize();
            return deobfuscator.Release();
        }
    }
}

extern "C"
int JniWrapperCreateDeobfuscator(const char* config, void** out) noexcept {
    try {
        NDeobfuscatorWrapper::TDeobfuscatorConfig configParser;
        if (!configParser.ParseMemory(config)) {
            TString message{TString::Join("Failed to parse config:\n", config, "\n")};
            configParser.PrintErrors(message);
            *out = strdup(message.c_str());
            return -1;
        }
        *out = CreateDeobfuscator(configParser);
        if (*out == nullptr) {
            *out = strdup("No deobfuscator configs set");
            return -1;
        }
    } catch (...) {
        return NJniWrapper::ProcessJniWrapperException((char**) out);
    }
    return 0;
}

extern "C"
void JniWrapperDestroyDeobfuscator(void* instance) noexcept {
    delete static_cast<NTextDeobfuscate::TTextDeobfuscator*>(instance);
}

extern "C"
int JniWrapperDeobfuscateText(
    void* instance,
    const char* uri,
    const char* metainfo Y_DECLARE_UNUSED,
    const void* data Y_DECLARE_UNUSED,
    size_t size Y_DECLARE_UNUSED,
    char** out) noexcept
{
    const NTextDeobfuscate::TTextDeobfuscator* deobfuscator =
        static_cast<const NTextDeobfuscate::TTextDeobfuscator*>(instance);
    try {
        TString text;
        deobfuscator->Replace(uri, text, false);
        *out = strdup(text.c_str());
        if (*out) {
            return 0;
        } else {
            return -1;
        }
    } catch (...) {
        return NJniWrapper::ProcessJniWrapperException(out);
    }
}

