#include "geobase_lookup.h"

#include <mail/so/libs/jniwrapper_base/jniwrapper_base.h>

#include <geobase/include/lookup.hpp>
#include <geobase/include/structs.hpp>

#include <library/cpp/json/writer/json.h>
#include <library/cpp/threading/thread_local/thread_local.h>

#include <util/generic/hash.h>
#include <util/generic/noncopyable.h>
#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/generic/utility.h>
#include <util/generic/vector.h>
#include <util/string/vector.h>

extern "C"
int JniWrapperCreateGeobaseLookup(const char* config, void** out) noexcept {
    try {
        *out = new NGeobase::NImpl::TLookup(config);
    } catch (...) {
        return NJniWrapper::ProcessJniWrapperException((char**) out);
    }
    return 0;
}

extern "C"
void JniWrapperDestroyGeobaseLookup(void* instance) noexcept {
    delete static_cast<NGeobase::NImpl::TLookup*>(instance);
}

class TGeobaseCache: public TNonCopyable {
private:
    typedef THashMap<std::string, TString> TMapType;
    const size_t MaxSize;
    TMapType Barrel1;
    TMapType Barrel2;
    TMapType* Master;
    TMapType* Slave;
    size_t MasterUniqs;

private:
    inline void CheckSizeLimit() {
        if (Master->size() >= MaxSize) {
            DoSwap(Master, Slave);
            MasterUniqs = 0;
            Master->clear();
        }
    }

public:
    inline TGeobaseCache(size_t maxSize)
        : MaxSize(maxSize)
        , Master(&Barrel1)
        , Slave(&Barrel2)
        , MasterUniqs(0)
    {
    }

    inline bool Get(const std::string& str, TString& result) {
        auto iter = Master->find(str);
        if (iter == Master->end()) {
            iter = Slave->find(str);
            if (iter == Slave->end()) {
                return false;
            } else {
                Master->insert(*iter);
                // It is important to store type before size limit check,
                // because it could clear slave and invalidate iterator
                result = iter->second;
                CheckSizeLimit();
            }
        } else {
            result = iter->second;
        }
        return true;
    }

    inline void Put(const std::string& str, TString&& result) {
        Master->emplace(str, std::move(result));
        ++MasterUniqs;
        CheckSizeLimit();
    }

    inline size_t Size() noexcept {
        return MasterUniqs + Slave->size();
    }
};

static const size_t MaxCacheSize = 16384;
static NThreading::TThreadLocalValue<TGeobaseCache> GeobaseCache;

extern "C"
int JniWrapperGetIpInfoFromGeobase(
    void* instance,
    const char* ipPtr,
    const char* metainfo Y_DECLARE_UNUSED,
    void* data Y_DECLARE_UNUSED,
    size_t size Y_DECLARE_UNUSED,
    char** out) noexcept
{
    try {
        std::string ip(ipPtr);
        TGeobaseCache& geobaseCache = GeobaseCache.GetRef(MaxCacheSize);
        TString result;
        if (geobaseCache.Get(ip, result)) {
            *out = strdup(result.c_str());
        } else {
            auto lookup =
                static_cast<const NGeobase::NImpl::TLookup*>(instance);
            result = NGeobaseLookup::GetIpInfoFromGeobase(lookup, ip);
            *out = strdup(result.c_str());
            geobaseCache.Put(ip, std::move(result));
        }
        if (*out) {
            return 0;
        } else {
            return -1;
        }
    } catch (...) {
        return NJniWrapper::ProcessJniWrapperException(out);
    }
}

TString NGeobaseLookup::GetIpInfoFromGeobase(
    const NGeobase::NImpl::TLookup* lookup,
    const std::string& ip)
{
    NGeobase::NImpl::TIpTraits ipTraits{lookup->GetTraitsByIp(ip)};
    NGeobase::NImpl::TRegion region{lookup->GetRegionByIp(ip)};
    NJsonWriter::TBuf writer(NJsonWriter::HEM_UNSAFE);
    writer.BeginObject();

    writer.WriteKey(TStringBuf("ip_traits"));
    writer.BeginObject();
    writer.WriteKey(TStringBuf("is_yandex_net"));
    writer.WriteBool(ipTraits.IsYandexNet());
    writer.WriteKey(TStringBuf("is_yandex_staff"));
    writer.WriteBool(ipTraits.IsYandexStaff());
    writer.WriteKey(TStringBuf("is_yandex_turbo"));
    writer.WriteBool(ipTraits.IsYandexTurbo());
    writer.WriteKey(TStringBuf("is_tor"));
    writer.WriteBool(ipTraits.IsTor());
    writer.WriteKey(TStringBuf("is_proxy"));
    writer.WriteBool(ipTraits.IsProxy());
    writer.WriteKey(TStringBuf("is_vpn"));
    writer.WriteBool(ipTraits.IsVpn());
    writer.WriteKey(TStringBuf("is_hosting"));
    writer.WriteBool(ipTraits.IsHosting());
    writer.WriteKey(TStringBuf("is_mobile"));
    writer.WriteBool(ipTraits.IsMobile());
    writer.WriteKey(TStringBuf("isp_name"));
    writer.WriteString(ipTraits.IspName);
    writer.WriteKey(TStringBuf("org_name"));
    writer.WriteString(ipTraits.OrgName);
    writer.WriteKey(TStringBuf("asn_list"));
    writer.BeginList();
    TVector<TString> asnList{SplitString(ipTraits.AsnList.c_str(), ",")};
    for (const auto& asn: asnList) {
        if (!asn.empty()) {
            writer.WriteString(TString::Join(TStringBuf("AS"), asn));
        }
    }
    writer.EndList();
    writer.EndObject();

    writer.WriteKey(TStringBuf("region_info"));
    writer.BeginObject();

    NGeobase::NImpl::TRegion country{region};
    while (country.GetEType() > NGeobase::ERegionType::COUNTRY) {
        country = lookup->GetRegionById(country.GetParentId());
    }
    TString countryName;
    TString isoName;
    if (country.GetEType() == NGeobase::ERegionType::COUNTRY) {
        if (country.IsFieldExists("iso_name")) {
            isoName = country.GetStrField("iso_name");
        }
        if (country.IsFieldExists("en_name")) {
            countryName = country.GetStrField("en_name");
        }
    }

    writer.WriteKey(TStringBuf("iso_name"));
    writer.WriteString(isoName);

    TString name;
    if (region.IsFieldExists("en_name")) {
        name = region.GetStrField("en_name");
    }
    writer.WriteKey(TStringBuf("name"));
    writer.WriteString(name);

    TString cityName;
    const NGeobase::NImpl::Id cityId = region.GetCityId();
    if (region.GetId() == cityId) {
        cityName = name;
    } else {
        NGeobase::NImpl::TRegion city{lookup->GetRegionById(cityId)};
        if (city.IsFieldExists("en_name")) {
            cityName = city.GetStrField("en_name");
        }
    }

    writer.WriteKey(TStringBuf("id"));
    writer.WriteInt(region.GetId());

    writer.WriteKey(TStringBuf("city"));
    writer.WriteString(cityName);

    writer.WriteKey(TStringBuf("city_id"));
    writer.WriteInt(cityId);

    writer.WriteKey(TStringBuf("country"));
    writer.WriteString(countryName);

    writer.WriteKey(TStringBuf("country_id"));
    writer.WriteInt(country.GetId());

    double latitude = 0;
    if (region.IsFieldExists("latitude")) {
        latitude = region.GetDoubleField("latitude");
    }
    writer.WriteKey(TStringBuf("latitude"));
    writer.WriteDouble(latitude);

    double longitude = 0;
    if (region.IsFieldExists("longitude")) {
        longitude = region.GetDoubleField("longitude");
    }
    writer.WriteKey(TStringBuf("longitude"));
    writer.WriteDouble(longitude);

    TString tzname;
    if (region.IsFieldExists("tzname")) {
        tzname = region.GetStrField("tzname");
    }
    writer.WriteKey(TStringBuf("tzname"));
    writer.WriteString(tzname);

    writer.EndObject();

    writer.EndObject();
    return writer.Str();
}

