#pragma once

#include <util/generic/deque.h>
#include <util/generic/maybe.h>
#include <util/generic/vector.h>
#include <library/cpp/json/json_value.h>
#include <mail/so/libs/primitives/types.h>
#include <util/datetime/base.h>

namespace NUnperson {
    class TLemmerCache;
}

namespace NHtmlSanMisc {
    class TOriginal : public TString {
        using TString::TString;
    };

    class TUnperson : public TString {
    public:
        TUnperson() = default;

        explicit TUnperson(const TOriginal &original);

        TUnperson &operator=(const TOriginal &original);
    };

    class TClipped : public TString {
    public:
        TClipped() = default;

        explicit TClipped(const TUnperson &unperson);

        TClipped &operator=(const TUnperson &unperson);
    };

    struct TUnpersonSet {
        TUnpersonSet() = default;

        explicit operator bool() const;

        TUnpersonSet &operator=(TOriginal original) noexcept;

        TUnpersonSet &operator=(TUnperson unperson) noexcept;

        TOriginal Original;
        TUnperson Unperson;
        TClipped Clipped;
    };

    struct TPersonalFilter {
        static TMaybe<TPersonalFilter> Parse(const NJson::TJsonValue &val);

        friend IOutputStream &operator<<(IOutputStream &stream, const TPersonalFilter &pf);

        TString LastType;
        ui32 Hams{};
        ui32 Spams{};
    };

    struct TTabPf {
        static TMaybe<TTabPf> Parse(const NJson::TJsonValue &val);

        friend IOutputStream &operator<<(IOutputStream &stream, const TTabPf &pf);

        TString LastTab;
    };

    struct TAbookInfo {
        TAbookInfo() = default;

        static TAbookInfo Parse(NJson::TJsonValue val);

        friend IOutputStream &operator<<(IOutputStream &stream, const TAbookInfo &ab);

        NJson::TJsonValue Original;

        ui32 SentCount{};
        ui32 ReceivedCount{};
        ui32 ReceivedReadCount{};
        ui32 DomainSentCount{};
        ui32 DomainReceivedCount{};
        ui32 DomainReceivedReadCount{};
        ui32 TimesContacted{};
        TString SourceType;
    };

    struct TSender {
        static TSender Parse(NJson::TJsonValue val);

        friend IOutputStream &operator<<(IOutputStream &stream, const TSender &sender);

        NJson::TJsonValue Original;
        TUid Uid;
        TSuid Suid;
        TLogin Login;
        TDomain Domain;

        TMaybe<TPersonalFilter> PersonalFilter;
        TMaybe<TTabPf> TabPf;
        TAbookInfo AbookInfo;

        NJson::TJsonValue RawUserMlFeatures;
        NJson::TJsonValue RawUserMlEmbeddings;

        bool ValidInReplyTo;
        bool ValidReferences;
    };

    struct TAddress {
        TAddress() = default;

        static TAddress Parse(const NJson::TJsonValue &jsonValue);

        friend IOutputStream &operator<<(IOutputStream &stream, const TAddress &address);

        TString Address;
        TInstant BornDate;
        bool Default{};
        bool Native{};
        bool Rpop{};
        bool Unsafe{};
        bool Validated{};
    };

    struct TLegacy;

    struct TUserInfo {
        TUserInfo() = default;

        static TUserInfo Parse(TLogin email, const NJson::TJsonValue &jsonValue);

        const TLegacy &AsLegacy() const;

        friend IOutputStream &operator<<(IOutputStream &stream, const TUserInfo &userInfo);

        TLogin Email;
        TVector<TAddress> AddressList;
        TVector<TString> PddAliases;
        bool IsMailList{};
        TSuid Suid;
        TString Country;
        TInstant RegDate;
        bool Hosted{};
        bool Lite{};
        TLogin Login;
        TUid Uid;
        TMaybe<TInstant> PhoneConfirmationDate;
        int Karma{};
        int KarmaStatus{};
    };
    struct TLegacy : TUserInfo {
        friend IOutputStream &operator<<(IOutputStream &stream, const TLegacy &legacy);
    };

    struct TDoc {
        TDoc() = default;

        static TDoc Parse(const NJson::TJsonValue &jsonValue);

        TUnpersonSet PureBody;
        TUnpersonSet Subject;
        TUnpersonSet BodyText;

        TString htmlBody;
        TString contentType;
        TString normalizedFrom;
        TString displayName;
        TString attachName;
        TString md5;
        bool hasAttachsize{};
        TString hid;
        TVector<TString> urls;
        TString UgcId;
        TString Ugc;
        TMaybe<float> pureBodyWmdDistance;
        TString pureBodyWmdNeighbourId;
        TString pureBodyWmdNeighbourLabels;

        TMaybe<float> subjectWmdDistance;
        TString subjectWmdNeighbourId;
        TString subjectWmdNeighbourLabels;

        TMaybe<float> attachNameWmdDistance;
        TString attachNameWmdNeighbourId;
        TString attachNameWmdNeighbourLabels;

        TInstant gatewayReceivedDate;

        NJson::TJsonValue Source;
    };

    struct TMatchedTemplate {
        TMatchedTemplate() = default;

        static TMatchedTemplate Parse(const NJson::TJsonValue& jsonValue);

        TDeque<TVector<TString>> CalculateHashes() const;

        static TString HashesAsJson(TDeque<TVector<TString>> hashes);

        long long StableSign{};
        TVector<TString> delta;
        NJson::TJsonValue Source;
    };

    struct TAnswer {
        TAnswer() = default;
        explicit TAnswer(TString queueId,
                         TVector<TDoc> docs,
                         TVector<TSender> senders,
                         TString hbfProjectId,
                         THashMap<TLogin, TUserInfo> userInfos,
                         TMaybe<TVector<float>> mailDssmEmbedding,
                         TVector<TString> dkimDomains,
                         TMaybe<NJson::TJsonValue> dkimStats,
                         TMaybe<TMatchedTemplate> matchedTemplate,
                         TMaybe<float> dssmDistance,
                         TString dssmNeighbourId,
                         TString dssmNeighbourLabels,
                         TMaybe<float> dssm2Distance,
                         TString dssm2NeighbourId,
                         TString dssm2NeighbourLabels,
                         TMaybe<float> dssmSubjectDistance,
                         TString dssmSubjectNeighbourId,
                         TString dssmSubjectNeighbourLabels,
                         TMaybe<bool> hasCryptaUserVector,
                         TLogin mailFromLogin,
                         TMaybe<TUserInfo> mailFromUserinfo,
                         TMaybe<NJson::TJsonValue> mailFromJsonUserinfo,
                         bool allFromSameOrgId,
                         TMaybe<NJson::TJsonValue> recipientsUserinfos,
                         TMaybe<NJson::TJsonValue> activityInfo,
                         NJson::TJsonValue senderMlFeatures,
                         NJson::TJsonValue senderMlEmbeddings,
                         TVector<TString> yaDiskInfos,
                         NJson::TJsonValue source) noexcept;

        static TAnswer Parse(const NJson::TJsonValue &jsonValue);

        const TDoc *FindTrueHtmlPart() const;

        const TDoc *FindVisiblePart() const;

        const TVector<TDoc> &Docs() const;

        const TVector<TSender> &GetSenders() const { return Senders; }

        const TSender* FindSenderByUid(const TUid& uid) const;

        const THashMap<TLogin, TUserInfo> &GetUserInfos() const { return UserInfos; }

        const TString &GetHbfProjectId() const;

        const TMaybe<TVector<float>> &GetMailDssmEmbedding() const;

        const TVector<TString>& GetDkimDomains() const;

        const TMaybe<NJson::TJsonValue>& GetDkimStats() const;

        const TMaybe<TMatchedTemplate> &GetMatchedTemplate() const { return MatchedTemplate; }

        const TMaybe<float>& GetDssmDistance() const { return DssmDistance; }

        const TString& GetDssmNeighbourId() const { return DssmNeighbourId; }

        const TString& GetDssmNeighbourLabels() const { return DssmNeighbourLabels; }

        const TMaybe<float>& GetDssm2Distance() const { return Dssm2Distance; }

        const TString& GetDssm2NeighbourId() const { return Dssm2NeighbourId; }

        const TString& GetDssm2NeighbourLabels() const { return Dssm2NeighbourLabels; }

        const TMaybe<float>& GetDssmSubjectDistance() const { return DssmSubjectDistance; }

        const TString& GetDssmSubjectNeighbourId() const { return DssmSubjectNeighbourId; }

        const TString& GetDssmSubjectNeighbourLabels() const { return DssmSubjectNeighbourLabels; }

        const TMaybe<bool>& GetHasCryptaUserVector() const { return HasCryptaUserVector; }

        const TString& GetQueueId() const { return QueueId; }

        const TLogin& GetMailFromLogin() const { return MailFromLogin; }

        const TMaybe<TUserInfo>& GetMailFromUserinfo() const { return MailFromUserinfo; }

        const TMaybe<NJson::TJsonValue>& GetMailFromJsonUserinfo() const { return MailFromJsonUserinfo; }

        bool GetAllFromSameOrgId() const { return AllFromSameOrgId; }

        const TMaybe<NJson::TJsonValue>& GetRecipientsUserinfos() const { return RecipientsUserinfos; }

        const TMaybe<NJson::TJsonValue>& GetActivityInfo() const { return ActivityInfo; }

        const NJson::TJsonValue& GetSenderMlFeatures() const { return SenderMlFeatures; }

        const NJson::TJsonValue& GetSenderMlEmbeddings() const { return SenderMlEmbeddings; }

        const TVector<TString>& GetYaDiskInfos() const { return YaDiskInfos; }

        const NJson::TJsonValue& GetSource() const { return Source; }

    private:
        TVector<TDoc> docs;
        TVector<TSender> Senders;
        mutable TMaybe<const TDoc *> visiblePart;
        TString HbfProjectId;
        THashMap<TLogin, TUserInfo> UserInfos;
        TMaybe<TVector<float>> MailDssmEmbedding;
        TVector<TString> DkimDomains;
        TMaybe<NJson::TJsonValue> DkimStats;
        TMaybe<TMatchedTemplate> MatchedTemplate;

        TMaybe<float> DssmDistance;
        TString DssmNeighbourId;
        TString DssmNeighbourLabels;

        TMaybe<float> Dssm2Distance;
        TString Dssm2NeighbourId;
        TString Dssm2NeighbourLabels;

        TMaybe<float> DssmSubjectDistance;
        TString DssmSubjectNeighbourId;
        TString DssmSubjectNeighbourLabels;

        TMaybe<bool> HasCryptaUserVector;

        TString QueueId;

        TLogin MailFromLogin;
        TMaybe<TUserInfo> MailFromUserinfo;
        TMaybe<NJson::TJsonValue> MailFromJsonUserinfo;
        bool AllFromSameOrgId{};
        TMaybe<NJson::TJsonValue> RecipientsUserinfos;
        TMaybe<NJson::TJsonValue> ActivityInfo;
        NJson::TJsonValue SenderMlFeatures;
        NJson::TJsonValue SenderMlEmbeddings;
        TVector<TString> YaDiskInfos;

        NJson::TJsonValue Source;
    };
} // namespace NHtmlSanMisc


Y_DECLARE_OUT_SPEC(inline, NHtmlSanMisc::TOriginal, s, v) { s << (const TString &) v; }

Y_DECLARE_OUT_SPEC(inline, NHtmlSanMisc::TUnperson, s, v) { s << (const TString &) v; }

Y_DECLARE_OUT_SPEC(inline, NHtmlSanMisc::TClipped, s, v) { s << (const TString &) v; }
