#include <library/cpp/testing/unittest/registar.h>
#include <mail/so/libs/html_sanitizer_misc/html_sanitizer_misc.h>
#include <library/cpp/json/json_reader.h>
#include <util/memory/blob.h>
#include <util/stream/file.h>
#include <library/cpp/testing/common/env.h>

Y_UNIT_TEST_SUITE(HtmlSanMiscTest) {
    using namespace NHtmlSanMisc;

    Y_UNIT_TEST(Main) {
        TFileInput stream(ArcadiaSourceRoot() +
                          "/mail/so/daemons/so2/so2/test/resources/ru/yandex/mail/so2/senders-so-response2.json");
        const TAnswer answer = TAnswer::Parse(NJson::ReadJsonTree(&stream, true));

        {
            const TVector<float> expected{0.10497,-0.02137943,-0.04814029,-0.10708023,5.676466E-5,-0.1020606,-0.02913166,0.10802248,-0.030209636,-0.030713191,-0.0014729317,-0.043176394,-0.12275905,-0.085216716,0.10527486,-0.06493091,-7.01757E-4,-1.0293835E-4,-0.02388493,-0.03062809,-0.0286277,0.0073051793,0.037110817,-0.020583577,-0.04603083,0.05919816,-0.026336329,-0.07143246,-2.8565092E-4,-0.020750755,0.09265288,0.028559212,0.038760666,-0.64015913,0.0010249426,0.051727995,0.124990866,-8.9937326E-5,-0.033946294,-0.046326507,0.04500864,0.048848104,0.020545827,-0.01355164,0.02334503,0.033121917,-0.22887482,0.019906195,-0.035542693,-9.961224E-4,-0.043685615,0.027959298,-1.2327275E-4,-0.016730689,0.15935987,-0.018098228,0.046281632,0.081052065,-0.0031474438,0.042622812,0.07991569,0.0072015096,0.09962737,0.05550275,0.033790506,0.07933341,-0.048883986,-0.24782209,0.057933647,0.16880532,0.12540218,-0.025016328,-0.11548603,0.0069551445,6.0120964E-4,0.062831365,0.14655125,0.039747067,2.026224E-4,-0.07354648,0.07856572,-6.525192E-4,-0.15852188,0.13312559,0.056205705,0.052351758,-0.009628636,-0.048423365,-0.10080063,0.1386579,-0.003188506,-0.15226462,-0.19055358,-0.10917566,0.0337462,0.06411741,-1.3535402E-5,0.0016142547,0.020452796,-0.025016043};
            UNIT_ASSERT_EQUAL(expected, answer.GetMailDssmEmbedding());
            UNIT_ASSERT_EQUAL(true, answer.GetActivityInfo().Defined());
        }

        {
            const TVector<TDoc>& docs = answer.Docs();

            UNIT_ASSERT_VALUES_EQUAL(1, docs.size());

            const TDoc& doc = docs.front();
            UNIT_ASSERT_VALUES_EQUAL("Hello, world", doc.PureBody.Original);
            UNIT_ASSERT_VALUES_EQUAL("Hello, world", doc.PureBody.Unperson);

            UNIT_ASSERT_VALUES_EQUAL("", doc.Subject.Original);
            UNIT_ASSERT_VALUES_EQUAL("", doc.Subject.Unperson);

            UNIT_ASSERT_VALUES_EQUAL("", doc.BodyText.Original);
            UNIT_ASSERT_VALUES_EQUAL("", doc.BodyText.Unperson);

            UNIT_ASSERT_VALUES_EQUAL("", doc.htmlBody);

            UNIT_ASSERT_VALUES_EQUAL("text/plain", doc.contentType);

            UNIT_ASSERT_VALUES_EQUAL("arttech051@gmail.com", doc.normalizedFrom);
            UNIT_ASSERT_VALUES_EQUAL("Art Tech", doc.displayName);

            UNIT_ASSERT_VALUES_EQUAL("", doc.attachName);

            UNIT_ASSERT_VALUES_EQUAL("", doc.md5);

            UNIT_ASSERT_VALUES_EQUAL(false, doc.hasAttachsize);

            UNIT_ASSERT_VALUES_EQUAL("1", doc.hid);

            UNIT_ASSERT_VALUES_EQUAL(TVector<TString>{}, doc.urls);

            UNIT_ASSERT_VALUES_EQUAL("", doc.UgcId);

            UNIT_ASSERT_VALUES_EQUAL("", doc.Ugc);

            UNIT_ASSERT_VALUES_EQUAL(0.9037391543388367, doc.pureBodyWmdDistance);

            UNIT_ASSERT_VALUES_EQUAL(Nothing(), doc.subjectWmdDistance);

            UNIT_ASSERT_VALUES_EQUAL("spam_samples_991949281_so_compains_171981210770334057",
                                     doc.pureBodyWmdNeighbourId);

            UNIT_ASSERT_VALUES_EQUAL(TInstant::Seconds(1583399650), doc.gatewayReceivedDate);
        }

        {
            const TVector<TSender>& senders = answer.GetSenders();

            UNIT_ASSERT_VALUES_EQUAL(4, senders.size());

            const TSender& sender = senders[1];

            UNIT_ASSERT_VALUES_EQUAL("183442461", sender.Suid);
            UNIT_ASSERT_VALUES_EQUAL("67061658", sender.Uid);
            UNIT_ASSERT_VALUES_EQUAL("arttech051@gmail.com", sender.Login);
            UNIT_ASSERT_VALUES_EQUAL("gmail.com", sender.Domain);

            const TPersonalFilter& pf = *sender.PersonalFilter;
            UNIT_ASSERT_VALUES_EQUAL(0, pf.Spams);
            UNIT_ASSERT_VALUES_EQUAL(1, pf.Hams);
            UNIT_ASSERT_VALUES_EQUAL("ham", pf.LastType);

            const TAbookInfo& abook = sender.AbookInfo;
            UNIT_ASSERT_VALUES_EQUAL(3, abook.SentCount);
            UNIT_ASSERT_VALUES_EQUAL(2, abook.ReceivedCount);
            UNIT_ASSERT_VALUES_EQUAL(2, abook.ReceivedReadCount);
            UNIT_ASSERT_VALUES_EQUAL(18, abook.DomainSentCount);
            UNIT_ASSERT_VALUES_EQUAL(23, abook.DomainReceivedCount);
            UNIT_ASSERT_VALUES_EQUAL(10, abook.DomainReceivedReadCount);
            UNIT_ASSERT_VALUES_EQUAL(3, abook.TimesContacted);
            UNIT_ASSERT_VALUES_EQUAL("ya_sent", abook.SourceType);

            UNIT_ASSERT_VALUES_EQUAL(true, sender.ValidInReplyTo);
            UNIT_ASSERT_VALUES_EQUAL(true, sender.ValidReferences);

            const TSender& anotherSender = senders[2];

            UNIT_ASSERT_VALUES_EQUAL(false, anotherSender.ValidInReplyTo);
            UNIT_ASSERT_VALUES_EQUAL(false, anotherSender.ValidReferences);

            const TSender& firstSender = senders[0];

            UNIT_ASSERT_VALUES_EQUAL(true, firstSender.ValidInReplyTo);
            UNIT_ASSERT_VALUES_EQUAL(false, firstSender.ValidReferences);
        }
        {
            const THashMap<TLogin, TUserInfo>& infos = answer.GetUserInfos();

            UNIT_ASSERT_VALUES_EQUAL(50, infos.size());

            const TUserInfo& userInfo = infos.at("analizer@yandex.ru");

            UNIT_ASSERT_VALUES_EQUAL(1, userInfo.AddressList.size());
            {
                const TAddress& address = userInfo.AddressList.front();

                UNIT_ASSERT_VALUES_EQUAL("Analizer@yandex.ru", address.Address);
                UNIT_ASSERT_VALUES_EQUAL(TInstant::MilliSeconds(1062696865000ull), address.BornDate);
                UNIT_ASSERT_VALUES_EQUAL(true, address.Default);
                UNIT_ASSERT_VALUES_EQUAL(true, address.Native);
                UNIT_ASSERT_VALUES_EQUAL(false, address.Rpop);
                UNIT_ASSERT_VALUES_EQUAL(false, address.Unsafe);
                UNIT_ASSERT_VALUES_EQUAL(true, address.Validated);
            }

            UNIT_ASSERT_VALUES_EQUAL(TVector<TString>{}, userInfo.PddAliases);
            UNIT_ASSERT_VALUES_EQUAL(false, userInfo.IsMailList);
            UNIT_ASSERT_VALUES_EQUAL("12054080", userInfo.Suid);
            UNIT_ASSERT_VALUES_EQUAL("ru", userInfo.Country);
            UNIT_ASSERT_VALUES_EQUAL(TInstant::ParseIso8601("2003-09-04 21:34:25"), userInfo.RegDate);
            UNIT_ASSERT_VALUES_EQUAL(false, userInfo.Hosted);
            UNIT_ASSERT_VALUES_EQUAL(false, userInfo.Lite);
            UNIT_ASSERT_VALUES_EQUAL("Analizer", userInfo.Login);
            UNIT_ASSERT_VALUES_EQUAL("5598601", userInfo.Uid);
            UNIT_ASSERT_VALUES_EQUAL(TInstant::Seconds(1505134059), userInfo.PhoneConfirmationDate);
            UNIT_ASSERT_VALUES_EQUAL(0, userInfo.Karma);
            UNIT_ASSERT_VALUES_EQUAL(6000, userInfo.KarmaStatus);
        }
        {
            auto map = answer.GetDkimStats()->GetMapSafe();
            UNIT_ASSERT_VALUES_EQUAL(5, map["total"].GetInteger());
        }
        UNIT_ASSERT_EQUAL(false, answer.GetHasCryptaUserVector().Defined());
        {
            auto map = answer.GetRecipientsUserinfos()->GetMapSafe();
            auto submap = map["Analizer+tag1@yandex.ru"].GetMapSafe();
            UNIT_ASSERT_VALUES_EQUAL(5598601, submap["uid"].GetInteger());
        }
    }

    Y_UNIT_TEST(Empty) {
        TFileInput stream(ArcadiaSourceRoot() +
                          "/mail/so/libs/html_sanitizer_misc/ut/empty_so2_answer.json");
        TAnswer::Parse(NJson::ReadJsonTree(&stream, true));
    }

    Y_UNIT_TEST(CryptaUserVector) {
        {
            TFileInput stream(
                ArcadiaSourceRoot() + "/mail/so/daemons/so2/so2/test/resources/ru/yandex/mail/so2/senders-so-response1.json");
            const TAnswer answer = TAnswer::Parse(NJson::ReadJsonTree(&stream, true));
            UNIT_ASSERT_EQUAL(true, answer.GetHasCryptaUserVector().Defined());
            UNIT_ASSERT_EQUAL(true, *answer.GetHasCryptaUserVector());
            UNIT_ASSERT_VALUES_EQUAL(false, answer.GetAllFromSameOrgId());

            auto dkimDomains = answer.GetDkimDomains();
            UNIT_ASSERT_VALUES_EQUAL(1, dkimDomains.size());
            UNIT_ASSERT_STRINGS_EQUAL("n-seminar.com", dkimDomains[0]);

            UNIT_ASSERT_VALUES_EQUAL(4, answer.GetSenderMlFeatures()["USER_FACTORS_HAM_LAST_2_WEEKS"].GetInteger());
            UNIT_ASSERT_EQUAL(false, answer.GetActivityInfo().Defined());
            UNIT_ASSERT_VALUES_EQUAL(2, answer.GetYaDiskInfos().size());
        }

        {
            TFileInput stream(
                ArcadiaSourceRoot() + "/mail/so/daemons/so2/so2/test/resources/ru/yandex/mail/so2/senders-so-response4.json");
            const TAnswer answer = TAnswer::Parse(NJson::ReadJsonTree(&stream, true));
            UNIT_ASSERT_EQUAL(true, answer.GetHasCryptaUserVector().Defined());
            UNIT_ASSERT_EQUAL(false, *answer.GetHasCryptaUserVector());
        }

    }

    Y_UNIT_TEST(AllFromSameOrgId) {
        TFileInput stream(
            ArcadiaSourceRoot() + "/mail/so/daemons/so2/so2/test/resources/ru/yandex/mail/so2/missing-mailfrom-so-response.json");
        const TAnswer answer = TAnswer::Parse(NJson::ReadJsonTree(&stream, true));
        UNIT_ASSERT_VALUES_EQUAL(true, answer.GetAllFromSameOrgId());
    }

    Y_UNIT_TEST(QueueId) {
            TFileInput stream(
                ArcadiaSourceRoot() + "/mail/so/daemons/so2/so2/test/resources/ru/yandex/mail/so2/tcp-info-rdns-so-response.json");
            const TAnswer answer = TAnswer::Parse(NJson::ReadJsonTree(&stream, true));
            UNIT_ASSERT_VALUES_EQUAL("5pNc1V9Q85-TFke4JuM", answer.GetQueueId());
    }

    Y_UNIT_TEST(TabPf) {
            TFileInput stream(
                ArcadiaSourceRoot() + "/mail/so/daemons/clip/clip/test/resources/ru/yandex/mail/so/clip/move1-so-response.json");
            const TAnswer answer = TAnswer::Parse(NJson::ReadJsonTree(&stream, true));
            const TVector<TSender>& senders = answer.GetSenders();

            UNIT_ASSERT_VALUES_EQUAL(1, senders.size());

            const TSender& sender = senders[0];

            const TTabPf& pf = *sender.TabPf;
            UNIT_ASSERT_VALUES_EQUAL("news", pf.LastTab);
    }
}
