package ru.yandex.data.compressor;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.base64.Base64;
import ru.yandex.base64.Base64Encoder;
import ru.yandex.base64.Base64Parser;

public interface Compressor {
    Pattern RE_COMMAS = Pattern.compile(",+");
    byte[] compress(final byte[] data) throws CompressorException;

    byte[] uncompress(final byte[] data) throws CompressorException;

    default String compressAndBase64(final String data) throws CompressorException {
        return compressAndBase64(data, Base64.INSTANCE);
    }

    default String compressAndBase64(final String data, final Base64 base64Type) throws CompressorException {
        Base64Encoder encoder = new Base64Encoder(base64Type);
        encoder.process(compress(data.getBytes(StandardCharsets.UTF_8)));
        return encoder.toString();
    }

    default String compressAndBase64(final byte[] data) throws CompressorException {
        return compressAndBase64(data, Base64.INSTANCE);
    }

    default String compressAndBase64(final byte[] data, final Base64 base64Type) throws CompressorException {
        Base64Encoder encoder = new Base64Encoder(base64Type);
        encoder.process(compress(data));
        return encoder.toString();
    }

    default String unbase64AndUncompress(final String data) throws CompressorException {
        return unbase64AndUncompress(data, Base64.INSTANCE);
    }

    default String unbase64AndUncompress(final String data, final Base64 type) throws CompressorException {
        if (data.length() == 0) {
            return null;
        }
        return new String(uncompress(base64DecodeUneven(data, type)), StandardCharsets.UTF_8);
    }

    default byte[] base64DecodeUneven(final String s) throws CompressorException {
        return base64DecodeUneven(s, Base64.INSTANCE);
    }

    default byte[] base64DecodeUneven(final String s, final Base64 base64Type) throws CompressorException {
        // workaround for stupid Arcadia's base64-url-encoding
        // see: https://a.yandex-team.ru/arc_vcs/library/cpp/string_utils/base64/base64.cpp
        Matcher m = RE_COMMAS.matcher(s);
        String data = m.find(0) ? s.substring(0, m.start()) : s;

        Base64Parser parser = Base64Parser.valueOf(base64Type.name());
        try {
            if (data.length() % 4 == 0) {
                return parser.apply(data);
            }
            return parser.apply(data + "=".repeat(4 - (data.length() % 4)));
        } catch (final IOException e) {
            throw new CompressorException("Base64-" + base64Type + "-decoding failed: " + e, e);
        }
    }
}
