#include <mail/so/libs/jniwrapper_base/jniwrapper_base.h>

#include <contrib/libs/lzo2/include/lzo/lzo1.h>
#include <contrib/libs/lzo2/include/lzo/lzo1a.h>
#include <contrib/libs/lzo2/include/lzo/lzo1b.h>
#include <contrib/libs/lzo2/include/lzo/lzo1c.h>
#include <contrib/libs/lzo2/include/lzo/lzo1f.h>
#include <contrib/libs/lzo2/include/lzo/lzo1x.h>
#include <contrib/libs/lzo2/include/lzo/lzo1y.h>
#include <contrib/libs/lzo2/include/lzo/lzo1z.h>
#include <contrib/libs/lzo2/include/lzo/lzo2a.h>

#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/generic/yexception.h>
#include <util/memory/blob.h>
#include <util/memory/tempbuf.h>

#include <array>

#include <util/system/tls.h>
#include <util/string/join.h>
#include <util/string/strip.h>

#define UNCOMPRESS_SUSPICIOUS_SIZE 5242880

class LzoCompressor {
private:
    std::array<unsigned char, LZO1X_999_MEM_COMPRESS + 1> m_WorkMem;
    const bool m_InitOk;

    static TString LZO_E_to_string(int e) {
        switch (e) {
            case LZO_E_OK:
                return "OK";
            case LZO_E_ERROR:
                return "ERROR";
            case LZO_E_OUT_OF_MEMORY:
                return "OUT_OF_MEMORY";
            case LZO_E_NOT_COMPRESSIBLE:
                return "NOT_COMPRESSIBLE";
            case LZO_E_INPUT_OVERRUN:
                return "INPUT_OVERRUN";
            case LZO_E_OUTPUT_OVERRUN:
                return "OUTPUT_OVERRUN";
            case LZO_E_LOOKBEHIND_OVERRUN:
                return "LOOKBEHIND_OVERRUN";
            case LZO_E_EOF_NOT_FOUND:
                return "EOF_NOT_FOUND";
            case LZO_E_INPUT_NOT_CONSUMED:
                return "INPUT_NOT_CONSUMED";
            case LZO_E_NOT_YET_IMPLEMENTED:
                return "NOT_YET_IMPLEMENTED";
            case LZO_E_INVALID_ARGUMENT:
                return "INVALID_ARGUMENT";
            case LZO_E_INVALID_ALIGNMENT:
                return "INVALID_ALIGNMENT";
            case LZO_E_OUTPUT_NOT_CONSUMED:
                return "OUTPUT_NOT_CONSUMED";
            case LZO_E_INTERNAL_ERROR:
                return "INTERNAL_ERROR";
            default:
                ythrow yexception() << "unknown error code: " << e;
        }
    }

public:
    LzoCompressor() noexcept
        : m_WorkMem(), m_InitOk((lzo_init() == LZO_E_OK))
    {}

    TVector<ui8> Compress(const unsigned char* in, size_t len, unsigned char algorithm, int constraint) const {
        TVector<ui8> result;
        if (!m_InitOk) {
            ythrow yexception() << "Can not init LZO-engine";
        }

        if (len == 0) {
            return result;
        }

        const lzo_uint outlen = len + len / 64 + 16 + 3;
        result.resize(5 + outlen);

        result[0] = 0xf0;
        result[1] = (unsigned char)((len >> 24) & 0xff);
        result[2] = (unsigned char)((len >> 16) & 0xff);
        result[3] = (unsigned char)((len >> 8) & 0xff);
        result[4] = (unsigned char)(len & 0xff);

        lzo_uint newlen = outlen;
        int e;
        switch (algorithm) {
            case 0: // LZO1
                if (constraint == 99) {
                    e = lzo1_99_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                } else {
                    e = lzo1_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                }
                break;
            case 1: // LZO1A
                if (constraint == 99) {
                    e = lzo1a_99_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                } else {
                    e = lzo1a_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                }
                break;
            case 2: // LZO1B
                switch (constraint) {
                    case 1:
                        e = lzo1b_1_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 2:
                        e = lzo1b_2_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 3:
                        e = lzo1b_3_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 4:
                        e = lzo1b_4_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 5:
                        e = lzo1b_5_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 6:
                        e = lzo1b_6_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 7:
                        e = lzo1b_7_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 8:
                        e = lzo1b_8_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 9:
                        e = lzo1b_9_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 99:
                        e = lzo1b_99_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 999:
                        e = lzo1b_999_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    default:
                        e = lzo1b_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0], constraint);
                }
                break;
            case 3: // LZO1C
                switch (constraint) {
                    case 1:
                        e = lzo1c_1_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 2:
                        e = lzo1c_2_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 3:
                        e = lzo1c_3_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 4:
                        e = lzo1c_4_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 5:
                        e = lzo1c_5_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 6:
                        e = lzo1c_6_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 7:
                        e = lzo1c_7_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 8:
                        e = lzo1c_8_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 9:
                        e = lzo1c_9_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 99:
                        e = lzo1c_99_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 999:
                        e = lzo1c_999_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    default:
                        e = lzo1c_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0], constraint);
                }
                break;
            case 4: // LZO1F
                if (constraint == 999) {
                    e = lzo1f_999_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                } else {
                    e = lzo1f_1_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                }
                break;
            case 6: // LZO1Y
                if (constraint == 999) {
                    e = lzo1y_999_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                } else {
                    e = lzo1y_1_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                }
                break;
            case 7: // LZO1Z
                e = lzo1z_999_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                break;
            case 8: // LZO2A
                e = lzo2a_999_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                break;
            default:    // LZO1X
                switch (constraint) {
                    case 11:
                        e = lzo1x_1_11_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 12:
                        e = lzo1x_1_12_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 15:
                        e = lzo1x_1_15_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    case 999:
                        e = lzo1x_999_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                        break;
                    default:
                        e = lzo1x_1_compress(in, len, &result[5], &newlen, (lzo_voidp)&m_WorkMem[0]);
                }
        }
        if (e != LZO_E_OK) {
            ythrow yexception() << "Error while compressing data: " << LZO_E_to_string(e);
        }
        if (newlen > outlen) {
            ythrow yexception() << "Wrong predicted length of compressed data";
        }

        result.resize(5 + newlen);

        return result;
    }

    TString Uncompress(const unsigned char* in, size_t len, unsigned char algorithm, int constraint) const {
        TString result;
        if (!m_InitOk) {
            ythrow yexception() << "Can not init LZO-engine";
        }

        if (len == 0) {
            result = "";
            return result;
        }

        if (in[0] < 0xF0 || in[0] > 0xF1) {
            ythrow yexception() << "Incorrect algorithm in LZO-compressed data";
        }

        lzo_uint outlen = ui32(in[1]) << 24 | ui32(in[2]) << 16 | ui32(in[3]) << 8 | ui32(in[4]);
        if (outlen == 0) {
            result = "";
            return result;
        }

        result.resize(outlen + 1);
        lzo_uint newlen = outlen;
        int e;
        switch (algorithm) {
            case 0: // LZO1
                e = lzo1_decompress(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                break;
            case 1: // LZO1A
                e = lzo1a_decompress(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                break;
            case 2: // LZO1B
                if (constraint == -1) {  // SAFE
                    e = lzo1b_decompress_safe(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                } else {
                    e = lzo1b_decompress(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                }
                break;
            case 3: // LZO1C
                if (constraint == -1) {
                    e = lzo1c_decompress_safe(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                } else {
                    e = lzo1c_decompress(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                }
                break;
            case 4: // LZO1F
                if (constraint == -1) {
                    e = lzo1f_decompress_safe(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                } else {
                    e = lzo1f_decompress(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                }
                break;
            case 6: // LZO1Y
                if (constraint == -1) {
                    e = lzo1y_decompress_safe(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                } else {
                    e = lzo1y_decompress(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                }
                break;
            case 7: // LZO1Z
                if (constraint == -1) {
                    e = lzo1z_decompress_safe(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                } else {
                    e = lzo1z_decompress(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                }
                break;
            case 8: // LZO2A
                if (constraint == -1) {
                    e = lzo2a_decompress_safe(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                } else {
                    e = lzo2a_decompress(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                }
                break;
            default:    // LZO1X
                if (constraint == -1) {
                    e = lzo1x_decompress_safe(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                } else {
                    e = lzo1x_decompress(in + 5, len - 5, (lzo_bytep)result.begin(), &newlen, (lzo_voidp)NULL);
                }
        }
        if (e != LZO_E_OK) {
            ythrow yexception() << "Error while uncompressing data: " << LZO_E_to_string(e);
        }
        if (outlen != newlen) {
            ythrow yexception() << "Wrong length in LZO-compressed message";
        }

        result.erase(newlen);

        return result;
    }
};

/*extern "C" JNIEXPORT jlong JNICALL
Java_ru_yandex_data_compressor_LzoLibrary_createInstance(JNIEnv* env, jclass) {
    try {
        return reinterpret_cast<jlong>(new LzoCompressor());
    } catch (...) {
        NJniWrapper::RethrowAsJavaException(env);
        return 0;
    }
}

extern "C" JNIEXPORT void JNICALL
Java_ru_yandex_data_compressor_LzoLibrary_destroyInstance(JNIEnv*, jclass, jlong instance) {
    delete reinterpret_cast<LzoCompressor*>(instance);
}*/

extern "C" JNIEXPORT jbyteArray JNICALL
Java_ru_yandex_data_compressor_LzoLibrary_compress(
    JNIEnv* env,
    jclass,
    jbyteArray inputData,
    jbyte algorithm,
    jint constraint)
{
    const LzoCompressor compressor;
    try {
        //jboolean copy;
        //unsigned char* Data = (unsigned char*) env->GetByteArrayElements(inputData, &copy);
        NJniWrapper::TByteArrayElements byteArray(env, inputData, JNI_ABORT);
        unsigned char* Data = (unsigned char*) byteArray.GetData();
        if (!Data) {
            ythrow TWithBackTrace<yexception>() << "Failed to GetByteArrayElements";
        }
        TVector<ui8> outputData =
            compressor.Compress(Data, env->GetArrayLength(inputData), (unsigned char) algorithm, constraint);
        size_t size = outputData.size();
        jbyteArray result = env->NewByteArray(size);
        env->SetByteArrayRegion(result, 0, size, (jbyte*)outputData.data());
        //env->ReleaseByteArrayElements(inputData, (jbyte*)Data, 0);
        return result;
    } catch (...) {
        NJniWrapper::RethrowAsJavaException(env);
        return 0;
    }
}

extern "C" JNIEXPORT jbyteArray JNICALL
Java_ru_yandex_data_compressor_LzoLibrary_uncompress(
    JNIEnv* env,
    jclass,
    jbyteArray inputData,
    jbyte algorithm,
    jint constraint)
{
    //const LzoCompressor* compressor = reinterpret_cast<const LzoCompressor*>(instance);
    const LzoCompressor compressor;
    try {
        //jboolean copy;
        //unsigned char* Data = (unsigned char*) env->GetByteArrayElements(inputData, &copy);
        NJniWrapper::TByteArrayElements byteArray(env, inputData, JNI_ABORT);
        unsigned char* Data = (unsigned char*) byteArray.GetData();
        if (!Data) {
            ythrow TWithBackTrace<yexception>() << "Failed to GetByteArrayElements";
        }
        TString outputData =
            compressor.Uncompress(Data, env->GetArrayLength(inputData), (unsigned char) algorithm, constraint);
        size_t size = outputData.size();
        jbyteArray result = env->NewByteArray(size);
        env->SetByteArrayRegion(result, 0, size, (jbyte*)outputData.data());
        //env->ReleaseByteArrayElements(inputData, (jbyte*)Data, 0);
        return result;
    } catch (...) {
        NJniWrapper::RethrowAsJavaException(env);
        return 0;
    }
}
