package ru.yandex.jni.fasttext;

import java.nio.file.Path;
import java.util.AbstractMap;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import ru.yandex.function.GenericAutoCloseable;

public class JniFastText implements GenericAutoCloseable<RuntimeException> {
    // i.e. do not destroy models and store them in static cache
    private static final boolean REUSE_MODELS =
        Boolean.parseBoolean(
            System.getProperty("jni-fasttext.reuse-models", "false"));
    private static final Map<Map.Entry<Path, Path>, Long> MODELS_CACHE;

    static {
        System.loadLibrary("jni-fasttext");
        if (REUSE_MODELS) {
            MODELS_CACHE = new HashMap<>();
        } else {
            MODELS_CACHE = null;
        }
    }

    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final Lock readLock = lock.readLock();
    private final Lock writeLock = lock.writeLock();
    private long handle;

    public JniFastText(
        final Path pathToModel,
        final Path stopWordList)
        throws JniFastTextException
    {
        try {
            if (REUSE_MODELS) {
                synchronized (MODELS_CACHE) {
                    Map.Entry<Path, Path> key =
                        new AbstractMap.SimpleImmutableEntry<>(
                            pathToModel,
                            stopWordList);
                    Long handle = MODELS_CACHE.get(key);
                    if (handle == null) {
                        handle = createInstance(
                            pathToModel.toString(),
                            Objects.toString(stopWordList, null));
                        MODELS_CACHE.put(key, handle);
                    }
                    this.handle = handle.longValue();
                }
            } else {
                handle = createInstance(
                    pathToModel.toString(),
                    Objects.toString(stopWordList, null));
            }
        } catch (RuntimeException e) {
            throw new JniFastTextException("Failed to create instance", e);
        }
    }

    public int getDimension() {
        readLock.lock();
        try {
            long handle = this.handle;
            if (handle == 0L) {
                return 0;
            }
            return getDimension(handle);
        } finally {
            readLock.unlock();
        }
    }

    public float[] createDoc(final String s) throws JniFastTextException {
        readLock.lock();
        try {
            long handle = this.handle;
            if (handle == 0L) {
                throw new JniFastTextException("JniFastText closed");
            }
            return createDoc(handle, s);
        } catch (RuntimeException e) {
            throw new JniFastTextException("Failed to create instance", e);
        } finally {
            readLock.unlock();
        }
    }

    public static float relaxedWmd(
        final float[] f1,
        final float[] f2,
        int cols)
        throws JniFastTextException
    {
        try {
            return doRelaxedWmd(f1, f2, cols);
        } catch (RuntimeException e) {
            throw new JniFastTextException(
                "Failed to calculate relaxed distance",
                e);
        }
    }

    public static float greedWmd(
        final float[] f1,
        final float[] f2,
        int cols)
        throws JniFastTextException
    {
        try {
            return doGreedWmd(f1, f2, cols);
        } catch (RuntimeException e) {
            throw new JniFastTextException(
                "Failed to calculate greed distance",
                e);
        }
    }

    @Override
    public void close() {
        if (!REUSE_MODELS) {
            writeLock.lock();
            try {
                if (handle != 0L) {
                    long handle = this.handle;
                    this.handle = 0L;
                    destroyInstance(handle);
                }
            } finally {
                writeLock.unlock();
            }
        }
    }

    @Override
    @SuppressWarnings("deprecation")
    protected void finalize() {
        close();
    }

    private static native long createInstance(
        String pathToModel,
        String stopWordList);

    private static native void destroyInstance(long instance);

    private static native int getDimension(long instance);

    private static native float[] createDoc(long instance, String text);

    private static native float doRelaxedWmd(float[] f1, float[] f2, int cols);

    private static native float doGreedWmd(float[] f1, float[] f2, int cols);
}

