package ru.yandex.lua.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.Comparator;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;
import core.org.luaj.vm2.Varargs;
import jse.org.luaj.vm2.lib.jse.CoerceJavaToLua;

import ru.yandex.charset.StreamEncoder;
import ru.yandex.compress.GzipOutputStream;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonDouble;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class JsonUtils {
    public static LuaValue jsonToLua(@Nullable JsonObject value) {
        if (value == null) {
            return LuaValue.NIL;
        }
        try {
            switch (value.type()) {
                case NULL:
                    return LuaValue.NIL;
                case BOOLEAN:
                    return CoerceJavaToLua.coerce(value.asBoolean());
                case LONG:
                    return CoerceJavaToLua.coerce(value.asLong());
                case DOUBLE:
                    return CoerceJavaToLua.coerce(value.asDouble());
                case STRING:
                    return CoerceJavaToLua.coerce(value.asString());
                case LIST: {
                    final JsonList list = value.asList();
                    final LuaTable table = new LuaTable(list.size(), 0);
                    int index = 1;
                    for (final JsonObject val : list) {
                        table.set(index++, jsonToLua(val));
                    }
                    return table;
                }
                case MAP: {
                    final JsonMap map = value.asMap();
                    final LuaTable table = new LuaTable(0, map.size());
                    for (final Map.Entry<String, JsonObject> kv : map.entrySet()) {
                        table.set(kv.getKey(), jsonToLua(kv.getValue()));
                    }
                    return table;
                }
            }
        } catch (JsonBadCastException ignored) {
        }
        return LuaValue.NIL;
    }

    public static JsonObject luaToJson(@Nullable LuaValue value, ContainerFactory containerFactory) {
        if (value == null) {
            return JsonNull.INSTANCE;
        }
        switch (value.type()) {
            case LuaValue.TNONE:
            case LuaValue.TNIL:
                return JsonNull.INSTANCE;
            case LuaValue.TBOOLEAN:
                return JsonBoolean.valueOf(value.toboolean());
            case LuaValue.TINT:
                return new JsonLong(value.toint());
            case LuaValue.TNUMBER:
                if (value.islong()) {
                    return new JsonLong(value.tolong());
                } else {
                    return new JsonDouble(value.todouble());
                }
            case LuaValue.TFUNCTION:
            case LuaValue.TLIGHTUSERDATA:
            case LuaValue.TUSERDATA:
            case LuaValue.TTHREAD:
            case LuaValue.TSTRING:
            case LuaValue.TVALUE:
                return new JsonString(value.tojstring());
            case LuaValue.TTABLE: {
                final LuaTable luaTable = value.checktable();
                if (luaTable.length() > 0) {
                    final JsonList list = new JsonList(containerFactory, luaTable.length());
                    for (int i = 1; i <= luaTable.length(); i++) {
                        list.add(luaToJson(luaTable.get(i), containerFactory));
                    }
                    return list;
                } else {
                    final LuaTable luaMap = value.checktable();
                    final JsonMap map = new JsonMap(containerFactory, luaMap.keyCount());

                    LuaValue key = LuaValue.NIL;
                    while (true) {
                        Varargs n = luaMap.next(key);
                        if ((key = n.arg1()).isnil()) {
                            break;
                        }
                        map.put(key.tojstring(), luaToJson(value.get(key), containerFactory));
                    }
                    return map;
                }
            }
        }
        return JsonNull.INSTANCE;
    }

    public static void luaToJson(
            @Nullable final LuaValue value,
            @Nonnull final JsonWriterBase writer)
            throws IOException {
        if (value == null) {
            writer.nullValue();
            return;
        }
        switch (value.type()) {
            case LuaValue.TNONE:
            case LuaValue.TNIL:
                writer.nullValue();
                break;
            case LuaValue.TBOOLEAN:
                writer.value(value.toboolean());
                break;
            case LuaValue.TINT:
                writer.value(value.toint());
                break;
            case LuaValue.TNUMBER:
                if (value.islong()) {
                    writer.value(value.tolong());
                } else {
                    writer.value(value.todouble());
                }
                break;
            case LuaValue.TFUNCTION:
            case LuaValue.TLIGHTUSERDATA:
            case LuaValue.TUSERDATA:
            case LuaValue.TTHREAD:
            case LuaValue.TSTRING:
            case LuaValue.TVALUE:
                writer.value(value.tojstring());
                break;
            case LuaValue.TTABLE: {
                final LuaTable table = value.checktable();
                if (table.length() > 0) {
                    writer.startArray();
                    for (LuaValue key : table.keys()) {
                        luaToJson(value.get(key), writer);
                    }
                    writer.endArray();
                } else {
                    writer.startObject();
                    for (LuaValue key : table.keys()) {
                        luaKey(key, writer);
                        luaToJson(value.get(key), writer);
                    }
                    writer.endObject();
                }
                break;
            }
        }
    }

    private static void luaKey(
            @Nullable final LuaValue key,
            @Nonnull final JsonWriterBase writer)
            throws IOException {
        if (key == null) {
            throw new NullPointerException();
        }
        switch (key.type()) {
            case LuaValue.TNUMBER:
                writer.key(Integer.toString(key.toint()));
                break;
            case LuaValue.TSTRING:
                writer.key(key.tojstring());
                break;
            default:
                throw new RuntimeException("unsupported key type:" + key.typename());
        }
    }

    public static LuaAsJson luaAsJson(@Nullable LuaValue value) {
        return new LuaAsJson(value);
    }

    public static class LuaAsJson implements JsonValue {
        @Nullable
        private final LuaValue value;

        public LuaAsJson(@Nullable LuaValue value) {
            this.value = value;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
                throws IOException {
            luaToJson(value, writer);
        }
    }

    public static byte[] compressData(
            @Nonnull final JsonValue data,
            @Nonnull final JsonType jsonType)
            throws IOException {
        final ByteArrayOutputStream compressed = new ByteArrayOutputStream();

        try (JsonWriterBase writer =
                     jsonType.create(
                             new StreamEncoder(
                                     new GzipOutputStream(compressed),
                                     Charset.defaultCharset()))) {
            writer.value(data);
        }

        return compressed.toByteArray();
    }


    @Nonnull
    public static JsonMap ini2json(@Nonnull IniConfig config, ContainerFactory containerFactory) throws ConfigException {
        final JsonMap result = new JsonMap(containerFactory);

        for (String key : config.keys()) {
            result.put(key, new JsonString(config.getString(key)));
        }

        for (var entry : config.sections().entrySet()) {
            result.put(entry.getKey(), ini2json(entry.getValue(), containerFactory));
        }

        return result;
    }

    public static void merge(@Nonnull JsonMap dst, @Nonnull JsonMap src) throws JsonBadCastException {
        for (var entry : src.entrySet()) {
            final JsonObject newValue = src.get(entry.getKey());
            if (newValue == null || newValue.type() == JsonObject.Type.NULL) {
                continue;
            }

            JsonObject oldValue = dst.get(entry.getKey());

            if (oldValue != null) {
                switch (oldValue.type()) {
                    case NULL:
                    case STRING:
                        oldValue = newValue;
                        break;
                    case BOOLEAN:
                        oldValue = JsonBoolean.valueOf(oldValue.asBoolean() || newValue.asBoolean());
                        break;
                    case LONG:
                        oldValue = JsonLong.valueOf(oldValue.asLong() + newValue.asLong());
                        break;
                    case DOUBLE:
                        oldValue = new JsonDouble(oldValue.asDouble() + newValue.asDouble());
                        break;
                    case LIST:
                        oldValue.asList().addAll(newValue.asList());
                        break;
                    case MAP:
                        merge(oldValue.asMap(), newValue.asMap());
                        break;
                }
            } else {
                oldValue = newValue;
            }
            dst.put(entry.getKey(), oldValue);
        }
    }

    public static JsonMap truncate(@Nonnull JsonMap dst, int maxNestedContainerSize) throws JsonBadCastException {
        if (dst.values().stream().allMatch(v -> v.type() == JsonObject.Type.LONG)) {
            if (dst.size() < maxNestedContainerSize) {
                return dst;
            }
            final JsonMap truncatedMap = new JsonMap(dst.containerFactory(), maxNestedContainerSize);
            dst.entrySet().stream().sorted(JsonMapEntryLongReverseComparator.INSTANCE).limit(maxNestedContainerSize).forEach(
                    e -> truncatedMap.put(e.getKey(), e.getValue())
            );
            return truncatedMap;
        } else {
            for (var entry : dst.entrySet()) {
                if (entry.getValue().type() == JsonObject.Type.MAP) {
                    dst.put(entry.getKey(), truncate(entry.getValue().asMap(), maxNestedContainerSize));
                }
            }
            return dst;
        }
    }

    public static LuaTable toLua(Collection<String> list) {
        final LuaTable dst = new LuaTable();
        int i = 1;
        for (String v : list) {
            dst.set(i++, LuaValue.valueOf(v));
        }
        return dst;
    }

    private static class JsonMapEntryLongReverseComparator implements Comparator<Map.Entry<String, JsonObject>> {
        public static final JsonMapEntryLongReverseComparator INSTANCE = new JsonMapEntryLongReverseComparator();

        @Override
        public int compare(Map.Entry<String, JsonObject> o1, Map.Entry<String, JsonObject> o2) {
            try {
                return -Long.compare(o1.getValue().asLong(), o2.getValue().asLong());
            } catch (JsonBadCastException e) {
                throw new RuntimeException(e);
            }
        }
    }
}
