package ru.yandex.parser.mail.errors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.errorprone.annotations.NoAllocation;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.io.StringBuilderWriter;

public class ErrorInfo extends AbstractStringBuilderable {
    public enum Component {
        CONNECT,
        MAILFROM,
        RCPT_TO,
        RECEIVED,
        SHINGLER,
        HNSW,
        INPUT
    }

    public enum Scope {
        // Connect scopes
        IP(Component.CONNECT),

        // MailFrom scopes
        SIZE(Component.MAILFROM),

        // RcptTo scopes
        SUID(Component.RCPT_TO),
        UID(Component.RCPT_TO),

        // Received scopes
        RECEIVED(Component.RECEIVED), // when Received: has syntax error
        FROM(Component.RECEIVED),
        BY(Component.RECEIVED),
        VIA(Component.RECEIVED),
        WITH(Component.RECEIVED),
        ID(Component.RECEIVED),
        FOR(Component.RECEIVED),
        DATE(Component.RECEIVED),

        // Shingler scopes
        SHINGLER(Component.SHINGLER),
        SHINGLER_RESPONSE(Component.SHINGLER),

        // Input errors
        MALFORMED_INPUT(Component.INPUT),
        NO_DOCS(Component.INPUT),
        NO_TEXT_PART(Component.INPUT),
        UNPROCESSABLE_INPUT(Component.INPUT),

        // Hnsw errors
        ZERO_NEIGHBOURS(Component.HNSW);

        @Nonnull
        private final Component component;

        Scope(@Nonnull final Component component) {
            this.component = component;
        }

        @NoAllocation
        @Nonnull
        public Component component() {
            return component;
        }
    }

    public enum Type {
        SYNTAX_ERROR,
        NON_STANDARD,
        FIELD_MISSING,
        EXTERNAL_COMPONENT_ERROR,
        UNEXPECTED_ENTRY,
        INTERNAL_ERROR,
        FRAUD
    }

    private static final String PREFIX = "ErrorInfo(";
    private static final int PREFIX_LENGTH = PREFIX.length();
    private static final int EXPECTED_CAUSE_LENGTH = 64;

    @Nonnull
    private final Scope scope;
    @Nonnull
    private final Type type;
    @Nonnull
    private final String message;
    @Nullable
    private final Throwable cause;

    public ErrorInfo(
        @Nonnull final Scope scope,
        @Nonnull final Type type,
        @Nonnull final String message)
    {
        this(scope, type, message, null);
    }

    public ErrorInfo(
        @Nonnull final Scope scope,
        @Nonnull final Type type,
        @Nonnull final String message,
        @Nullable final Throwable cause)
    {
        this.scope = scope;
        this.type = type;
        this.message = message;
        this.cause = cause;
    }

    @NoAllocation
    @Nonnull
    public Scope scope() {
        return scope;
    }

    @NoAllocation
    @Nonnull
    public Type type() {
        return type;
    }

    @NoAllocation
    @Nonnull
    public String message() {
        return message;
    }

    @NoAllocation
    @Nullable
    public Throwable cause() {
        return cause;
    }

    @NoAllocation
    @Override
    public int expectedStringLength() {
        int len = 4
            + PREFIX_LENGTH
            + scope.component().toString().length()
            + scope.toString().length()
            + type.toString().length()
            + message.length();
        if (cause != null) {
            len += 1 + EXPECTED_CAUSE_LENGTH;
        }
        return len;
    }

    @Override
    public void toStringBuilder(@Nonnull final StringBuilder sb) {
        sb.append(PREFIX);
        sb.append(scope.component().toString());
        sb.append(',');
        sb.append(scope.toString());
        sb.append(',');
        sb.append(type.toString());
        sb.append(',');
        sb.append(message);
        if (cause != null) {
            sb.append(',');
            sb.append(cause.getMessage());
        }
        sb.append(')');
    }

    public void describeTo(final StringBuilder sb) {
        sb.append("Error in component: ");
        sb.append(scope.component().toString());
        sb.append(", scope: ");
        sb.append(scope.toString());
        sb.append(", type: ");
        sb.append(type.toString());
        sb.append(", reason: ");
        sb.append(message);
        if (cause != null) {
            sb.append(", caused by: ");
            cause.printStackTrace(new StringBuilderWriter(sb));
        }
    }
}

