package ru.yandex.parser.mail.received;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.detect.locale.LocaleDetector;
import ru.yandex.parser.mail.envelope.SmtpEnvelopeHolder;
import ru.yandex.parser.mail.errors.ErrorInfo;
import ru.yandex.url.processor.CommonZoneExtractor;

public class ReceivedChainParser implements Consumer<ErrorInfo> {
    private static final Pattern YANDEX_HOST_PATTERN =
        Pattern.compile("(?:.*[.]yandex[.](?:net|ru)|^cso-yandex[.]ru)$");
    private static final Pattern PROTOCOL_PATTERN =
        Pattern.compile("(?:E?SMTPS?A?|HTTPS?)");
    private static final Pattern ID_PATTERN = Pattern.compile("\\w+-\\w+");
    private static final Pattern MAIL_FRONT =
        Pattern.compile("\\s*[(](?<mailfront>[^/)]+)/Yandex[)]\\s*");

    enum State {
        INITIAL {
            @Override
            public void process(
                final ReceivedChainParser chainParser,
                final ReceivedInfo receivedInfo)
            {
                String by = extractDomain(receivedInfo.by());
                if (by == null || !YANDEX_HOST_PATTERN.matcher(by).matches()) {
                    if (!chainParser.skipExternalBys) {
                        chainParser.byPrecoditionFailed(by);
                    }
                } else {
                    chainParser.state = State.PROCESSING;
                    chainParser.extractYandexSmtpId(receivedInfo);
                    processTrustedEntry(chainParser, receivedInfo);
                    chainParser.skipExternalBys = false;
                    if (by.endsWith(".cmail.yandex.net")) {
                        chainParser.cmail = true;
                    }
                }
            }
        },
        POP3 {
            @Override
            public void process(
                final ReceivedChainParser chainParser,
                final ReceivedInfo receivedInfo)
            {
                ExtendedDomain fromDomain = receivedInfo.from();
                chainParser.prevFrom = extractDomain(fromDomain);
                if (chainParser.prevFrom != null) {
                    chainParser.state = State.YANDEX_CHAIN_END;
                }
            }
        },
        PROCESSING {
            @Override
            public void process(
                final ReceivedChainParser chainParser,
                final ReceivedInfo receivedInfo)
            {
                String by = extractDomain(receivedInfo.by());
                if (by == null
                    || !YANDEX_HOST_PATTERN.matcher(by).matches())
                {
                    chainParser.byPrecoditionFailed(by);
                } else {
                    if ("POP3".equals(receivedInfo.protocol())) {
                        ExtendedDomain fromDomain = receivedInfo.from();
                        String from = extractDomain(fromDomain);
                        if (from == null) {
                            chainParser.state = State.YANDEX_CHAIN_END;
                        } else {
                            chainParser.state = State.POP3;
                            chainParser.prevFrom = null;
                            chainParser.pop3 = true;
                        }
                    } else if (by.equals(chainParser.prevHelo)) {
                        chainParser.extractYandexSmtpId(receivedInfo);
                        processTrustedEntry(chainParser, receivedInfo);
                    } else {
                        chainParser.byPrecoditionFailed(by);
                    }
                    if (by.endsWith(".cmail.yandex.net")) {
                        chainParser.cmail = true;
                    }
                }
            }
        },
        YANDEX_CHAIN_END {
            @Override
            public void process(
                final ReceivedChainParser chainParser,
                final ReceivedInfo receivedInfo)
            {
                return;
            }
        },
        FAILED {
            @Override
            public void process(
                final ReceivedChainParser chainParser,
                final ReceivedInfo receivedInfo)
            {
                return;
            }
        };

        public abstract void process(
            final ReceivedChainParser chainParser,
            final ReceivedInfo receivedInfo);
    }

    private final ReceivedParser parser = new ReceivedParser();
    private final List<String> allYandexSmtpIds = new ArrayList<>();
    private final List<InetAddress> fromAddrs = new ArrayList<>();
    private final Set<String> recipients = new LinkedHashSet<>();
    private final Predicate<? super InetAddress> yandexNets;
    private boolean skipExternalBys;
    private State state = State.INITIAL;
    private ErrorInfo errorInfo = null;
    private String yandexSmtpId = null;
    private String prevHelo = null;
    private String prevFrom = null;
    private InetAddress prevFromAddr = null;
    private String byComment = null;
    private boolean pop3 = false;
    private boolean cmail = false;

    public ReceivedChainParser(
        final Predicate<? super InetAddress> yandexNets)
    {
        this(yandexNets, false);
    }

    public ReceivedChainParser(
        final Predicate<? super InetAddress> yandexNets,
        final boolean skipExternalBys)
    {
        this.yandexNets = yandexNets;
        this.skipExternalBys = skipExternalBys;
    }

    @Override
    public void accept(final ErrorInfo errorInfo) {
        if (state.compareTo(State.FAILED) < 0) {
            state = State.FAILED;
            this.errorInfo = errorInfo;
        }
    }

    public Predicate<? super InetAddress> yandexNets() {
        return yandexNets;
    }

    public ErrorInfo errorInfo() {
        return errorInfo;
    }

    public String yandexSmtpId() {
        return yandexSmtpId;
    }

    public List<String> allYandexSmtpIds() {
        return allYandexSmtpIds;
    }

    public String sourceDomain() {
        return sourceDomain(null);
    }

    public String sourceDomain(final SmtpEnvelopeHolder envelope) {
        return sourceDomain("yandex.net", envelope);
    }

    public String sourceDomain(
        final String internalSenderDomain,
        final SmtpEnvelopeHolder envelope)
    {
        String domain;
        if (pop3) {
            if (prevFrom == null) {
                domain = "pop.pop";
            } else {
                domain = CommonZoneExtractor.INSTANCE.apply(prevFrom);
            }
        } else {
            String from;
            InetAddress addr;
            if (envelope != null
                && prevFrom == null
                && prevFromAddr == null
                && envelope.envelope().hasConnectInfo())
            {
                from = normalizeDomain(
                    envelope.envelope().getConnectInfo().getRemoteHost());
                addr = envelope.ip();
            } else {
                from = prevFrom;
                addr = prevFromAddr;
            }

            if ((errorInfo == null || "localhost".equals(from))
                && addr != null
                && yandexNets.test(addr))
            {
                domain = internalSenderDomain;
            } else if (from == null) {
                if (addr == null) {
                    domain = null;
                } else {
                    domain = normalizeAddr(addr);
                }
            } else {
                domain = CommonZoneExtractor.INSTANCE.apply(from);
            }
        }
        return domain;
    }

    private String normalizeAddr(final InetAddress addr) {
        String result = addr.getHostAddress();
        int dot = result.lastIndexOf('.');
        if (dot != -1) {
            result = result.substring(0, dot + 1) + '0';
        }
        return result;
    }

    public String fullSenderHost() {
        return prevFrom;
    }

    public List<InetAddress> sourceIps() {
        return fromAddrs;
    }

    public Set<String> recipients() {
        return recipients;
    }

    public String byComment() {
        return byComment;
    }

    public String mailFront() {
        if (byComment != null) {
            Matcher matcher = MAIL_FRONT.matcher(byComment);
            if (matcher.matches()) {
                return matcher.group("mailfront");
            }
        }
        return null;
    }

    public boolean cmail() {
        return cmail;
    }

    private static String normalizeDomain(final String domain) {
        if (domain == null
            || domain.isEmpty()
            || "unknown".equalsIgnoreCase(domain))
        {
            return null;
        } else {
            return LocaleDetector.INSTANCE.toLowerCase(domain);
        }
    }

    private static String extractHeloDomain(
        final ExtendedDomain extendedDomain)
    {
        if (extendedDomain == null) {
            return null;
        } else {
            String domain = extendedDomain.heloDomain();
            if (domain == null || domain.isEmpty()) {
                return null;
            } else {
                return LocaleDetector.INSTANCE.toLowerCase(domain);
            }
        }
    }

    private static String extractDomain(final ExtendedDomain extendedDomain) {
        if (extendedDomain == null) {
            return null;
        } else {
            return normalizeDomain(extendedDomain.domain());
        }
    }

    private void byPrecoditionFailed(final String by) {
        byPrecoditionFailed(by, prevHelo);
    }

    private void byPrecoditionFailed(final String by, final String expected) {
        accept(
            new ErrorInfo(
                ErrorInfo.Scope.BY,
                ErrorInfo.Type.FIELD_MISSING,
                "In state " + state +
                " expected trusted entry for <" + expected
                + ">, but BY was <" + by + '>'));
    }

    private void extractYandexSmtpId(final ReceivedInfo receivedInfo) {
        String protocol = receivedInfo.protocol();
        if (protocol != null && PROTOCOL_PATTERN.matcher(protocol).matches()) {
            String id = receivedInfo.id();
            if (id != null && ID_PATTERN.matcher(id).matches()) {
                yandexSmtpId = id;
                allYandexSmtpIds.add(id);
            }
        }
    }

    private void validateYandexIp(final ExtendedDomain from) {
        prevFromAddr = from.tcpInfoAddress();
        if (prevFromAddr == null) {
            prevHelo = null;
            prevFrom = null;
            accept(
                new ErrorInfo(
                    ErrorInfo.Scope.FROM,
                    ErrorInfo.Type.FIELD_MISSING,
                    "No TCPInfo for " + from));
        } else {
            fromAddrs.add(prevFromAddr);
            if (!yandexNets.test(prevFromAddr)) {
                prevHelo = normalizeAddr(from.address());
                prevFrom = prevHelo;
                accept(
                    new ErrorInfo(
                        ErrorInfo.Scope.FROM,
                        ErrorInfo.Type.FRAUD,
                        "Yandex MX impersonation detected, TCPInfo address "
                        + "doesn't belong to Yandex: " + from));
            }
        }
    }

    @SuppressWarnings("ReferenceEquality")
    public void process(final String received) {
        if (state.compareTo(State.PROCESSING) > 0) {
            return;
        }
        ReceivedInfo receivedInfo = parser.parse(this, received);
        // If receivedInfo is null, parser will report error and state will be
        // changed to FAILED
        if (state.compareTo(State.PROCESSING) > 0) {
            return;
        }
        String prevId = yandexSmtpId;
        state.process(this, receivedInfo);
        if (yandexSmtpId != null && yandexSmtpId != prevId) {
            byComment = receivedInfo.byComment();
        }
    }

    private static void processTrustedEntry(
        final ReceivedChainParser chainParser,
        final ReceivedInfo receivedInfo)
    {
        ExtendedDomain fromDomain = receivedInfo.from();
        String helo = extractHeloDomain(fromDomain);
        chainParser.prevHelo = helo;
        chainParser.prevFrom = extractDomain(fromDomain);
        String recipient = receivedInfo.recipient();
        if (recipient != null && !recipient.isEmpty()) {
            chainParser.recipients.add(recipient);
        }
        if (helo == null) {
            chainParser.state = State.YANDEX_CHAIN_END;
        } else if (YANDEX_HOST_PATTERN.matcher(helo).matches()) {
            chainParser.validateYandexIp(fromDomain);
        } else {
            chainParser.state = State.YANDEX_CHAIN_END;
            chainParser.prevFromAddr = fromDomain.tcpInfoAddress();
            if (chainParser.prevFromAddr != null) {
                chainParser.fromAddrs.add(chainParser.prevFromAddr);
            }
        }
    }
}

