package ru.yandex.parser.mail.received;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.function.Consumer;
import java.util.regex.Pattern;

import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.parser.mail.errors.ErrorInfo;
import ru.yandex.util.string.StringUtils;

public class ReceivedInfoBuilder {
    private static final Pattern TIMESTAMP_NORMALIZER =
        Pattern.compile("(?:[\t][ \t]*|[ ][ \t]+)");
    private static final DateTimeFormatter[] TIME_FORMATS = {
        DateTimeFormat.forPattern("d' 'MMM' 'yyyy' 'HH:mm:ss' 'Z"),
        DateTimeFormat.forPattern("d' 'MMM' 'yyyy' 'HH:mm' 'Z"),
        DateTimeFormat.forPattern("d' 'MMM' 'yyyy' 'HH:mm:ssZ"),
        DateTimeFormat.forPattern("d' 'MMM' 'yyyy' 'HH:mmZ")
    };

    private final Consumer<? super ErrorInfo> errorsConsumer;
    private final char[] data;
    private ExtendedDomain from = null;
    private ExtendedDomain by = null;
    private String byComment = null;
    private String protocol = null;
    private String id = null;
    private String recipient = null;
    private long timestamp = 0L;

    private String extendedDomainDomain = null;
    private String extendedDomainAddress = null;
    private String tcpInfoDomain = null;
    private String tcpInfoAddress = null;
    private int extendedDomainDomainStart = -1;
    private int extendedDomainAddressStart = -1;
    private int tcpInfoDomainStart = -1;
    private int tcpInfoAddressStart = -1;
    private int protocolStart = -1;
    private int idStart = -1;
    private int recipientStart = -1;
    private int timestampStart = -1;
    private int byCommentStart = -1;

    public ReceivedInfoBuilder(
        final Consumer<? super ErrorInfo> errorsConsumer,
        final char[] data)
    {
        this.errorsConsumer = errorsConsumer;
        this.data = data;
    }

    public void extendedDomainDomainStart(final int pos) {
        extendedDomainDomainStart = pos;
    }

    public void extendedDomainDomainEnd(final int pos) {
        extendedDomainDomain = new String(
            data,
            extendedDomainDomainStart,
            pos - extendedDomainDomainStart);
    }

    public void extendedDomainAddressStart(final int pos) {
        extendedDomainAddressStart = pos;
    }

    public void extendedDomainAddressEnd(final int pos) {
        extendedDomainAddress = new String(
            data,
            extendedDomainAddressStart,
            pos - extendedDomainAddressStart);
    }

    public void tcpInfoDomainStart(final int pos) {
        tcpInfoDomainStart = pos;
    }

    public void tcpInfoDomainEnd(final int pos) {
        tcpInfoDomain = new String(
            data,
            tcpInfoDomainStart,
            pos - tcpInfoDomainStart);
    }

    public void tcpInfoAddressStart(final int pos) {
        tcpInfoAddressStart = pos;
    }

    public void tcpInfoAddressEnd(final int pos) {
        tcpInfoAddress = new String(
            data,
            tcpInfoAddressStart,
            pos - tcpInfoAddressStart);
    }

    public void commitFrom() {
        from = parseExtendedDomain(ErrorInfo.Scope.FROM);
    }

    public void startBy() {
        extendedDomainDomain = null;
        extendedDomainAddress = null;
        tcpInfoDomain = null;
        tcpInfoAddress = null;
    }

    public void commitBy() {
        by = parseExtendedDomain(ErrorInfo.Scope.BY);
    }

    public void byCommentStart(final int pos) {
        if (byCommentStart == -1) {
            byCommentStart = pos;
        }
    }

    public void byCommentEnd(final int pos) {
        if (byCommentStart != -1 && byComment == null) {
            byComment = new String(data, byCommentStart, pos - byCommentStart);
        }
    }

    public void protocolStart(final int pos) {
        protocolStart = pos;
    }

    public void protocolEnd(final int pos) {
        protocol = new String(data, protocolStart, pos - protocolStart);
    }

    public void idStart(final int pos) {
        idStart = pos;
    }

    public void idEnd(final int pos) {
        id = new String(data, idStart, pos - idStart);
    }

    public void recipientStart(final int pos) {
        recipientStart = pos;
    }

    public void recipientEnd(int pos) {
        if (data[recipientStart] == '<' && data[pos - 1] == '>') {
            ++recipientStart;
            --pos;
        }
        recipient = new String(data, recipientStart, pos - recipientStart);
    }

    public void timestampStart(final int pos) {
        timestampStart = pos;
    }

    public void timestampEnd(final int pos) {
        String str =
            new String(data, timestampStart, pos - timestampStart);
        str = TIMESTAMP_NORMALIZER.matcher(str).replaceAll(" ").trim();
        for (DateTimeFormatter formatter: TIME_FORMATS) {
            try {
                timestamp = formatter.parseMillis(str);
                return;
            } catch (Throwable tt) {
            }
        }
        // Just ignore malformed timestamp
    }

    private ExtendedDomain parseExtendedDomain(final ErrorInfo.Scope scope) {
        InetAddress address =
            parseInetAddress("TcpInfo address", tcpInfoAddress, scope);
        TcpInfo tcpInfo;
        if (address == null) {
            tcpInfo = null;
        } else {
            tcpInfo = new TcpInfo(tcpInfoDomain, address);
        }
        return new ExtendedDomain(
            extendedDomainDomain,
            parseInetAddress("address", extendedDomainAddress, scope),
            tcpInfo);
    }

    private InetAddress parseInetAddress(
        final String name,
        final String addressString,
        final ErrorInfo.Scope scope)
    {
        InetAddress address = null;
        if (addressString != null && addressString.length() > 2) {
            String str = addressString;
            if (str.charAt(0) == '['
                && str.charAt(addressString.length() - 1) == ']')
            {
                str = str.substring(1, addressString.length() - 1);
            }
            if (StringUtils.startsWithIgnoreCase(str, "ipv6:")) {
                str = str.substring(5);
            }
            try {
                address = InetAddress.getByName(str);
            } catch (UnknownHostException e) {
                errorsConsumer.accept(
                    new ErrorInfo(
                        scope,
                        ErrorInfo.Type.SYNTAX_ERROR,
                        "Failed to parse " + name + " from <"
                        + addressString + '>',
                        e));
            }
        }
        return address;
    }

    public ReceivedInfo build() {
        return new ReceivedInfo(
            from,
            by,
            byComment,
            protocol,
            id,
            recipient,
            timestamp);
    }
}

