package ru.yandex.expr_parser;

import java.io.IOException;
import java.io.StringReader;
import java.text.ParseException;
import java.util.ArrayDeque;
import java.util.Deque;

import ru.yandex.expr_parser.generated.Tokenizer;

public class QueryParser {
    private final Deque<Token> readyTokens = new ArrayDeque<>();
    private final Deque<Token> operatorStack = new ArrayDeque<>();
    private final Deque<Integer> pars = new ArrayDeque<>();
    private final Tokenizer tokenizer;

    public QueryParser(final String query) {
        this(new Tokenizer(new StringReader(query)));
    }

    private QueryParser(final Tokenizer tokenizer) {
        this.tokenizer = tokenizer;
    }

    private ParseException exc(final Token token) {
        return new ParseException(
                "Unexpected token " + token + " at pos: " + tokenizer.pos(),
                tokenizer.pos());
    }


    private QueryAtom makeTree() throws ParseException {
        final Token topToken = readyTokens.pollLast();
        assert topToken != null;
        switch (topToken.type()) {
            case FIELD:
            case INTEGER:
            case DOUBLE:
            case STRING:
                return new FieldQuery(topToken);
            case AND:
                return new AndQuery(makeTree(), makeTree());
            case OR:
                return new OrQuery(makeTree(), makeTree());
            case RE:
                return new ReQuery(makeTree());
            case NOT:
                return new NotQuery(makeTree());
            case LPAR:
            case RPAR:
            case WHITESPACE:
            case EOF:
                throw new ParseException("Wrong tree: " + readyTokens.toString(), 0);
        }
        return null;
    }

    public QueryAtom parse()
            throws IOException, ParseException
    {
        tokenizer.state(Tokenizer.YYINITIAL);
        boolean run = true;
        while (run) {
            final Token token = tokenizer.nextToken().createToken(tokenizer);
            switch (token.type()) {
                case FIELD:
                case INTEGER:
                case DOUBLE:
                case STRING:
                    readyTokens.add(token);
                    break;
                case AND:
                case OR:{
                    while(true) {
                        final Token topToken = operatorStack.peekLast();
                        if(topToken == null || !(topToken.type() == Token.Type.NOT || topToken.type() == Token.Type.RE ||
                                (topToken.type() == Token.Type.AND || topToken.type() == Token.Type.OR) && topToken.type().precedence() >= token.type().precedence()
                        )
                        ) {
                            break;
                        }
                        readyTokens.add(operatorStack.pollLast());
                    }
                    operatorStack.add(token);
                    break;
                }
                case RE:
                case NOT:
                    operatorStack.add(token);
                    break;
                case LPAR:
                    operatorStack.add(token);
                    pars.push(tokenizer.pos());
                    break;
                case RPAR:
                    if (pars.isEmpty()) {
                        throw exc(token);
                    } else {
                        pars.pop();
                    }
                    while(true) {
                        final Token topToken = operatorStack.peekLast();
                        if(topToken == null || topToken.type() == Token.Type.LPAR) {
                            break;
                        }
                        readyTokens.add(operatorStack.pollLast());
                    }
                    if(!operatorStack.isEmpty() && operatorStack.peekLast().type() == Token.Type.LPAR)
                        operatorStack.pollLast();
                    break;
                case WHITESPACE:
                    break;
                case EOF:
                    if (pars.isEmpty()) {
                        tokenizer.unstate();
                        run = false;
                        break;
                    } else {
                        int pos = pars.peek();
                        throw new ParseException(
                                "Unclosed parenthesis at pos: " + pos,
                                pos);
                    }
                default:
                    throw exc(token);
            }
        }
        while(!operatorStack.isEmpty()) {
            readyTokens.add(operatorStack.pollLast());
        }

        return makeTree();
    }
    public static void main(final String... args) throws Exception {
        new QueryParser(args[0])
                .parse()
                .accept(new QueryPrintingVisitor(System.out));
    }
}

