package ru.yandex.rules_parser;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;

import ru.yandex.parser.string.EnumParser;

class RawRule {
    enum Type {
        R_ANTI,
        R_ALG,
        R_PRESENT,
        R_ABSENT,
        R_LV,
        R_RE,
        R_BEXPR,
        R_ARITHMETIC,
        ;

        static Type parse(String s) throws IllegalArgumentException, NullPointerException {
            return new EnumParser<>(Type.class).apply(s);
        }
    }

    @Override
    public String toString() {
        return name + "," + type + "," + weight + "," + fields + "," + body;
    }

    RawRule(final HashMap<Parser.KeyWord, String> parts) throws RuleException {
        {
            List<String> ruleParts = new LinkedList<>(Arrays.asList(parts.get(Parser.KeyWord.RULE).split("\\s")));
            ruleParts.removeIf(String::isEmpty);

            name = ruleParts.get(0);

            int offset = 1;

            try {
                type = Type.parse(ruleParts.get(offset));
                offset++;
            } catch (Exception ignored) {
            }

            if (type == Type.R_ANTI) {
                body = String.join(" ", ruleParts.subList(offset, ruleParts.size()));
            } else {
                try {
                    weight = Double.parseDouble(ruleParts.get(offset));
                    offset++;
                } catch (NumberFormatException ignored) {
                    fields.add(ruleParts.get(offset++));
                    weight = Double.parseDouble(ruleParts.get(offset));
                }
            }
        }

        if (type == null) {
            if (parts.containsKey(Parser.KeyWord.RE))
                type = Type.R_RE;
            else if (parts.containsKey(Parser.KeyWord.BEXPR))
                type = Type.R_BEXPR;
            else if (parts.containsKey(Parser.KeyWord.ARITHMETIC))
                type = Type.R_ARITHMETIC;
        }

        if (type == null) {
            throw new RuleException("cannot detect rule type for " + parts);
        }

        {
            final String field = parts.get(Parser.KeyWord.FIELD);
            if(field != null)
                fields.addAll(Arrays.asList(field.split("\\s")));
        }
        if (fields.isEmpty() && type == Type.R_PRESENT) {
            throw new RuleException("cannot detect fields for " + parts);
        }

        switch (type) {
            case R_RE:
                body = parts.get(Parser.KeyWord.RE);
                break;
            case R_BEXPR:
                body = parts.get(Parser.KeyWord.BEXPR);
                break;
            case R_ARITHMETIC:
                body = parts.get(Parser.KeyWord.ARITHMETIC);
                break;
            case R_PRESENT:
            case R_ABSENT:
            case R_ANTI:
            case R_ALG:
                break;
        }
    }

    private String name;
    private double weight;
    private List<String> fields = new LinkedList<>();
    private String body;
    private Type type;
}
