package ru.yandex.parser.mail.senders;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.net.uri.fast.FastUri;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.url.processor.StringUrlsExtractor;
import ru.yandex.url.processor.UrlInfo;
import ru.yandex.util.string.StringUtils;

public class SendersContext {
    private final StringUrlsExtractor urlsExtractor =
        new StringUrlsExtractor();
    private final String sender;
    private final String headers;

    public SendersContext(final String headers) {
        this.headers = prepareHeaders(headers);
        sender = extractSender();
    }

    public SendersContext(final String sender, final String headers) {
        this.sender = sender;
        this.headers = prepareHeaders(headers);
    }

    private static String prepareHeaders(final String headers) {
        if (StringUtils.startsWith(headers, '\n')) {
            if (StringUtils.endsWith(headers, '\n')) {
                return headers;
            } else {
                return StringUtils.concat(headers, '\n');
            }
        } else if (StringUtils.endsWith(headers, '\n')) {
            return StringUtils.concat('\n', headers);
        } else {
            return StringUtils.concat('\n', headers, '\n');
        }
    }

    // Take sender from headers because on so_in mailfrom may contain invalid
    // data for emails collected via imap
    // So, always trust what is written in headers
    private String extractSender() {
        String sender = extractFirstEmail("\nfrom: ");
        if (sender == null) {
            sender = extractFirstEmail("\nreply-to: ");
        }
        return sender;
    }

    public String getSender() {
        return sender;
    }

    public String getHdrSender() {
        return extractFirstEmail("\nsender: ");
    }

    // Needle must be in the form '\n' + <lowercase-header-name> + ": "
    @Nullable
    public String getHeader(final String needle) {
        int idx = headers.indexOf(needle);
        if (idx == -1) {
            return null;
        } else {
            int start = idx + needle.length();
            int end = headers.indexOf('\n', start);
            return headers.substring(start, end).trim();
        }
    }

    @Nullable
    public String extractFirstEmail(final String headerName) {
        String header = getHeader(headerName);
        if (header != null) {
            List<UrlInfo> urls = urlsExtractor.extractUrls(header);
            int size = urls.size();
            for (int i = 0; i < size; ++i) {
                FastUri uri = urls.get(i).url();
                if ("mailto".equals(uri.scheme())) {
                    return MailAliases.INSTANCE.normalizeEmail(
                        uri.schemeSpecificPart());
                }
            }
        }
        return null;
    }

    public SendersInfo extractSenders() {
        Set<String> distinctSenders = new HashSet<>();
        SenderType[] types = SenderType.values();
        SendersInfo senders = new SendersInfo(types.length);
        for (SenderType type: types) {
            String sender = type.extract(this);
            if (sender != null && distinctSenders.add(sender)) {
                senders.add(new SenderInfo(type, sender));
            }
        }
        return senders;
    }

    @Nullable
    public static SenderType detectSenderType(final String documentUrl) {
        SenderType[] types = SenderType.values();
        for (int i = types.length; i-- > 0;) {
            SenderType type = types[i];
            if (documentUrl.startsWith(type.sendersPrefix())) {
                return type;
            }
        }
        return null;
    }

    @Nullable
    public static SenderType detectSenderDomainType(final String documentUrl) {
        SenderType[] types = SenderType.values();
        for (int i = types.length; i-- > 0;) {
            SenderType type = types[i];
            if (documentUrl.startsWith(type.sendersDomainPrefix())) {
                return type;
            }
        }
        return null;
    }

    @Nullable
    public static SenderType detectPfilterType(final String documentUrl) {
        SenderType[] types = SenderType.values();
        for (int i = types.length; i-- > 0;) {
            SenderType type = types[i];
            if (documentUrl.startsWith(type.pfiltersPrefix())) {
                return type;
            }
        }
        return null;
    }

    @Nullable
    public static SenderType detectTabPfType(final String documentUrl) {
        SenderType[] types = SenderType.values();
        for (int i = types.length; i-- > 0;) {
            SenderType type = types[i];
            if (documentUrl.startsWith(type.tabPfPrefix())) {
                return type;
            }
        }
        return null;
    }
}

