package ru.yandex.client.so.shingler;

import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.errorprone.annotations.NoAllocation;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.function.StringBuilderable;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;

public class ComplShingleInfo extends EnumMap<ComplScheme, Map<String, List<Object>>>
    implements GeneralShingleInfo<ComplScheme>
{
    private static final long serialVersionUID = 0L;
    @Nullable
    private DailyStats todayStats = null;
    @Nullable
    private HistoryStats historyStats = null;
    @Nullable
    private UserWeights userWeights = null;

    public ComplShingleInfo() {
        super(ComplScheme.class);
    }

    public ComplShingleInfo(@Nonnull final Map<ComplScheme, List<JsonMap>> jsonInfo)
        throws ShingleException
    {
        super(ComplScheme.class);
        loadInfo(jsonInfo);
    }

    public ComplShingleInfo(@Nonnull final ComplScheme scheme, @Nonnull List<JsonMap> jsonCountersList)
        throws ShingleException
    {
        super(ComplScheme.class);
        loadSchemeCounters(scheme, jsonCountersList);
    }

    public ComplShingleInfo(@Nonnull final Set<ComplScheme> schemes, @Nonnull final Map<String, Object> counters)
        throws ShingleException
    {
        super(ComplScheme.class);
        for (final ComplScheme scheme : schemes) {
            loadSchemeCounters(scheme, counters);
        }
    }

    public ComplShingleInfo(@Nonnull final ComplScheme scheme, @Nonnull JsonList jsonCountersList)
        throws ShingleException, JsonBadCastException
    {
        super(ComplScheme.class);
        loadSchemeCounters(scheme, jsonCountersList);
    }

    public ComplShingleInfo(@Nonnull final ComplScheme scheme, @Nonnull JsonMap jsonCounters)
        throws ShingleException, JsonBadCastException
    {
        super(ComplScheme.class);
        loadSchemeCounters(scheme, jsonCounters);
    }

    public ComplShingleInfo(@Nonnull final ComplScheme scheme, @Nonnull final Map<String, Object> counters)
            throws ShingleException
    {
        super(ComplScheme.class);
        loadSchemeCounters(scheme, counters);
    }

    public ComplShingleInfo(@Nonnull final ComplScheme scheme, @Nonnull final String counter, final Object value)
        throws ShingleException
    {
        super(ComplScheme.class);
        setSchemeCounter(scheme, counter, value);
    }

    public ComplShingleInfo(
        @Nonnull final ComplScheme scheme,
        @Nonnull final String counter,
        final long shingle)
        throws ShingleException
    {
        super(ComplScheme.class);
        setSchemeCounter(scheme, counter, shingle);
    }

    public ComplShingleInfo(final GeneralShingleInfo<ComplScheme> shingleInfo) throws ShingleException {
        super(ComplScheme.class);
        addInfo(shingleInfo);
    }

    @Override
    public ComplShingleInfo copy() {
        try {
            return new ComplShingleInfo(this);
        } catch (Exception e) {
            return null;
        }
    }

    @Override
    public boolean containsKey(ComplScheme scheme) {
        return super.containsKey(scheme);
    }

    @Override
    public Map<String, List<Object>> get(ComplScheme scheme) {
        return super.get(scheme);
    }

    @Override
    public Map<String, List<Object>> remove(ComplScheme scheme) {
        return super.remove(scheme);
    }

    @Override
    public Map<String, List<Object>> computeIfAbsent(
        ComplScheme scheme,
        Function<ComplScheme, Map<String, List<Object>>> mappingFunction)
    {
        return super.computeIfAbsent(scheme, mappingFunction);
    }

    @Nonnull
    @Override
    public Set<ComplScheme> keySet() {
        return super.keySet();
    }

    @Nonnull
    @Override
    public Set<Map.Entry<ComplScheme, Map<String, List<Object>>>> entrySet() {
        return super.entrySet();
    }

    @Nonnull
    @Override
    public String name() {
        return "Compl";
    }

    @Nullable
    public DailyStats getTodayStats() {
        if (todayStats == null && get(ComplScheme.TODAY_ABUSES) != null) {
            todayStats = new DailyStats(get(ComplScheme.TODAY_ABUSES));
        }
        return todayStats;
    }

    public void setTodayStats(@Nonnull DailyStats todayStats) {
        this.todayStats = todayStats;
    }

    @Nullable
    public HistoryStats getHistoryStats() {
        if (historyStats == null && get(ComplScheme.HISTORY_ABUSES) != null) {
            historyStats = new HistoryStats(get(ComplScheme.HISTORY_ABUSES));
        }
        return historyStats;
    }

    public void setHistoryStats(@Nonnull HistoryStats historyStats) {
        this.historyStats = historyStats;
    }

    @Nullable
    public UserWeights getUserWeights() {
        if (userWeights == null && get(ComplScheme.USER_WEIGHTS) != null) {
            userWeights = new UserWeights(get(ComplScheme.USER_WEIGHTS));
        }
        return userWeights;
    }

    public void setUserWeights(@Nonnull UserWeights userWeights) {
        this.userWeights = userWeights;
    }

    public static class DailyStats extends AbstractStringBuilderable {
        private static final String PREFIX = "DailyStats(";
        private static final int PREFIX_LENGTH = PREFIX.length();

        private final long firsttime;
        private final long lasttime;
        private final long day;
        private final long ham;
        private final long spam;
        private final long malic;
        private final long persHam;
        private final long persSpam;
        private final long complaintHam;
        private final long complaintSpam;
        private final long virusCount;
        private final long uniqSdSpam;
        private final long uniqSdHam;
        private final long expertSpam;
        private final long expertHam;

        public DailyStats(@Nonnull final JsonMap map) throws JsonBadCastException {
            firsttime = map.get("firsttime").asLong();
            lasttime = map.get("lasttime").asLong();
            day = map.get("day").asLong();
            ham = map.get("ham").asLong();
            spam = map.get("spam").asLong();
            malic = map.get("malic").asLong();
            persHam = map.get("pers_ham").asLong();
            persSpam = map.get("pers_spam").asLong();
            complaintHam = map.get("complaint_ham").asLong();
            complaintSpam = map.get("complaint_spam").asLong();
            virusCount = map.get("virus_count").asLong();
            uniqSdSpam = map.get("uniq_sd_spam").asLong();
            uniqSdHam = map.get("uniq_sd_ham").asLong();
            expertSpam = map.get("expert_spam").asLong();
            expertHam = map.get("expert_ham").asLong();
        }

        public DailyStats(@Nonnull final Map<String, List<Object>> map) {
            firsttime = map.containsKey("firsttime") ? (int) map.get("firsttime").get(0) : 0L;
            lasttime = map.containsKey("lasttime") ? (int) map.get("lasttime").get(0) : 0L;
            day = map.containsKey("day") ? (int) map.get("day").get(0) : 0L;
            ham = map.containsKey("ham") ? (Long) map.get("ham").get(0) : 0L;
            spam = map.containsKey("spam") ? (Long) map.get("spam").get(0) : 0L;
            malic = map.containsKey("malic") ? (Long) map.get("malic").get(0) : 0L;
            persHam = map.containsKey("pers_ham") ? (Long) map.get("pers_ham").get(0) : 0L;
            persSpam = map.containsKey("pers_spam") ? (Long) map.get("pers_spam").get(0) : 0L;
            complaintHam = map.containsKey("complaint_ham") ? (Long) map.get("complaint_ham").get(0) : 0L;
            complaintSpam = map.containsKey("complaint_spam") ? (Long) map.get("complaint_spam").get(0) : 0L;
            virusCount = map.containsKey("virus_count") ? (Long) map.get("virus_count").get(0) : 0L;
            uniqSdSpam = map.containsKey("uniq_sd_spam") ? (Long) map.get("uniq_sd_spam").get(0) : 0L;
            uniqSdHam = map.containsKey("uniq_sd_ham") ? (Long) map.get("uniq_sd_ham").get(0) : 0L;
            expertSpam = map.containsKey("expert_spam") ? (Long) map.get("expert_spam").get(0) : 0L;
            expertHam = map.containsKey("expert_ham") ? (Long) map.get("expert_ham").get(0) : 0L;
        }

        @NoAllocation
        public long getFirsttime() {
            return firsttime;
        }

        @NoAllocation
        public long getLasttime() {
            return lasttime;
        }

        @NoAllocation
        public long getDay() {
            return day;
        }

        @NoAllocation
        public long getHam() {
            return ham;
        }

        @NoAllocation
        public long getSpam() {
            return spam;
        }

        @NoAllocation
        public long getMalic() {
            return malic;
        }

        @NoAllocation
        public long getExpertHam() {
            return expertHam;
        }

        @NoAllocation
        public long getPersHam() {
            return persHam;
        }

        @NoAllocation
        public long getPersSpam() {
            return persSpam;
        }

        @NoAllocation
        public long getComplaintHam() {
            return complaintHam;
        }

        @NoAllocation
        public long getComplaintSpam() {
            return complaintSpam;
        }

        @NoAllocation
        public long getVirusCount() {
            return virusCount;
        }

        @NoAllocation
        public long getUniqSdSpam() {
            return uniqSdSpam;
        }

        @NoAllocation
        public long getUniqSdHam() {
            return uniqSdHam;
        }

        @NoAllocation
        public long getExpertSpam() {
            return expertSpam;
        }

        protected void toStringBuilderCore(@Nonnull final StringBuilder sb) {
            sb.append(ham);
            sb.append(',');
            sb.append(malic);
            sb.append(',');
            sb.append(spam);
            sb.append(',');
            sb.append(persHam);
            sb.append(',');
            sb.append(persSpam);
            sb.append(',');
            sb.append(uniqSdHam);
            sb.append(',');
            sb.append(uniqSdSpam);
            sb.append(',');
            sb.append(complaintHam);
            sb.append(',');
            sb.append(complaintSpam);
            sb.append(',');
            sb.append(expertSpam);
            sb.append(',');
            sb.append(virusCount);
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            sb.append(PREFIX);
            toStringBuilderCore(sb);
            sb.append(')');
        }

        @NoAllocation
        protected int expectedStringLengthCore() {
            return 10
                + StringBuilderable.calcExpectedStringLength(ham)
                + StringBuilderable.calcExpectedStringLength(malic)
                + StringBuilderable.calcExpectedStringLength(spam)
                + StringBuilderable.calcExpectedStringLength(persHam)
                + StringBuilderable.calcExpectedStringLength(persSpam)
                + StringBuilderable.calcExpectedStringLength(uniqSdHam)
                + StringBuilderable.calcExpectedStringLength(uniqSdSpam)
                + StringBuilderable.calcExpectedStringLength(complaintHam)
                + StringBuilderable.calcExpectedStringLength(complaintSpam)
                + StringBuilderable.calcExpectedStringLength(expertSpam)
                + StringBuilderable.calcExpectedStringLength(virusCount);
        }

        @NoAllocation
        @Override
        public int expectedStringLength() {
            return PREFIX_LENGTH + 1 + expectedStringLengthCore();
        }
    }

    public static class HistoryStats extends DailyStats {
        private static final String PREFIX = "HistoryStats(";
        private static final int PREFIX_LENGTH = PREFIX.length();

        private final long dayCountWithComplaint;
        private final long dayCountWithVirus;

        public HistoryStats(@Nonnull final JsonMap map) throws JsonBadCastException {
            super(map);
            dayCountWithComplaint = map.get("day_count_with_complaint").asLong();
            dayCountWithVirus = map.get("day_count_with_virus").asLong();
        }

        public HistoryStats(@Nonnull final Map<String, List<Object>> map) {
            super(map);
            dayCountWithComplaint = map.containsKey("day_count_with_complaint")
                ? (int) map.get("day_count_with_complaint").get(0) : 0L;
            dayCountWithVirus = map.containsKey("day_count_with_virus")
                ? (int) map.get("day_count_with_virus").get(0) : 0L;
        }

        @NoAllocation
        public long getDayCountWithComplaint() {
            return dayCountWithComplaint;
        }

        @NoAllocation
        public long getDayCountWithVirus() {
            return dayCountWithVirus;
        }

        @Override
        public void toStringBuilderCore(@Nonnull final StringBuilder sb) {
            super.toStringBuilderCore(sb);
            sb.append(',');
            sb.append(dayCountWithComplaint);
            sb.append(',');
            sb.append(dayCountWithVirus);
        }

        @NoAllocation
        @Override
        public int expectedStringLengthCore() {
            return PREFIX_LENGTH
                + super.expectedStringLengthCore()
                + 2
                + StringBuilderable.calcExpectedStringLength(dayCountWithComplaint)
                + StringBuilderable.calcExpectedStringLength(dayCountWithVirus);
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            sb.append(PREFIX);
            toStringBuilderCore(sb);
            sb.append(')');
        }
    }

    public static class UserWeights extends AbstractStringBuilderable {
        private static final String PREFIX = "UserWeights(";
        private static final int PREFIX_LENGTH = PREFIX.length();

        @Nonnull
        private final UserWeight userWeightIn;
        @Nonnull
        private final UserWeight userWeightOut;

        public UserWeights(final JsonMap map) throws JsonBadCastException {
            userWeightIn = UserWeight.parse_in(map);
            userWeightOut = UserWeight.parse_out(map);
        }

        public UserWeights(final Map<String, List<Object>> map) {
            userWeightIn = UserWeight.parse_in(map);
            userWeightOut = UserWeight.parse_out(map);
        }

        @NoAllocation
        @Nonnull
        public UserWeight getUserWeightIn() {
            return userWeightIn;
        }

        @NoAllocation
        @Nonnull
        public UserWeight getUserWeightOut() {
            return userWeightOut;
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            sb.append(PREFIX);
            sb.append("In");
            StringBuilderable.toStringBuilder(sb, userWeightIn);
            sb.append(',');
            sb.append("Out");
            StringBuilderable.toStringBuilder(sb, userWeightOut);
            sb.append(')');
        }

        @NoAllocation
        @Override
        public int expectedStringLength() {
            return PREFIX_LENGTH
                + 7
                + StringBuilderable.calcExpectedStringLength(userWeightIn)
                + StringBuilderable.calcExpectedStringLength(userWeightOut);
        }
    }

    public static class UserWeight extends AbstractStringBuilderable {
        public static final UserWeight EMPTY = new UserWeight();
        private final long creation;
        private final long update;
        private final double weight;
        private final long duration;
        private final boolean malic;

        UserWeight() {
            creation = 0;
            update = 0;
            weight = 0;
            duration = 0;
            malic = false;
        }

        UserWeight(long creation, long update, double weight, long duration, boolean malic) {
            this.creation = creation;
            this.update = update;
            this.weight = weight;
            this.duration = duration;
            this.malic = malic;
        }

        @Nonnull
        public static UserWeight parse_in(@Nonnull final Map<String, List<Object>> map) {
            return new UserWeight(
                map.containsKey("win_timeset") ? (Long) map.get("win_timeset").get(0) : 0L,
                map.containsKey("win_lastset") ? (Long) map.get("win_lastset").get(0) : 0L,
                map.containsKey("win_weight") ? (double) map.get("win_weight").get(0) : 0.0,
                map.containsKey("win_during") ? (int) map.get("win_during").get(0) : 0L,
                map.containsKey("win_during") && ((int) map.get("should_be_malic_in").get(0) != 0)
            );
        }

        @Nonnull
        public static UserWeight parse_in(@Nonnull final JsonMap map) throws JsonBadCastException {
            return new UserWeight(
                map.get("win_timeset").asLong(),
                map.get("win_lastset").asLong(),
                map.get("win_weight").asDouble(),
                map.get("win_during").asLong(),
                map.get("should_be_malic_in").asLong() != 0
            );
        }

        @Nonnull
        public static UserWeight parse_out(@Nonnull final Map<String, List<Object>> map) {
            return new UserWeight(
                map.containsKey("wout_timeset") ? (Long) map.get("wout_timeset").get(0) : 0L,
                map.containsKey("wout_lastset") ? (Long) map.get("wout_lastset").get(0) : 0L,
                map.containsKey("wout_weight") ? (double) map.get("wout_weight").get(0) : 0.0,
                map.containsKey("wout_during") ? (int) map.get("wout_during").get(0) : 0L,
                map.containsKey("wout_during") && ((int) map.get("should_be_malic_out").get(0) != 0)
            );
        }

        @Nonnull
        public static UserWeight parse_out(@Nonnull final JsonMap map) throws JsonBadCastException {
            return new UserWeight(
                map.get("wout_timeset").asLong(),
                map.get("wout_lastset").asLong(),
                map.get("wout_weight").asDouble(),
                map.get("wout_during").asLong(),
                map.get("should_be_malic_out").asLong() != 0
            );
        }

        @NoAllocation
        public long getCreation() {
            return creation;
        }

        @NoAllocation
        public long getUpdate() {
            return update;
        }

        @NoAllocation
        public double getWeight() {
            return weight;
        }

        @NoAllocation
        public long getDuration() {
            return duration;
        }

        @NoAllocation
        public boolean isMalic() {
            return malic;
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            sb.append("(");
            sb.append(creation);
            sb.append(',');
            sb.append(update);
            sb.append(',');
            sb.append(weight);
            sb.append(',');
            sb.append(duration);
            sb.append(',');
            sb.append(malic ? 1 : 0);
            sb.append(')');
        }

        @NoAllocation
        @Override
        public int expectedStringLength() {
            return 7
                + StringBuilderable.calcExpectedStringLength(creation)
                + StringBuilderable.calcExpectedStringLength(update)
                + StringBuilderable.calcExpectedStringLength(weight)
                + StringBuilderable.calcExpectedStringLength(duration);
        }
    }
}
