package ru.yandex.client.so.shingler;

import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.LongStream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.errorprone.annotations.NoAllocation;

import ru.yandex.digest.Fnv;
import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.function.StringBuilderable;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonObject;

public class DictShingleInfo extends AbstractStringBuilderable
{
    public static final String SHIDENT = "<0x";
    public static final int SHIDENT_LENGTH = SHIDENT.length();

    private static final String PREFIX = "DictShingleInfo(";
    private static final int PREFIX_LENGTH = PREFIX.length();
    private static final String FLAGS_TAG = "fla=";

    public enum MailType {
        INBOX("in"),
        OUTBOX("out"),
        CORP("corp"),
        COUNT("");

        private final String route;
        private static final HashMap<String, MailType> routeToMailType = new HashMap<>();

        static {
            for (MailType mailType : MailType.values()) {
                routeToMailType.put(mailType.route(), mailType);
            }
        }

        MailType(final String route) {
            this.route = route;
        }

        public String route() {
            return route;
        }

        public static MailType fromRoute(final String route) {
            return routeToMailType.get(route);
        }
    }

    public enum FieldType {
        UNKNOWN(-1, null),
        FIRST_TIME(1, "ft"),
        LAST_TIME(2, "lt"),
        HAM(3, "h"),
        SPAM(4, "s"),
        COMPLHAM(5, "ch"),
        COMPLSPAM(6, "cs"),
        VIRUS(7, "v"),
        FISHING(8, "f"),
        FISHING_YANDEX(9, "fy"),
        HACKED(10, "hk"),
        EXPERT_COMPLHAM(11, "ech"),
        EXPERT_COMPLSPAM(12, "ecs"),
        COUNT(13, null);

        private final int id;
        private final String logSign;

        FieldType(final int id, final String logSign) {
            this.id = id;
            this.logSign = logSign;
        }

        public int id() {
            return id;
        }

        public String logSign() {
            return logSign;
        }
    }

    public enum Time {
        TODAY(0, "t", "tod"),
        YESTERDAY(1, "y", "yes"),
        HISTORY(2, "h", "his"),
        COUNT(3, null, null);

        private final int id;
        private final String logSign;
        private final String tag;

        Time(final int id, final String logSign, final String tag) {
            this.id = id;
            this.logSign = logSign;
            this.tag = tag;
        }

        public int id() {
            return id;
        }

        public String logSign() {
            return logSign;
        }

        public String tag() {
            return tag;
        }
    };

    private static final HashMap<Long, Time> times = new HashMap<>();
    static {
        for (final Time time: Time.values()) {
            times.put((long)time.id(), time);
        }
    }

    @Nonnull
    private Long shingle;
    @Nullable
    private final Map<Time, Fields> counters;
    private Long langFlags;
    private MailType mailType;

    public DictShingleInfo() {
        langFlags = 0L;
        counters = new HashMap<>();
        mailType = MailType.INBOX;
        shingle = 0L;
    }

    public DictShingleInfo(@Nonnull final String value) throws ShingleException {
        langFlags = 0L;
        counters = new HashMap<>();
        mailType = MailType.INBOX;
        try {
            int n = value.indexOf('-');
            shingle =
                Long.parseUnsignedLong(
                    value.substring(value.startsWith(SHIDENT) ? SHIDENT_LENGTH : 0, n), 16);
            n = value.indexOf(": ");
            final String s = value.substring(n + 2);
            Time time;
            for (int id = 0; id < Time.COUNT.id(); id++) {
                time = times.get((long)id);
                n = s.indexOf(time.tag + "=") + 2;
                counters.put(time, new Fields(s.substring(n + time.tag.length(), s.indexOf("' ", n))));
            }
            n = s.indexOf(FLAGS_TAG) + FLAGS_TAG.length() + 1;
            langFlags = Long.parseUnsignedLong(s.substring(n, s.indexOf("'", n)));
        } catch (Exception e) {
            shingle = 0L;
            throw new ShingleException("Parsing of shingle's info failed: " + e, e);
        }
    }

    public DictShingleInfo(@Nonnull final JsonObject jsonValue, final String route)
        throws JsonBadCastException, ShingleException
    {
        langFlags = 0L;
        counters = new HashMap<>();
        try {
            mailType = MailType.fromRoute(route.toLowerCase(Locale.ROOT));
            shingle = Long.parseUnsignedLong(jsonValue.asString());
        } catch (NumberFormatException e) {
            throw new ShingleException(
                "DictShingleInfo failed to parse jsonValue=" + jsonValue.asString() + " : " + e,
                e);
        }
        if (mailType == null) {
            mailType = MailType.INBOX;
        }
    }

    public DictShingleInfo(final Shingle shingle) {
        langFlags = 0L;
        counters = new HashMap<>();
        mailType = MailType.INBOX;
        this.shingle = shingle.shingleHash();
    }

    public Long shingle() {
        return shingle;
    }

    public Map<Time, Fields> counters() {
        return counters;
    }

    public Long languageRaw() {
        return langFlags;
    }

    public void setLanguageRaw(Long langFlags) {
        this.langFlags = langFlags;
    }

    public MailType mailType() {
        return mailType;
    }

    public void setMailType(final MailType mailType) {
        this.mailType = mailType;
    }

    @Override
    public String toString() {
        return Long.toHexString(shingle).toUpperCase() + "-" + langFlags + "--1-" + mailType.ordinal();
    }

    public static Long calcShingle(final String word) {
        long result = 0L;
        if (!word.isEmpty()) {
            result = Fnv.fnv64(word);
        }
        return result;
    }

    @NoAllocation
    @Override
    public int expectedStringLength() {
        int length = PREFIX_LENGTH + 2 + Long.toHexString(shingle).length();
        for (int id = 0; id < Time.COUNT.id(); id++) {
            length += 1 + (counters == null ? 0 :
                StringBuilderable.calcExpectedStringLength(counters.get(times.get((long)id))));
        }
        return length;
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append(PREFIX);
        sb.append(Long.toHexString(shingle).toUpperCase());
        sb.append('-');
        sb.append(langFlags);
        for (int id = 0; id < Time.COUNT.id(); id++) {
            sb.append('-');
            StringBuilderable.toStringBuilder(sb, counters.get(times.get((long)id)));
        }
        sb.append(')');
    }

    public static class Fields extends EnumMap<FieldType, Long> implements StringBuilderable
    {
        private static final long serialVersionUID = 0L;
        private static final HashMap<Long, FieldType> fieldTypes = new HashMap<>();
        static {
            for (final FieldType fieldType: FieldType.values()) {
                fieldTypes.put((long)fieldType.id(), fieldType);
            }
        }

        public Fields() {
            super(FieldType.class);
            for (final FieldType fieldType: FieldType.values()) {
                put(fieldType, 0L);
            }
        }

        public Fields(final String fieldInfo)
            throws ShingleException
        {
            super(FieldType.class);
            for (final FieldType fieldType: FieldType.values()) {
                put(fieldType, 0L);
            }
            try {
                final List<Long> counters =
                    Arrays.stream(fieldInfo.split("-")).map(Long::parseLong).collect(Collectors.toList());
                for (int id = 0; id < counters.size(); id++) {
                    put(fieldTypes.get(FieldType.FIRST_TIME.id + (long)id), counters.get(id));
                }
            } catch (Exception e) {
                throw new ShingleException("Parsing of shingle's counters failed: " + e, e);
            }
        }

        @Override
        public String toString() {
            return '-' + IntStream.range(3, FieldType.COUNT.id()).boxed()
                .map(id -> Long.toString(get(fieldTypes.get((long)id))))
                .collect(Collectors.joining("-"));
        }

        public String toLog() {
            return LongStream.range(1L, FieldType.COUNT.id()).boxed()
                .map(fieldTypes::get).map(ft -> ft.logSign() + "=" + get(ft))
                .collect(Collectors.joining(","));
        }

        @NoAllocation
        @Override
        public int expectedStringLength() {
            int length = 0;
            for (int id = 3; id < FieldType.COUNT.id(); id++) {
                length += 1 + StringBuilderable.calcExpectedStringLength(get(fieldTypes.get((long)id)));
            }
            return length;
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            for (int id = 3; id < FieldType.COUNT.id(); id++) {
                sb.append('-');
                sb.append(get(fieldTypes.get((long)id)));
            }
        }

        @Override
        public boolean isEmpty() {
            for (int id = 1; id < FieldType.COUNT.id(); id++) {
                if (get(fieldTypes.get((long)id)) != 0) {
                    return false;
                }
            }
            return true;
        }

        @Override
        public void clear() {
            for (int id = 1; id < FieldType.COUNT.id(); id++) {
                put(fieldTypes.get((long)id), 0L);
            }
        }
    }
}
