package ru.yandex.client.so.shingler;

import java.util.concurrent.Future;
import java.util.function.Supplier;

import javax.annotation.Nonnull;

import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;

import ru.yandex.concurrent.CompletedFuture;
import ru.yandex.data.compressor.CompressorException;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.CallbackFutureBase;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;

public class DictShinglerClient extends AbstractShinglerClient<DictShinglerClient, DictShinglerResult>
{
    private static final String NAME = "Dict";
    private static final CompletedFuture<DictShinglerResult> EMPTY_RESULT =
        new CompletedFuture<>(DictShinglerResult.EMPTY);

    public enum RequestType {
        UNKNOWN(null),
        ABUSE("dict_abuse"),
        PUT("dict_put"),
        GET("dict_get"),
        VIRUS("dict_virus"),
        FISHING("dict_fishing"),
        FISHING_YANDEX("dict_fishing_yan"),
        HACKED("dict_hacked"),
        COUNT(null);

        private final String uri;

        RequestType(final String uri) {
            this.uri = uri;
        }

        public String uri() {
            return uri;
        }
    };

    public DictShinglerClient(
        @Nonnull final SharedConnectingIOReactor reactor,
        @Nonnull final ImmutableHttpHostConfig config)
    {
        super(reactor, config);
    }

    protected DictShinglerClient(
        @Nonnull final CloseableHttpAsyncClient client,
        @Nonnull final DictShinglerClient sample)
    {
        super(client, sample);
    }

    @Override
    public DictShinglerClient adjust(@Nonnull final CloseableHttpAsyncClient client) {
        return new DictShinglerClient(client, this);
    }

    final static String getGetQuery(@Nonnull final Shingles shingles) throws CompressorException {
        return getGetQuery(new DictShinglerResult(shingles));
    }

    final static String getGetQuery(@Nonnull final DictShinglerResult shingles)
        throws CompressorException
    {
        return shingles.getQuery();
    }

    @Override
    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<DictShinglerResult> getShingles(
        @Nonnull final AbstractShinglesMap<?> shingles,
        @Nonnull final Supplier<? extends HttpClientContext> contextGenerator,
        @Nonnull final FutureCallback<? super DictShinglerResult> callback)
        throws ShinglerClientException, ShingleException
    {
        if (shingles.isEmpty()) {
            callback.completed(DictShinglerResult.EMPTY);
            return EMPTY_RESULT;
        }
        GetQueryCallbackProxy callbackProxy = new GetQueryCallbackProxy(callback);
        try {
            execute(
                host,
                new BasicAsyncRequestProducerGenerator(
                    RequestType.GET.uri(),
                    getGetQuery((Shingles) shingles),
                    ContentType.APPLICATION_JSON),
                AsyncStringConsumerFactory.OK,
                contextGenerator,
                callbackProxy);
        } catch (CompressorException e) {
            throw new ShinglerClientException(NAME, "Failed to form GET request to Dict Shingler" + e.toString(), e);
        }
        return callbackProxy;
    }

    @Override
    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(
        @Nonnull final DictShinglerResult shinglesData,
        @Nonnull final Supplier<? extends HttpClientContext> contextGenerator,
        @Nonnull final FutureCallback<String> callback)
        throws ShinglerClientException, ShingleException
    {
        if (shinglesData.isEmpty()) {
            callback.completed(OK);
            return SUCCESS_PUT_RESULT;
        }
        PutQueryCallbackProxy callbackProxy = new PutQueryCallbackProxy(callback);
        try {
            execute(
                host,
                new BasicAsyncRequestProducerGenerator(
                    RequestType.PUT.uri(),
                    getPutQuery(shinglesData),
                    ContentType.APPLICATION_OCTET_STREAM),
                EmptyAsyncConsumerFactory.OK,
                contextGenerator,
                callbackProxy);
        } catch (Exception e) {
            throw new ShinglerClientException(NAME, "Failed to form PUT request to Dict Shingler: " + e.toString(), e);
        }
        return callbackProxy;
    }

    public static String getPutQuery(@Nonnull final DictShinglerResult shingles)
        throws ShinglerClientException
    {
        try {
            return shingles.getQuery();
        } catch (CompressorException e) {
            throw new ShinglerClientException(
                NAME,
                "Failed to prepare PUT request to Dict Shingler: " + e.toString(),
                e);
        }
    }

    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(@Nonnull final String query)
    {
        return putShingles(query, httpClientContextGenerator(), StringFutureCallback.INSTANCE);
    }

    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(
        @Nonnull final String query,
        @Nonnull final FutureCallback<String> callback)
    {
        return putShingles(query, httpClientContextGenerator(), callback);
    }

    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(
        @Nonnull final String query,
        @Nonnull final Supplier<? extends HttpClientContext> contextGenerator,
        @Nonnull final FutureCallback<String> callback)
    {
        PutQueryCallbackProxy callbackProxy = new PutQueryCallbackProxy(callback);
        execute(
            host,
            new BasicAsyncRequestProducerGenerator(query),
            EmptyAsyncConsumerFactory.OK,
            contextGenerator,
            callbackProxy);
        return callbackProxy;
    }

    private static class GetQueryCallbackProxy extends CallbackFutureBase<DictShinglerResult, String>
    {
        @Nonnull
        GetQueryCallbackProxy(@Nonnull final FutureCallback<? super DictShinglerResult> callback) {
            super(callback);
        }

        @Override
        @Nonnull
        protected DictShinglerResult convertResult(@Nonnull final String response) {
            try {
                return new DictShinglerResult(response);
            } catch (ShingleException e) {
                failed(e);
                return DictShinglerResult.EMPTY;
            }
        }

        @Override
        protected void onComplete(final DictShinglerResult result) {
            callback.completed(result);
        }

        @Override
        protected void onFailure(final Exception e) {
            //System.err.println("DictShinglerClient.GetQueryCallbackProxy failed: " + e.toString());
            callback.failed(e);
        }
    }

    private static class PutQueryCallbackProxy extends CallbackFutureBase<String, Void>
    {
        @Nonnull
        PutQueryCallbackProxy(@Nonnull final FutureCallback<String> callback) {
            super(callback);
        }

        @Override
        @Nonnull
        protected String convertResult(@Nonnull final Void unused) {
            return OK;
        }

        @Override
        protected void onFailure(final Exception e) {
            //System.err.println("DictShinglerClient.PutQueryCallbackProxy failed: " + e.toString());
            //callback.failed(e);
            callback.completed(null);
        }
    }
}
