package ru.yandex.client.so.shingler;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.errorprone.annotations.NoAllocation;

import ru.yandex.data.compressor.CompressorException;
import ru.yandex.data.compressor.DataCompressor;
import ru.yandex.function.StringBuilderable;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.parser.mail.errors.ErrorInfo;

public class DictShinglerResult extends AbstractShinglesMap<List<DictShingleInfo>>
{
    public static final DictShinglerResult EMPTY = new DictShinglerResult();
    public static final ShingleType DICT_SHINGLE = ShingleType.UNKNOWN;

    private static final long serialVersionUID = 0L;
    private static final String PREFIX = "DictShinglerResult(";
    private static final int PREFIX_LENGTH = PREFIX.length();
    private static final List<DictShingleInfo> EMPTY_DICT_SHINGLE_RESULT = Collections.emptyList();

    @Nonnull
    private final List<ErrorInfo> errors;

    public DictShinglerResult() {
        super();
        errors = Collections.emptyList();
    }

    public DictShinglerResult(@Nonnull final Exception e) {
        super();
        put(DICT_SHINGLE, new ArrayList<>());
        errors = new ArrayList<>(1);
        errors.add(
            new ErrorInfo(
                ErrorInfo.Scope.SHINGLER,
                ErrorInfo.Type.EXTERNAL_COMPONENT_ERROR,
                "Dictionary Shingler failed to respond",
                e));
    }

    public DictShinglerResult(@Nonnull final String response)
        throws ShingleException
    {
        super();
        put(DICT_SHINGLE, new ArrayList<>());
        String uncompressedResponse;
        try {
            uncompressedResponse = DataCompressor.LZO.unbase64AndUncompress(response);
        } catch (CompressorException e) {
            throw new ShingleException("Uncompressing of DictShingler's response failed: " + e.toString());
        }
        errors = new ArrayList<>();
        boolean empty = true;
        int end = -1;
        while (uncompressedResponse != null) {
            int start = uncompressedResponse.indexOf('<', end + 1);
            end = uncompressedResponse.indexOf('>', start + 1);
            if (start == -1 || end == -1) {
                break;
            }
            empty = false;
            String str = uncompressedResponse.substring(start, end);
            if (str.startsWith(DictShingleInfo.SHIDENT)) {
                try {
                    get(DICT_SHINGLE).add(new DictShingleInfo(str));
                } catch (ShingleException e) {
                    errors.add(
                        new ErrorInfo(
                            ErrorInfo.Scope.SHINGLER_RESPONSE,
                            ErrorInfo.Type.SYNTAX_ERROR,
                            "Failed to parse shingle from " + str,
                            e));
                }
            }
        }
        if (empty) {
            errors.add(
                new ErrorInfo(
                    ErrorInfo.Scope.SHINGLER_RESPONSE,
                    ErrorInfo.Type.FIELD_MISSING,
                    "No shingles found in '" + response + '\''));
        }
    }

    public DictShinglerResult(@Nonnull final JsonList jsonList, final String route)
        throws JsonBadCastException, ShingleException
    {
        super();
        put(DICT_SHINGLE, new ArrayList<>());
        for (final JsonObject jsonObject : jsonList) {
            get(DICT_SHINGLE).add(new DictShingleInfo(jsonObject, route));
        }
        errors = new ArrayList<>();
    }

    public DictShinglerResult(@Nonnull final Shingles shingles) {
        super();
        put(DICT_SHINGLE, new ArrayList<>());
        for (final Map.Entry<ShingleType, Map<Long, Shingle>> entry : shingles.entrySet()) {
            for (final Map.Entry<Long, Shingle> shingleInfo: entry.getValue().entrySet()) {
                get(DICT_SHINGLE).add(new DictShingleInfo(shingleInfo.getValue()));
            }
        }
        errors = new ArrayList<>();
    }

    public DictShinglerResult(final DictShinglerResult data) {
        errors = new ArrayList<>(data.errors());
        for (final Map.Entry<ShingleType, List<DictShingleInfo>> entry : data.entrySet()) {
            for (final DictShingleInfo result : entry.getValue()) {
                computeIfAbsent(entry.getKey(), x -> new ArrayList<>()).add(result);
            }
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public void addAll(final AbstractShinglesMap<?> data) throws ShingleException {
        for (final Map.Entry<ShingleType, List<DictShingleInfo>> entry
                : ((AbstractShinglesMap<List<DictShingleInfo>>)data).entrySet()) {
            computeIfAbsent(entry.getKey(), x -> new ArrayList<>()).addAll(entry.getValue());
        }
    }

    @Override
    public DictShinglerResult getEmpty() {
        return EMPTY;
    }

    @NoAllocation
    @Nonnull
    public List<DictShingleInfo> shinglesFor(final ShingleType type) {
        return getOrDefault(0, EMPTY_DICT_SHINGLE_RESULT);
    }

    @NoAllocation
    @Nonnull
    public List<ErrorInfo> errors() {
        return errors;
    }

    @Override
    public int expectedStringLength() {
        int len = PREFIX_LENGTH + 4 - 1 + (!isEmpty() ? (get(DICT_SHINGLE).size() << 1) : 0);
        if (!isEmpty()) {
            for (final DictShingleInfo shingleInfo : get(DICT_SHINGLE)) {
                len += StringBuilderable.calcExpectedStringLength(shingleInfo);
            }
        }
        len += StringBuilderable.calcExpectedStringLength(errors);
        return len;
    }

    @Override
    public void toStringBuilder(@Nonnull final StringBuilder sb) {
        sb.append(PREFIX);
        sb.append('[');
        if (!isEmpty()) {
            boolean first = true;
            for (final DictShingleInfo shingleInfo : get(DICT_SHINGLE)) {
                if (first) {
                    first = false;
                } else {
                    sb.append(',');
                }
                StringBuilderable.toStringBuilder(sb, shingleInfo);
            }
        }
        sb.append(']');
        sb.append(',');
        StringBuilderable.toStringBuilder(sb, errors);
        sb.append(')');
    }

    public String getQuery() throws CompressorException {
        if (isEmpty()) {
            return null;
        }
        StringBuilder sb = new StringBuilder("42=");
        for (final DictShingleInfo shingleInfo : get(DICT_SHINGLE)) {
            sb.append(shingleInfo);
            sb.append(',');
        }
        sb.append('&');
        return DataCompressor.LZO.compressAndBase64(sb.toString());
    }

    public String getHumanQuery() {
        if (isEmpty()) {
            return null;
        }
        StringBuilder sb = new StringBuilder("42=");
        for (final DictShingleInfo shingleInfo : get(DICT_SHINGLE)) {
            sb.append(shingleInfo);
            sb.append(',');
        }
        sb.append('&');
        return sb.toString();
    }

    @Override
    public boolean isEmpty() {
        return (size() < 1 || get(DICT_SHINGLE).size() < 1);
    }
}
