package ru.yandex.client.so.shingler;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;

public enum FreemailScheme implements Scheme {
    COUNTERS(SchemeType.DB, FreemailFieldSet.COUNTERS, "uuid"),
    COUNTERS_GET(SchemeType.DB, FreemailFieldSet.COUNTERS, "uuid", "date"),
    RECIPIENTS_MAX(SchemeType.DB, FreemailFieldSet.RECIPIENTS_MAX, Map.of(
        "recepients_count", 0L,
        "recepients_max_count", Map.of("f1", "recepients_count", "f2", "recepients_max_count")
    ), "uuid"),
    ACTIVE(SchemeType.DB, FreemailFieldSet.INFO, "uuid", "date"),
    TIME(SchemeType.DB, FreemailFieldSet.TIME, "uuid"),
    TIME_GET(SchemeType.DB, FreemailFieldSet.INFO, "uuid", "date"),
    COMPLAINT(SchemeType.DB, FreemailFieldSet.COMPLAINT, "uuid", "hash", "spam"),
    COMPLAINT_GET(SchemeType.DB, FreemailFieldSet.COMPLAINT, "uuid", "date"),
    GEO(SchemeType.DB, FreemailFieldSet.GEO, "uuid", "geo"),
    GEO_GET(SchemeType.DB, FreemailFieldSet.GEO, "uuid", "date"),
    BOUNCE(SchemeType.DB, FreemailFieldSet.BOUNCE, "uuid"),
    BOUNCE_GET(SchemeType.DB, FreemailFieldSet.BOUNCE, "uuid", "date"),
    BOUNCE_TOTALS(SchemeType.DB, FreemailFieldSet.BOUNCE_TOTALS, "uuid"),
    BOUNCE_HISTORY(SchemeType.DB, FreemailFieldSet.BOUNCE_HISTORY, "uuid", "date"),
    DOMAIN(SchemeType.DB, FreemailFieldSet.DOMAIN, "domain", "date"),
    DOMAIN_ALL(SchemeType.DB, FreemailFieldSet.DOMAIN, "domain", "date"),
    CURRENT(SchemeType.CURRENT_PERIOD, FreemailFieldSet.CURRENT),
    PDD(SchemeType.PROXY, FreemailFieldSet.PDD, "param.domain");

    private static final Map<String, Set<FreemailScheme>> fieldToScheme = new HashMap<>();

    private final SchemeType schemeType;
    private final FreemailFieldSet fieldSet;
    @SuppressWarnings("ImmutableEnumChecker")
    private final Set<String> keyFields;
    @SuppressWarnings("ImmutableEnumChecker")
    private final Map<String, ?> predefinedFields;

    static {
        for (FreemailScheme scheme : FreemailScheme.values()) {
            for (Map.Entry<String, String> entry : scheme.fields().entrySet()) {
                fieldToScheme.computeIfAbsent(entry.getKey(), x -> new HashSet<>()).add(scheme);
            }
        }
    }

    FreemailScheme(final SchemeType schemeType, final FreemailFieldSet fieldSet, final String... keyFields) {
        this.schemeType = schemeType;
        this.fieldSet = fieldSet;
        this.keyFields = Set.of(keyFields);
        predefinedFields = null;
    }

    FreemailScheme(
        final SchemeType schemeType,
        final FreemailFieldSet fieldSet,
        final Map<String, ?> predefinedFields,
        final String... keyFields)
    {
        this.schemeType = schemeType;
        this.fieldSet = fieldSet;
        this.keyFields = Set.of(keyFields);
        this.predefinedFields = Map.copyOf(predefinedFields);
    }

    @Override
    public SchemeType schemeType() {
        return schemeType;
    }

    @Override
    public FreemailFieldSet fieldSet() {
        return fieldSet;
    }

    @Override
    public Map<String, String> fields() {
        return fieldSet.fields();
    }

    @Override
    public Map<String, ?> predefinedFields() {
        return predefinedFields;
    }

    @Override
    public Set<String> keyFields() {
        return keyFields;
    }

    @Override
    public Scheme fromName(String name) {
        return FreemailScheme.valueOf(name);
    }

    @Nonnull
    public static Set<FreemailScheme> schemesFromField(final String field) throws RuntimeException {
        Set<FreemailScheme> schemes = fieldToScheme.get(field);
        if (schemes == null) {
            throw new NullPointerException("FreemailScheme.schemesFromField returns null for field '" + field + "'");
        }
        return schemes;
    }

    @Override
    public String shingleField() {
        return "uuid";
    }
}
