package ru.yandex.client.so.shingler;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;

import javax.annotation.Nonnull;

import com.google.errorprone.annotations.NoAllocation;

import ru.yandex.function.StringBuilderable;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;

public interface GeneralShingleInfo<S extends Scheme>
{
    long serialVersionUID = 0L;

    boolean containsKey(S scheme);

    Map<String, List<Object>> get(S scheme);

    Map<String, List<Object>> remove(S scheme);

    Map<String, List<Object>> computeIfAbsent(S scheme, Function<S, Map<String, List<Object>>> mappingFunction);

    Set<S> keySet();

    Set<Map.Entry<S, Map<String, List<Object>>>> entrySet();

    int size();

    GeneralShingleInfo<S> copy();

    @Nonnull
    default String name() {
        return "General";
    }

    static Object castValue(@Nonnull final String className, final Object value) {
        if (value instanceof Number) {
            switch (className) {
                case "Integer":
                    return ((Number) value).intValue();
                case "Byte":
                    return ((Number) value).byteValue();
                case "Short":
                    return ((Number) value).shortValue();
                case "Double":
                    return ((Number) value).doubleValue();
                case "String":
                    return value.toString();
                default:
                    return ((Number) value).longValue();
            }
        } else if (value instanceof String) {
            switch (className) {
                case "ULong":
                    return ((String) value).startsWith("-") ? Long.parseLong((String) value)
                            : Long.parseUnsignedLong((String) value);
                case "Long":
                    return Long.parseLong((String) value);
                case "Integer":
                    return Integer.parseInt((String) value);
                case "Byte":
                    return Byte.parseByte((String) value);
                case "Short":
                    return Short.parseShort((String) value);
                case "Double":
                    return Double.parseDouble((String) value);
                default:
                    return value;
            }
        } else {
            return value;
        }
        //return className.equals("ULong") ? (Long)value : Class.forName(CLASS_PATH + className).cast(value);
    }

    default Object parseJsonValue(@Nonnull final String className, final JsonObject jsonValue)
            throws JsonBadCastException, ShingleException
    {
        if (jsonValue == null) {
            return null;
        }
        final Object value;
        switch(jsonValue.type()) {  // expected trivial types only
            case LONG:
                switch (className) {
                    case "Integer":
                        value = (int) jsonValue.asLong();
                        break;
                    case "Byte":
                        value = (byte) jsonValue.asLong();
                        break;
                    case "Short":
                        value = (short) jsonValue.asLong();
                        break;
                    case "Double":
                        value = jsonValue.asDouble();
                        break;
                    default:
                        value = jsonValue.asLong();
                        break;
                }
                break;
            case STRING:
                switch (className) {
                    case "ULong":
                        value = jsonValue.asString().startsWith("-") ? Long.parseLong(jsonValue.asString())
                                : Long.parseUnsignedLong(jsonValue.asString());
                        break;
                    case "Long":
                        value = Long.parseLong(jsonValue.asString());
                        break;
                    case "Integer":
                        value = Integer.parseInt(jsonValue.asString());
                        break;
                    case "Byte":
                        value = Byte.parseByte(jsonValue.asString());
                        break;
                    case "Short":
                        value = Short.parseShort(jsonValue.asString());
                        break;
                    case "Double":
                        value = jsonValue.asDouble();
                        break;
                    default:
                        value = jsonValue.asString();
                        break;
                }
                break;
            case BOOLEAN:
                value = jsonValue.asBoolean();
                break;
            case DOUBLE:
                value = jsonValue.asDouble();
                break;
            default:
                throw new ShingleException(name() + "ShingleInfo failed: encountered unsupported type '"
                    + jsonValue.type() + "' for counter of '" + className + "' type in data");
        }
        return value;
    }

    @SuppressWarnings("unchecked")
    default void setSchemeCounter(@Nonnull final S scheme, @Nonnull final String counter, final Object value)
        throws ShingleException
    {
        try {
            Object finalValue, oldValue;
            String className, counterName;
            AtomicReference<String> key = new AtomicReference<>("");
            if (scheme.fields().containsKey(counter)) {
                counterName = counter;
                className = scheme.fields().get(counter);
            } else {
                scheme.fields().forEach((k, v) -> {
                    if (k.startsWith(counter + '.')) {
                        key.set(k.substring(counter.length() + 1));
                    }
                });
                if (!key.get().isEmpty()) {
                    counterName = counter + '.' + key.get();
                    className = scheme.fields().get(counterName);
                } else {
                    return;       // we ignore counters/fields that are not described in scheme
                }
            }
            if (value instanceof List) {
                List<Object> values = (List<Object>) value;
                List<Object> counterValues = computeIfAbsent(scheme, x -> new HashMap<>())
                     .computeIfAbsent(counterName, x -> new ArrayList<>(values.size()));
                ((ArrayList<Object>) counterValues).ensureCapacity(values.size());
                for (int i = 0; i < values.size(); i++) {
                    finalValue = castValue(
                        className,
                        key.get().isEmpty() ? values.get(i) : ((Map<String, Object>) values.get(i)).get(key.get()));
                    if (i >= counterValues.size()) {
                        counterValues.add(finalValue);
                    } else {
                        counterValues.set(i, finalValue);
                    }
                }
            } else {
                finalValue = castValue(
                    className,
                    key.get().isEmpty() ? value : ((Map<String, Object>) value).get(key.get()));
                List<Object> counterValues = computeIfAbsent(scheme, x -> new HashMap<>())
                    .computeIfAbsent(counterName, x -> new ArrayList<>());
                oldValue = counterValues.size() > 0 ? castValue(className, counterValues.get(0)) : null;
                if (oldValue == null) {
                    counterValues.add(finalValue);
                } else {
                    counterValues.set(0, finalValue);
                }
            }
        } catch (Exception e) {
            throw new ShingleException(name() + "ShingleInfo failed to parse counter '" + counter
                + "' in data for scheme '" + scheme.name().toLowerCase() + "': " + e, e);
        }
    }

    @SuppressWarnings("unchecked")
    default void addSchemeCounter(@Nonnull final S scheme, @Nonnull final String counter, final List<Object> values)
        throws ShingleException
    {
        try {
            Object value, oldValue;
            String className, counterName;
            AtomicReference<String> key = new AtomicReference<>("");
            if (scheme.fields().containsKey(counter)) {
                counterName = counter;
                className = scheme.fields().get(counter);
            } else {
                scheme.fields().forEach((k, v) -> {
                    if (k.startsWith(counter + '.')) {
                        key.set(k.substring(counter.length() + 1));
                    }
                });
                if (!key.get().isEmpty()) {
                    counterName = counter + '.' + key.get();
                    className = scheme.fields().get(counterName);
                } else {
                    return;       // we ignore counters/fields that are not described in scheme
                }
            }
            List<Object> counterValues = computeIfAbsent(scheme, x -> new HashMap<>())
                .computeIfAbsent(counterName, x -> new ArrayList<>(values.size()));
            ((ArrayList<Object>) counterValues).ensureCapacity(values.size());
            for (int i = 0; i < values.size(); i++) {
                value = castValue(
                    className,
                    key.get().isEmpty() ? values.get(i) : ((Map<String, Object>) values.get(i)).get(key.get()));
                if (i >= counterValues.size()) {
                    counterValues.add(value);
                } else {
                    oldValue = castValue(className, counterValues.get(i));
                    switch (className) {
                        case "Long":
                        case "ULong":
                            counterValues.set(i, (Long) oldValue + (Long) value);
                            break;
                        case "Integer":
                            counterValues.set(i, (Integer) oldValue + (Integer) value);
                            break;
                        case "Byte":
                            counterValues.set(i, (Byte) oldValue + (Byte) value);
                            break;
                        case "Short":
                            counterValues.set(i, (Short) oldValue + (Short) value);
                            break;
                        case "Double":
                            counterValues.set(i, (Double) oldValue + (Double) value);
                            break;
                        case "String":
                            counterValues.set(i, (String) oldValue + (String) value);
                            break;
                        default:
                            counterValues.add(value);
                    }
                }
            }
        } catch (Exception e) {
            throw new ShingleException(name() + "ShingleInfo failed to parse counter '" + counter
                + "' in data for scheme '" + scheme.name().toLowerCase() + "': " + e, e);
        }
    }

    default void loadSchemeCounters(@Nonnull final S scheme, @Nonnull final Map<String, Object> counters)
        throws ShingleException
    {
        for (final String counter : counters.keySet()) {
            setSchemeCounter(scheme, counter, counters.get(counter));
        }
    }

    default void loadSchemeCounters(@Nonnull final S scheme, @Nonnull final JsonMap jsonCounters)
        throws ShingleException
    {
        for (final String counter : jsonCounters.keySet()) {
            try {
                final String className;
                final JsonObject jsonValue;
                final String counterName;
                if (scheme.fields().containsKey(counter)) {
                    counterName = counter;
                    className = scheme.fields().get(counter);
                    jsonValue = jsonCounters.get(counter);
                } else {
                    AtomicReference<String> key = new AtomicReference<>("");
                    scheme.fields().forEach((k, v) -> {
                        if (k.startsWith(counter + '.')) {
                            key.set(k.substring(counter.length() + 1));
                        }
                    });
                    if (!key.get().isEmpty()) {
                        counterName = counter + '.' + key.get();
                        className = scheme.fields().get(counterName);
                        jsonValue = jsonCounters.get(counter).asMap().get(key.get());
                    } else {
                        continue;       // we ignore counters/fields that are not described
                        // throw new ShingleException(name() + "ShingleInfo failed: unknown key '" + counter
                        //    + "' in data");
                    }
                }
                computeIfAbsent(scheme, x -> new HashMap<>())
                    .computeIfAbsent(counterName, x -> new ArrayList<>())
                    .add(castValue(className, parseJsonValue(className, jsonValue)));
            } catch (Exception e) {
                throw new ShingleException(name() + "ShingleInfo failed to parse counter '" + counter
                    + "' in data for scheme '" + scheme.name().toLowerCase() + "': " + e, e);
            }
        }
    }

    default void checkCounters(@Nonnull final S scheme) throws ShingleException {
        if (containsKey(scheme)) {
            for (final String keyField : scheme.keyFields()) {
                if (!get(scheme).containsKey(keyField)) {
                    throw new ShingleException(name() + "ShingleInfo: there is no key field '" + keyField + "' for "
                        + "scheme '" + scheme.name().toLowerCase() + "' in data");
                }
            }
        }
    }

    default void loadSchemeCounters(@Nonnull final S scheme, @Nonnull final List<JsonMap> jsonCountersList)
        throws ShingleException
    {
        for (final JsonMap jsonCounters : jsonCountersList) {
            loadSchemeCounters(scheme, jsonCounters);
        }
        checkCounters(scheme);
    }

    default void loadSchemeCounters(@Nonnull final S scheme, @Nonnull final JsonList jsonCountersList)
        throws ShingleException, JsonBadCastException
    {
        for (final JsonObject jsonCounters : jsonCountersList) {
            loadSchemeCounters(scheme, jsonCounters.asMap());
        }
        checkCounters(scheme);
    }

    default void loadInfo(@Nonnull final Map<S, List<JsonMap>> jsonInfo) throws ShingleException {
        for (final Map.Entry<S, List<JsonMap>> entry : jsonInfo.entrySet()) {
            loadSchemeCounters(entry.getKey(), entry.getValue());
        }
    }

    default void addInfo(@Nonnull final GeneralShingleInfo<S> shingleInfo) throws ShingleException {
        for (final Map.Entry<S, Map<String, List<Object>>> schemeCountersInfo : shingleInfo.entrySet()) {
            for (final Map.Entry<String, List<Object>> counterInfo : schemeCountersInfo.getValue().entrySet()) {
                if (counterInfo.getKey().equals(schemeCountersInfo.getKey().shingleField())
                        || counterInfo.getKey().equals(schemeCountersInfo.getKey().shingleTypeField()))
                {
                    setSchemeCounter(schemeCountersInfo.getKey(), counterInfo.getKey(), counterInfo.getValue());
                } else {
                    addSchemeCounter(schemeCountersInfo.getKey(), counterInfo.getKey(), counterInfo.getValue());
                }
            }
        }
    }

    default void toStringBuilder(@Nonnull final StringBuilder sb) {
        boolean first;
        boolean first2;
        sb.append("{");
        for (Map.Entry<S, Map<String, List<Object>>> entry : entrySet()) {
            final S scheme = entry.getKey();
            sb.append(scheme.toString());
            sb.append(":");
            first = true;
            for (final Map.Entry<String, List<Object>> counter : entry.getValue().entrySet()) {
                if (first) {
                    first = false;
                } else {
                    sb.append(';');
                }
                final String className = scheme.fields().get(counter.getKey());
                sb.append(counter.getKey());
                sb.append("=");
                first2 = true;
                for (final Object value : counter.getValue()) {
                    if (first2) {
                        first2 = false;
                    } else {
                        sb.append(',');
                    }
                    sb.append(castValue(className, value).toString());
                }
            }
        }
        sb.append("}");
    }

    @NoAllocation
    default int expectedStringLength() {
        int len = 2 + size();
        for (Map.Entry<S, Map<String, List<Object>>> entry : entrySet()) {
            final S scheme = entry.getKey();
            len += scheme.toString().length();
            len += (entry.getValue().size() << 1) - 1;
            for (Map.Entry<String, List<Object>> counter : entry.getValue().entrySet()) {
                final String className = scheme.fields().get(counter.getKey());
                len += StringBuilderable.calcExpectedStringLength(counter.getKey()) + counter.getValue().size() - 1;
                for (final Object value : counter.getValue()) {
                    len += castValue(className, value).toString().length();
                }
            }
        }
        return len;
    }
}
