package ru.yandex.client.so.shingler;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Future;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;

import ru.yandex.concurrent.CompletedFuture;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.CallbackFutureBase;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriterBase;

public abstract class GeneralShinglerClient<S extends Scheme>
    extends AbstractShinglerClient<GeneralShinglerClient<S>, GeneralShingles<S>>
{
    public static final String URI = "/api/v1";
    public static final String NAME = "General";

    enum RequestType {
        GET("Get") {
            @Override
            public void writeCounterValues(
                final JsonWriterBase writer,
                final Scheme scheme,
                final Map<String, List<Object>> countersInfo,
                final Set<String> counters)
                throws ShingleException, IOException
            {
                writer.startObject();
                for (final String counter : counters) {
                    writeCounterValuesHelper(writer, scheme, countersInfo, counter, countersInfo.get(counter).get(0));
                }
                writer.endObject();
            }
        },
        PUT("Update") {
            @Override
            public void writeCounterValues(
                final JsonWriterBase writer,
                final Scheme scheme,
                final Map<String, List<Object>> countersInfo,
                final Set<String> counters)
                throws ShingleException, IOException
            {
                final String counter0 = counters.iterator().next();
                final List<String> sortedCounters = counters.stream().sorted().collect(Collectors.toList());
                for (int i = 0; i < countersInfo.get(counter0).size(); i++) {
                    writer.startObject();
                    for (final String counter : sortedCounters) {
                        if (i < countersInfo.get(counter).size()) {
                            final Object value =
                                (scheme.predefinedFields() != null && scheme.predefinedFields().containsKey(counter))
                                    ? scheme.predefinedFields().get(counter) : countersInfo.get(counter).get(i);
                            writeCounterValuesHelper(writer, scheme, countersInfo, counter, value);
                        }
                    }
                    if (scheme.predefinedFields() != null) {
                        for (final String counter
                                : scheme.predefinedFields().keySet().stream().sorted().collect(Collectors.toList())) {
                            if (!counters.contains(counter)) {
                                writeCounterValuesHelper(
                                    writer,
                                    scheme,
                                    countersInfo,
                                    counter,
                                    scheme.predefinedFields().get(counter));
                            }
                        }
                    }
                    writer.endObject();
                }
            }
        };

        private final String action;
        @SuppressWarnings("ImmutableEnumChecker")
        private String shinglerName;

        RequestType(final String action) {
            this.action = action;
        }

        public String action() {
            return action;
        }

        public String shinglerName() {
            return shinglerName;
        }

        public void setShinglerName(final String shinglerName) {
            this.shinglerName = shinglerName;
        }

        static void writeCounterValue(final JsonWriterBase writer, final Object value, final String className)
            throws IOException
        {
            if (className.equals("ULong")) {
                writer.value((long) GeneralShingleInfo.castValue(className, value), false, true);
            } else {
                writer.value(GeneralShingleInfo.castValue(className, value));
            }
        }

        @SuppressWarnings("unchecked")
        void writeCounterValuesHelper(
            final JsonWriterBase writer,
            final Scheme scheme,
            final Map<String, List<Object>> countersInfo,
            final String counter,
            final Object value)
            throws ShingleException, IOException
        {
            final String className = scheme.fields().get(counter);
            if (counter.contains(".")) {
                final String key = counter.substring(0, counter.indexOf("."));
                final String subkey = counter.substring(counter.indexOf(".") + 1);
                writer.key(key);
                writer.startObject();
                {
                    writer.key(subkey);
                    try {
                        writeCounterValue(writer, value, className);
                    } catch (Exception e) {
                        throw new ShingleException(shinglerName() + "ShinglerClient's " + name() + " query failed to "
                            + "write counter " + counter + "=" + value + " (" + className + "): " + e, e);
                    }
                }
                writer.endObject();
            } else if (className.equals("Map")) {
                final Map<String, ?> valueMap = (Map<String, ?>) value;
                if (valueMap != null) {
                    writer.key(counter);
                    writer.startObject();
                    {
                        for (final String key : valueMap.keySet().stream().sorted().collect(Collectors.toList())) {
                            final String type = valueMap.get(key).getClass().getTypeName();
                            try {
                                writer.key(key);
                                writeCounterValue(writer, valueMap.get(key), type);
                            } catch (Exception e) {
                                throw new ShingleException(shinglerName() + "ShinglerClient's " + name() + " query "
                                    + "failed to write counter " + counter + "=" + value + " (" + type + "): "
                                    + e, e);
                            }
                        }
                    }
                    writer.endObject();
                }
            } else {
                if (!countersInfo.containsKey(counter)) {
                    throw new ShingleException(shinglerName() + "ShinglerClient's " + name() + " query failed: "
                        + "undetermined counter '" + counter + "'");
                }
                writer.key(counter);
                if (countersInfo.get(counter).size() != 1 && this == GET) {
                    writer.startArray();
                    try {
                        for (final Object counterValue : countersInfo.get(counter)) {
                            writeCounterValue(writer, counterValue, className);
                        }
                    } catch (Exception e) {
                        throw new ShingleException(shinglerName() + "ShinglerClient's " + name() + " query failed "
                            + "to write counter '" + counter + "' (" + className + "): " + e, e);
                    }
                    writer.endArray();
                } else {
                    try {
                        writeCounterValue(writer, value, className);
                    } catch (Exception e) {
                        throw new ShingleException(shinglerName() + "ShinglerClient's " + name() + " query failed "
                            + "to write counter " + counter + "=" + value + " (" + className + "): " + e, e);
                    }
                }
            }
        }

        public abstract void writeCounterValues(
            final JsonWriterBase writer,
            final Scheme scheme,
            final Map<String, List<Object>> countersInfo,
            final Set<String> counters
        ) throws ShingleException, IOException;
    }

    private static final String TYPE = "type";
    private static final String SCHEME = "scheme";
    private static final String FIELDS = "fields";

    protected GeneralShinglerClient(
        @Nonnull final SharedConnectingIOReactor reactor,
        @Nonnull final ImmutableHttpHostConfig config)
    {
        super(reactor, config);
        for (final RequestType requestType : RequestType.values()) {
            requestType.setShinglerName(name());
        }
    }

    protected GeneralShinglerClient(
        @Nonnull final CloseableHttpAsyncClient client,
        @Nonnull final GeneralShinglerClient<S> sample)
    {
        super(client, sample);
        for (final RequestType requestType : RequestType.values()) {
            requestType.setShinglerName(name());
        }
    }

    @Override
    public abstract GeneralShinglerClient<S> adjust(@Nonnull final CloseableHttpAsyncClient client);

    protected abstract GeneralShingles<S> emptyResult();

    protected abstract GeneralShingles<S> createShingles(@Nonnull final JsonObject jsonObject)
        throws JsonException, ShingleException;

    protected static String name() {
        return NAME;
    }

    protected static <T> boolean equalsSets(final Set<T> set1, final Set<T> set2) {
        if (set1 == null || set2 == null || set1.size() != set2.size()) {
            return false;
        }
        for (final T s : set1) {
            if (!set2.contains(s)) {
                return false;
            }
        }
        return true;
    }

    protected Map<Set<S>, Set<String>> getCountersBySchemes(
        @Nonnull final Map<String, List<Object>> counters,
        @Nonnull final Set<S> schemesFilter)
    {
        final Map<Set<S>, Set<String>> data = new HashMap<>();
        final Map<String, Set<S>> countersSchemes = new HashMap<>();
        for (final S scheme : schemesFilter) {
            for (final String counter : scheme.fields().keySet()) {
                if (counters.containsKey(counter)) {
                    countersSchemes.computeIfAbsent(counter, x -> new HashSet<>()).add(scheme);
                }
            }
        }
        for (final Map.Entry<String, Set<S>> entry : countersSchemes.entrySet()) {
            if (data.isEmpty()) {
                data.computeIfAbsent(entry.getValue(), x -> new HashSet<>()).add(entry.getKey());
            } else {
                boolean newSet = true;
                for (final Map.Entry<Set<S>, Set<String>> dataEntry : data.entrySet()) {
                    if (equalsSets(entry.getValue(), dataEntry.getKey())) {
                        data.computeIfAbsent(dataEntry.getKey(), x -> new HashSet<>()).add(entry.getKey());
                        newSet = false;
                        break;
                    }
                }
                if (newSet) {
                    Objects.requireNonNull(data.computeIfAbsent(entry.getValue(), x -> new HashSet<>()))
                        .add(entry.getKey());
                }
            }
        }
        return data;
    }

    protected Map<Set<String>, Set<S>> getSchemesByCounters(
        @Nonnull final GeneralShingles<S> shingles,
        @Nonnull final Map<S, Map<String, List<Object>>> countersInfo)
    {
        final Map<S, Set<String>> schemeCounters = new HashMap<>();
        final Map<Set<String>, Set<S>> data = new HashMap<>();
        for (final Map.Entry<ShingleType, Map<Long, GeneralShingleInfo<S>>> shingleTypeInfo : shingles.entrySet()) {
            for (final GeneralShingleInfo<S> entry : shingleTypeInfo.getValue().values()) {
                for (Map.Entry<S, Map<String, List<Object>>> shingleInfo : entry.entrySet()) {
                    schemeCounters.put(shingleInfo.getKey(), shingleInfo.getValue().keySet());
                    countersInfo.computeIfAbsent(shingleInfo.getKey(), x -> new HashMap<>());
                    for (final Map.Entry<String, List<Object>> counterInfo : shingleInfo.getValue().entrySet()) {
                        countersInfo.get(shingleInfo.getKey())
                            .computeIfAbsent(counterInfo.getKey(), x -> new ArrayList<>())
                            .addAll(counterInfo.getValue());
                    }
                }
            }
        }
        for (final Map.Entry<S, Set<String>> entry : schemeCounters.entrySet()) {
            if (data.isEmpty()) {
                data.computeIfAbsent(entry.getValue(), x -> new HashSet<>()).add(entry.getKey());
            } else {
                boolean newSet = true;
                for (final Map.Entry<Set<String>, Set<S>> dataEntry : data.entrySet()) {
                    if (equalsSets(entry.getValue(), dataEntry.getKey())) {
                        data.computeIfAbsent(dataEntry.getKey(), x -> new HashSet<>()).add(entry.getKey());
                        newSet = false;
                        break;
                    }
                }
                if (newSet) {
                    Objects.requireNonNull(data.computeIfAbsent(entry.getValue(), x -> new HashSet<>()))
                        .add(entry.getKey());
                }
            }
        }
        return data;
    }

    // assumptions: 1) lists of counters values inside the same scheme have the same length;
    //              2) homonymous counters inside two different schemes have the same type.
    public String getGetQuery(
        @Nonnull final Map<String, List<Object>> counters,
        @Nonnull final Set<S> schemesFilter)
        throws ShingleException
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        if (counters.size() < 1 || schemesFilter.size() < 1) {
            return sbw.toString();
        }
        try (JsonWriterBase writer = (JsonWriterBase) JsonType.NORMAL.create(sbw)) {
            writer.startArray();
            for (final Map.Entry<Set<S>, Set<String>> entry : getCountersBySchemes(counters, schemesFilter).entrySet())
            {
                writer.startObject();
                {
                    writer.key(TYPE);
                    writer.value(RequestType.GET.action());

                    writer.key(SCHEME);
                    writer.startArray();
                    {
                        for (final S scheme : entry.getKey().stream().sorted(Comparator.comparing(S::name))
                                .collect(Collectors.toList())) {
                            writer.value(scheme.name().toLowerCase(Locale.ROOT));
                        }
                    }
                    writer.endArray();

                    writer.key(FIELDS);
                    writer.startArray();
                    {
                        RequestType.GET.writeCounterValues(
                            writer,
                            schemesFilter.iterator().next(),
                            counters,
                            entry.getValue());
                    }
                    writer.endArray();
                }
                writer.endObject();
            }
            writer.endArray();
        } catch (IOException e) {
            /*impossible*/
        }
        return sbw.toString();
    }

    public String getGetQuery(@Nonnull GeneralShingles<S> shingles)
        throws ShingleException
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        if (shingles.size() < 1) {
            return sbw.toString();
        }
        final Set<S> schemes = new HashSet<>();
        final Map<String, List<Object>> counters = new HashMap<>();
        for (final Map.Entry<ShingleType, Map<Long, GeneralShingleInfo<S>>> shingleTypeInfo : shingles.entrySet()) {
            for (final GeneralShingleInfo<S> entry : shingleTypeInfo.getValue().values()) {
                for (Map.Entry<S, Map<String, List<Object>>> shingleInfo : entry.entrySet()) {
                    if (equalsSets(shingleInfo.getKey().keyFields(), shingleInfo.getValue().keySet())) {
                        schemes.add(shingleInfo.getKey());
                    }
                    for (final Map.Entry<String, List<Object>> counterInfo : shingleInfo.getValue().entrySet()) {
                        counters.computeIfAbsent(counterInfo.getKey(), x -> new ArrayList<>());
                        for (final Object value : counterInfo.getValue()) {
                            if (!counters.get(counterInfo.getKey()).contains(value)) {
                                counters.get(counterInfo.getKey()).add(value);
                            }
                        }
                    }
                }
            }
        }
        return getGetQuery(counters, schemes);
    }

    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<? extends GeneralShingles<S>> getShingles(
        @Nonnull final Map<String, List<Object>> counters,
        @Nonnull final Set<S> schemesFilter,
        @Nonnull final Supplier<? extends HttpClientContext> contextGenerator,
        @Nonnull final FutureCallback<? super GeneralShingles<S>> callback)
        throws ShinglerClientException, ShingleException
    {
        if (counters.isEmpty() || schemesFilter.isEmpty()) {
            callback.completed(null);
            return new CompletedFuture<>(emptyResult());
        }
        return getShingles(getGetQuery(counters, schemesFilter), contextGenerator, callback);
    }

    @Override
    @Nonnull
    @SuppressWarnings({"FutureReturnValueIgnored", "unchecked"})
    public Future<? extends GeneralShingles<S>> getShingles(
        @Nonnull final AbstractShinglesMap<?> shingles,
        @Nonnull final Supplier<? extends HttpClientContext> contextGenerator,
        @Nonnull final FutureCallback<? super GeneralShingles<S>> callback)
        throws ShinglerClientException, ShingleException
    {
        try {
            if (shingles.isEmpty()) {
                callback.completed(((GeneralShingles<S>) shingles.getEmpty()));
                return new CompletedFuture<>(emptyResult());
            }
            return getShingles(getGetQuery((GeneralShingles<S>) shingles), contextGenerator, callback);
        } catch (Exception e) {
            callback.completed(null);
            return new CompletedFuture<>(emptyResult());
        }
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<? extends GeneralShingles<S>> getShingles(
        @Nonnull final String query,
        @Nonnull final Supplier<? extends HttpClientContext> contextGenerator,
        @Nonnull final FutureCallback<? super GeneralShingles<S>> callback)
        throws ShinglerClientException, ShingleException
    {
        GetQueryCallbackProxy<S> callbackProxy = new GetQueryCallbackProxy<>(callback);
        execute(
            host,
            new BasicAsyncRequestProducerGenerator(
                URI,
                query,
                ContentType.APPLICATION_JSON),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            contextGenerator,
            callbackProxy);
        return callbackProxy;
    }

    protected void writePutQueryHelper(
        @Nonnull final JsonWriterBase writer,
        @Nonnull final Map<String, List<Object>> countersInfo,
        @Nonnull final Set<S> schemes,
        @Nonnull final Set<String> counters,
        @Nonnull S schemeBasic)
        throws ShingleException, IOException
    {
        writer.startObject();
        {
            writer.key(TYPE);
            writer.value(RequestType.PUT.action());

            writer.key(SCHEME);
            writer.startArray();
            {
                for (final S scheme : schemes.stream().sorted(Comparator.comparing(S::name))
                        .collect(Collectors.toList())) {
                    writer.value(scheme.name().toLowerCase(Locale.ROOT));
                }
            }
            writer.endArray();

            writer.key(FIELDS);
            writer.startArray();
            {
                RequestType.PUT.writeCounterValues(writer, schemeBasic, countersInfo, counters);
            }
            writer.endArray();
        }
        writer.endObject();
    }

    protected void writePutQueryHelper(
        @Nonnull final JsonWriterBase writer,
        @Nonnull final Map<String, List<Object>> countersInfo,
        @Nonnull final S scheme,
        @Nonnull final Set<String> counters)
        throws ShingleException, IOException
    {
        writer.startObject();
        {
            writer.key(TYPE);
            writer.value(RequestType.PUT.action());

            writer.key(SCHEME);
            writer.startArray();
            {
                writer.value(scheme.name().toLowerCase(Locale.ROOT));
            }
            writer.endArray();

            writer.key(FIELDS);
            writer.startArray();
            {
                RequestType.PUT.writeCounterValues(writer, scheme, countersInfo, counters);
            }
            writer.endArray();
        }
        writer.endObject();
    }

    protected void writePutQueryHelper1(
        @Nonnull final JsonWriterBase writer,
        @Nonnull final Map<String, List<Object>> countersInfo,
        @Nonnull final Set<S> schemes,
        @Nonnull final Set<String> counters,
        boolean schemesGroupping)
        throws ShingleException, IOException
    {
        if (schemesGroupping) {
            final S scheme = schemes.iterator().next();
            writePutQueryHelper(writer, countersInfo, schemes, counters, scheme);
        } else {
            for (final S scheme : schemes.stream().sorted(Comparator.comparing(S::name))
                    .collect(Collectors.toList())) {
                writePutQueryHelper(writer, countersInfo, scheme, counters);
            }
        }
    }

    protected void writePutQueryHelper2(
        @Nonnull final JsonWriterBase writer,
        @Nonnull final Map<S, Map<String, List<Object>>> countersInfo,
        @Nonnull final Set<S> schemes,
        @Nonnull final Set<String> counters,
        boolean schemesGroupping)
        throws ShingleException, IOException
    {
        if (schemesGroupping) {
            final S scheme = schemes.iterator().next();
            writePutQueryHelper(writer, countersInfo.get(scheme), schemes, counters, scheme);
        } else {
            for (final S scheme : schemes.stream().sorted(Comparator.comparing(S::name))
                    .collect(Collectors.toList())) {
                writePutQueryHelper(writer, countersInfo.get(scheme), scheme, counters);
            }
        }
    }

    // assumptions: 1) lists of counters values inside the same scheme have the same length;
    //              2) homonymous counters inside two different schemes have the same type.
    public String getPutQuery(
        @Nonnull final Map<String, List<Object>> counters,
        @Nonnull final Set<S> schemesFilter,
        boolean schemesGroupping)
        throws ShingleException
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        if (counters.size() < 1 || schemesFilter.size() < 1) {
            return sbw.toString();
        }
        try (JsonWriterBase writer = (JsonWriterBase) JsonType.NORMAL.create(sbw)) {
            writer.startArray();
            for (final Map.Entry<Set<S>, Set<String>> entry : getCountersBySchemes(counters, schemesFilter).entrySet())
            {
                writePutQueryHelper1(writer, counters, entry.getKey(), entry.getValue(), schemesGroupping);
            }
            writer.endArray();
        } catch (IOException e) {
            /*impossible*/
        }
        return sbw.toString();
    }

    // assumptions: same counter in different schemes has same value (because of the identity of its meaning between
    // different schemes)
    public String getPutQuery(@Nonnull final GeneralShingles<S> shingles, boolean schemesGroupping)
        throws ShingleException
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        if (shingles.size() < 1) {
            return sbw.toString();
        }
        final Map<S, Map<String, List<Object>>> countersInfo = new HashMap<>();
        final Map<Set<String>, Set<S>> data = getSchemesByCounters(shingles, countersInfo);
        try (JsonWriterBase writer = (JsonWriterBase) JsonType.NORMAL.create(sbw)) {
            writer.startArray();
            for (final Map.Entry<Set<String>, Set<S>> entry
                    : data.entrySet().stream().sorted(Comparator.comparing(
                        s -> s.getValue().stream().map(S::name).sorted().collect(Collectors.joining(","))))
                        .collect(Collectors.toList()))
            {
                writePutQueryHelper2(writer, countersInfo, entry.getValue(), entry.getKey(), schemesGroupping);
            }
            writer.endArray();
        } catch (IOException e) {
            /*impossible*/
        }
        return sbw.toString();
    }

    public String getPutQuery(@Nonnull final GeneralShingles<S> shingles) throws ShingleException {
        return getPutQuery(shingles, true);
    }

    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(
        @Nonnull final Map<String, List<Object>> counters,
        @Nonnull final Set<S> schemesFilter,
        @Nonnull final FutureCallback<String> callback)
        throws ShingleException
    {
        if (counters.isEmpty() || schemesFilter.isEmpty()) {
            callback.completed(OK);
            return SUCCESS_PUT_RESULT;
        }
        return putShingles(getPutQuery(counters, schemesFilter, true), callback);
    }

    @Override
    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(
        @Nonnull final GeneralShingles<S> shingles,
        @Nonnull final Supplier<? extends HttpClientContext> contextGenerator,
        @Nonnull final FutureCallback<String> callback)
        throws ShinglerClientException, ShingleException
    {
        if (shingles.isEmpty()) {
            callback.completed(OK);
            return SUCCESS_PUT_RESULT;
        }
        return putShingles(getPutQuery(shingles), contextGenerator, callback);
    }

    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(@Nonnull final String query)
    {
        return putShingles(query, httpClientContextGenerator(), StringFutureCallback.INSTANCE);
    }

    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(
        @Nonnull final String query,
        @Nonnull final FutureCallback<String> callback)
    {
        return putShingles(query, httpClientContextGenerator(), callback);
    }

    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(
        @Nonnull final String query,
        @Nonnull final Supplier<? extends HttpClientContext> contextGenerator,
        @Nonnull final FutureCallback<String> callback)
    {
        PutQueryCallbackProxy<S> callbackProxy = new PutQueryCallbackProxy<>(callback, name());
        execute(
            host,
            new BasicAsyncRequestProducerGenerator(URI, query, ContentType.APPLICATION_JSON),
            EmptyAsyncConsumerFactory.OK,
            contextGenerator,
            callbackProxy);
        return callbackProxy;
    }

    private class GetQueryCallbackProxy<T extends Scheme>
        extends CallbackFutureBase<GeneralShingles<T>, JsonObject>
    {
        GetQueryCallbackProxy(@Nonnull final FutureCallback<? super GeneralShingles<T>> callback) {
            super(callback);
        }

        @SuppressWarnings("unchecked")
        @Override
        @Nonnull
        protected GeneralShingles<T> convertResult(@Nonnull final JsonObject jsonObject) {
            try {
                return (GeneralShingles<T>) createShingles(jsonObject);
            } catch (JsonException | ShingleException e) {
                System.err.println("GetQueryCallbackProxy.convertResult failed: " + e);
                failed(e);
                return (GeneralShingles<T>) emptyResult();
            }
        }

        @Override
        protected void onComplete(final GeneralShingles<T> result) {
            callback.completed(result);
        }

        @Override
        protected void onFailure(final Exception e) {
            //System.err.println("GeneralShinglerClient.GetQueryCallbackProxy failed: " + e.toString());
            callback.failed(e);
        }
    }

    private static class PutQueryCallbackProxy<T extends Scheme> extends CallbackFutureBase<String, Void>
    {
        private final String name;

        PutQueryCallbackProxy(@Nonnull final FutureCallback<String> callback, final String name) {
            super(callback);
            this.name = name;
        }

        @Override
        @Nonnull
        protected String convertResult(@Nonnull final Void unused) {
            return OK;
        }

        @Override
        protected void onFailure(final Exception e) {
            System.err.println("GeneralShinglerClient.PutQueryCallbackProxy for shingler " + name + " failed: "
                + e.toString());
            //callback.failed(e);
            callback.completed(null);
        }
    }
}
