package ru.yandex.client.so.shingler;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;

import com.google.errorprone.annotations.NoAllocation;

import ru.yandex.function.StringBuilderable;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;

public abstract class GeneralShingles<S extends Scheme> extends AbstractShinglesMap<Map<Long, GeneralShingleInfo<S>>>
{
    public static final ShingleType SHINGLE = ShingleType.UNKNOWN;

    private static final String NAME = "General";
    private static final String TYPE = "type";
    private static final String GET = "Get";
    private static final String SCHEME = "scheme";
    private static final String FIELDS = "fields";
    private static final String FIND = "find";
    private static final long serialVersionUID = 0L;
    private static final String PREFIX = "Shingles(";
    private static final int PREFIX_LENGTH = PREFIX.length();

    protected GeneralShingles() {
    }

    @Override
    public abstract GeneralShingles<S> getEmpty();

    protected abstract GeneralShingleInfo<S> createShingleInfo();

    protected abstract GeneralShingleInfo<S> createShingleInfo(final Map<S, List<JsonMap>> jsonInfo)
        throws ShingleException;

    protected abstract GeneralShingleInfo<S> createShingleInfo(final S scheme, final List<JsonMap> jsonCountersList)
        throws ShingleException;

    protected abstract GeneralShingleInfo<S> createShingleInfo(final S scheme, final JsonList jsonCountersList)
        throws ShingleException, JsonBadCastException;

    protected abstract GeneralShingleInfo<S> createShingleInfo(final S scheme, final JsonMap jsonCounters)
        throws ShingleException, JsonException;

    protected abstract GeneralShingleInfo<S> createShingleInfo(final S scheme, final Map<String, Object> counters)
        throws ShingleException;

    protected abstract GeneralShingleInfo<S> createShingleInfo(final Set<S> schemes, final Map<String, Object> counters)
        throws ShingleException;

    protected abstract GeneralShingleInfo<S> createShingleInfo(final S scheme, final long shingle)
        throws ShingleException;

    protected abstract GeneralShingleInfo<S> createShingleInfo(final GeneralShingleInfo<S> shingleInfo)
        throws ShingleException;

    protected abstract S schemeFromName(final String schemeName);

    @SuppressWarnings("unchecked")
    @Override
    public void addAll(final AbstractShinglesMap<?> data) throws ShingleException
    {
        for (final Map.Entry<ShingleType, Map<Long, GeneralShingleInfo<S>>> entry
                : ((AbstractShinglesMap<Map<Long, GeneralShingleInfo<S>>>)data).entrySet())
        {
            for (final Map.Entry<Long, GeneralShingleInfo<S>> shingleInfo : entry.getValue().entrySet()) {
                mergeShingleInfo(entry.getKey(), shingleInfo.getKey(), shingleInfo.getValue());
            }
        }
    }

    public String name() {
        return NAME;
    }

    protected GeneralShingles(@Nonnull final JsonObject jsonResponse)
        throws JsonException, ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        parseResponse(jsonResponse);
    }

    protected GeneralShingles(@Nonnull final Map<S, List<JsonMap>> jsonInfo)
        throws JsonException, ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        for (final S scheme : jsonInfo.keySet()) {
            parseResponse(scheme, jsonInfo.get(scheme));
        }
    }

    protected GeneralShingles(
        @Nonnull final Set<S> schemes,
        @Nonnull final Map<String, Object> counters)
        throws ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        loadCounters(schemes, counters);
    }

    protected GeneralShingles(
        @Nonnull final Set<S> schemes,
        @Nonnull final List<Map<String, Object>> countersList)
        throws ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        loadCounters(schemes, countersList);
    }

    protected GeneralShingles(
        @Nonnull final S scheme,
        @Nonnull final List<JsonMap> jsonCountersList)
        throws JsonException, ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        parseResponse(scheme, jsonCountersList);
    }

    protected GeneralShingles(
        @Nonnull final S scheme,
        @Nonnull final JsonList jsonCountersList)
        throws JsonException, ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        parseResponse(scheme, jsonCountersList);
    }

    protected GeneralShingles(
        @Nonnull final S scheme,
        @Nonnull final JsonMap jsonCounters)
        throws JsonException, ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        parseResponse(scheme, jsonCounters);
    }

    protected GeneralShingles(
        @Nonnull final JsonList jsonCountersList,
        @Nonnull final Set<S> schemes)
        throws JsonException, ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        for (final S scheme : schemes) {
            parseResponse(scheme, jsonCountersList);
        }
    }

    protected GeneralShingles(
        @Nonnull final Shingles shingles,
        @Nonnull final Set<S> schemes)
        throws ShingleException
    {
        parseResponse(shingles, schemes);
    }

    protected GeneralShingles(@Nonnull final GeneralShingles<S> data) throws ShingleException
    {
        super();
        for (final Map.Entry<ShingleType, Map<Long, GeneralShingleInfo<S>>> entry : data.entrySet()) {
            put(entry.getKey(), new HashMap<>());
            for (final Map.Entry<Long, GeneralShingleInfo<S>> shingleInfo : entry.getValue().entrySet()) {
                get(entry.getKey()).put(shingleInfo.getKey(), createShingleInfo(shingleInfo.getValue()));
            }
        }
    }

    protected void parseResponse(final Shingles shingles, final Set<S> schemes) throws ShingleException
    {
        if (shingles != null && schemes != null) {
            for (final Map.Entry<ShingleType, Map<Long, Shingle>> entry : shingles.entrySet()) {
                Map<Long, GeneralShingleInfo<S>> shingleTypeInfo =
                    computeIfAbsent(entry.getKey(), x -> new HashMap<>());
                for (final Long shingle : entry.getValue().keySet()) {
                    shingleTypeInfo.computeIfAbsent(shingle, x -> createShingleInfo());
                    for (final S scheme : schemes) {
                        if (scheme.shingleField() != null) {
                            get(entry.getKey()).get(shingle).setSchemeCounter(scheme, scheme.shingleField(), shingle);
                        }
                        if (scheme.shingleTypeField() != null) {
                            get(entry.getKey()).get(shingle).setSchemeCounter(
                                scheme,
                                scheme.shingleTypeField(),
                                (byte) entry.getKey().id());
                        }
                    }
                }
            }
        }
    }

    protected void parseResponse(final S scheme, final JsonMap jsonCounters)
        throws JsonException, ShingleException
    {
        if (jsonCounters == null) {
            System.err.println(name() + "Shingles.parseResponse: input JsonMap is null!");
            return;
        }
        long shingle = 0L;
        if (jsonCounters.containsKey(scheme.shingleField())) {
            if (jsonCounters.get(scheme.shingleField()).type() == JsonObject.Type.STRING) {
                if (scheme.fields().get(scheme.shingleField()).equals("ULong")) {
                    shingle = jsonCounters.getString(scheme.shingleField()).startsWith("-")
                        ? Long.parseLong(jsonCounters.getString(scheme.shingleField()))
                        : Long.parseUnsignedLong(jsonCounters.getString(scheme.shingleField()));
                } else {
                    shingle = jsonCounters.getLong(scheme.shingleField());
                }
            } else {
                shingle = jsonCounters.getLong(scheme.shingleField());
            }
        }
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        if (get(SHINGLE).containsKey(shingle)) {
            mergeShingleInfo(shingle, scheme, jsonCounters);
        } else {
            get(SHINGLE).put(shingle, createShingleInfo(scheme, jsonCounters));
        }
    }

    protected void parseResponse(final S scheme, final JsonList jsonCountersList)
        throws JsonException, ShingleException
    {
        for (final JsonObject jsonInfoForScheme : jsonCountersList) {
            parseResponse(scheme, jsonInfoForScheme.asMap());
        }
    }

    protected void parseResponse(final Set<S> schemes, final JsonList jsonCountersList)
        throws JsonException, ShingleException
    {
        if (schemes != null) {
            for (final S scheme : schemes) {
                parseResponse(scheme, jsonCountersList);
            }
        }
    }

    protected void parseResponse(final S scheme, final List<JsonMap> jsonCountersList)
        throws JsonException, ShingleException
    {
        if (jsonCountersList != null) {
            for (final JsonMap jsonCounters : jsonCountersList) {
                parseResponse(scheme, jsonCounters);
            }
        }
    }

    protected void parseResponse(@Nonnull final JsonObject jsonResponse)
        throws JsonException, ShingleException
    {
        for (final JsonObject answer : jsonResponse.asList()) {
            final String schemeName = answer.get(SCHEME).asString();
            if (schemeName == null) {
                throw new ShingleException(name() + "Shingles: unknown scheme in response");
            }
            final S scheme = schemeFromName(schemeName.toUpperCase());
            if (scheme == null) {
                throw new ShingleException(name() + "Shingles: unknown scheme: '" + schemeName + "' in response");
            }
            parseResponse(scheme, answer.get(FIND).asList());
        }
    }

    public void loadUpdateRequest(@Nonnull final JsonObject jsonRequest)
        throws JsonException, ShingleException
    {
        for (final JsonObject request : jsonRequest.asList()) {
            if (request instanceof JsonMap) {
                final JsonMap data = request.asMap();
                if (data.containsKey(TYPE) && data.getString(TYPE).equals(GET)) {
                    continue;
                }
                final JsonList schemes = data.get(SCHEME).asList();
                for (final JsonObject schemeObj : schemes) {
                    final S scheme = schemeFromName(schemeObj.asString().toUpperCase());
                    if (scheme == null) {
                        throw new ShingleException(name() + "Shingles: unknown scheme: '" + schemeObj.asString()
                            + "' in response");
                    }
                    parseResponse(scheme, data.get(FIELDS).asList());
                }
            }
        }
    }

    @SuppressWarnings("unchecked")
    public void loadCounters(@Nonnull Set<S> schemes, @Nonnull final Map<String, Object> counters)
        throws ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        if (schemes.size() < 1) {
            return;
        }
        String shingleField = schemes.iterator().next().shingleField();
        if (counters.containsKey(shingleField)) {
            final GeneralShingleInfo<S> shingleInfo = createShingleInfo(schemes, counters);
            if (counters.get(shingleField) instanceof List) {
                get(SHINGLE).put((Long) ((List<Object>) counters.get(shingleField)).get(0), shingleInfo);
            } else {
                get(SHINGLE).put((Long) counters.get(shingleField), shingleInfo);
            }
        } else {
            for (final S scheme : schemes) {
                for (Map.Entry<Long, GeneralShingleInfo<S>> shingleInfo : get(SHINGLE).entrySet()) {
                    if (shingleInfo.getValue().containsKey(scheme)) {
                        Map<String, List<Object>> countersInfo = shingleInfo.getValue().get(scheme);
                        for (Map.Entry<String, Object> entry : counters.entrySet()) {
                            countersInfo.computeIfAbsent(entry.getKey(), x -> new ArrayList<>())
                                .add(entry.getValue());
                        }
                        if (scheme.keyFields().contains(scheme.shingleField())
                                && !countersInfo.containsKey(scheme.shingleField()))
                        {
                            countersInfo.computeIfAbsent(scheme.shingleField(), x -> new ArrayList<>())
                                .add(shingleInfo.getKey());
                        }
                    }
                }
            }
        }
    }

    public void loadCounters(@Nonnull Set<S> schemes, @Nonnull final List<Map<String, Object>> countersList)
        throws ShingleException
    {
        for (final Map<String, Object> counters : countersList) {
            loadCounters(schemes, counters);
        }
    }

    protected void mergeShingleInfo(final long shingle, final S scheme, final ShingleType shingleType)
        throws ShingleException
    {
        computeIfAbsent(shingleType, x -> new HashMap<>());
        if (get(shingleType).containsKey(shingle)) {
            if (scheme.shingleField() != null) {
                get(shingleType).get(shingle).setSchemeCounter(scheme, scheme.shingleField(), shingle);
            }
        }
    }

    protected void mergeShingleInfo(final long shingle, final S scheme, final JsonMap jsonCounters)
        throws ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        if (get(SHINGLE).containsKey(shingle)) {
            get(SHINGLE).get(shingle).loadSchemeCounters(scheme, jsonCounters);
        }
    }

    protected void mergeShingleInfo(final long shingle, final S scheme, final JsonList jsonCountersList)
        throws ShingleException, JsonBadCastException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        if (get(SHINGLE).containsKey(shingle)) {
            get(SHINGLE).get(shingle).loadSchemeCounters(scheme, jsonCountersList);
        }
    }

    protected void mergeShingleInfo(final long shingle, final S scheme, final List<JsonMap> jsonCountersList)
        throws ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        if (get(SHINGLE).containsKey(shingle)) {
            get(SHINGLE).get(shingle).loadSchemeCounters(scheme, jsonCountersList);
        }
    }

    protected void mergeShingleInfo(final long shingle, final Map<S, List<JsonMap>> jsonInfo)
        throws ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        if (get(SHINGLE).containsKey(shingle)) {
            get(SHINGLE).get(shingle).loadInfo(jsonInfo);
        }
    }

    protected void mergeShingleInfo(final long shingle, final GeneralShingleInfo<S> shingleInfo)
        throws ShingleException
    {
        computeIfAbsent(SHINGLE, x -> new HashMap<>());
        if (get(SHINGLE).containsKey(shingle)) {
            get(SHINGLE).get(shingle).addInfo(shingleInfo);
        }
    }

    protected void mergeShingleInfo(
        final ShingleType shingleType,
        final long shingle,
        final GeneralShingleInfo<S> shingleInfo)
        throws ShingleException
    {
        computeIfAbsent(shingleType, x -> new HashMap<>());
        if (get(shingleType).containsKey(shingle)) {
            get(shingleType).get(shingle).addInfo(shingleInfo);
        } else {
            get(shingleType).put(shingle, shingleInfo.copy());
        }
    }

    protected void mergeShingleInfo(
        final ShingleType shingleType,
        final long shingle,
        final S scheme,
        final Map<String, Object> counters)
        throws ShingleException
    {
        computeIfAbsent(shingleType, x -> new HashMap<>());
        if (get(shingleType).containsKey(shingle)) {
            for (final Map.Entry<String, Object> counterInfo : counters.entrySet()) {
                get(shingleType).get(shingle).setSchemeCounter(scheme, counterInfo.getKey(), counterInfo.getValue());
            }
        }
    }

    protected void mergeShingleInfo(
        final ShingleType shingleType,
        final long shingle,
        final Map<S, Map<String, Object>> info)
        throws ShingleException
    {
        computeIfAbsent(shingleType, x -> new HashMap<>());
        if (get(shingleType).containsKey(shingle)) {
            for (final Map.Entry<S, Map<String, Object>> entry : info.entrySet()) {
                for (final Map.Entry<String, Object> counterInfo : entry.getValue().entrySet()) {
                    get(shingleType).get(shingle)
                        .setSchemeCounter(entry.getKey(), counterInfo.getKey(), counterInfo.getValue());
                }
            }
        }
    }

    public void addShingleInfo(
        final ShingleType shingleType,
        final long shingle,
        final S scheme,
        final Map<String, Object> counters)
        throws ShingleException
    {
        computeIfAbsent(shingleType, x -> new HashMap<>());
        if (get(shingleType).containsKey(shingle)) {
            for (final Map.Entry<String, Object> counterInfo : counters.entrySet()) {
                get(shingleType).get(shingle).setSchemeCounter(scheme, counterInfo.getKey(), counterInfo.getValue());
            }
        } else {
            get(shingleType).put(shingle, createShingleInfo(scheme, counters));
        }
    }

    @NoAllocation
    @Nonnull
    public Map<ShingleType, Map<Long, GeneralShingleInfo<S>>> getResults() {
        return this;
    }

    @Override
    public void toStringBuilder(@Nonnull final StringBuilder sb) {
        sb.append(name()).append(PREFIX).append('{');
        boolean first = true;
        boolean first2;
        for (Map.Entry<ShingleType, Map<Long, GeneralShingleInfo<S>>> entry : entrySet()) {
            if (first) {
                first = false;
            } else {
                sb.append(',');
            }
            sb.append(entry.getKey().name());
            sb.append("=");
            first2 = true;
            for (final Map.Entry<Long, GeneralShingleInfo<S>> shingleInfo: entry.getValue().entrySet()) {
                if (first2) {
                    first2 = false;
                } else {
                    sb.append(',');
                }
                sb.append(shingleInfo.getKey().toString());
                sb.append(":[");
                sb.append(shingleInfo.getValue());
                sb.append(']');
            }
        }
        sb.append("})");
    }

    @NoAllocation
    @Override
    public int expectedStringLength() {
        int len = name().length() + PREFIX_LENGTH + 3 + (size() << 1) - 1;
        for (Map.Entry<ShingleType, Map<Long, GeneralShingleInfo<S>>> entry : entrySet()) {
            len += (entry.getValue().size() << 2) - 1 + entry.getKey().name().length();
            for (Map.Entry<Long, GeneralShingleInfo<S>> shingleInfo : entry.getValue().entrySet()) {
                len += StringBuilderable.calcExpectedStringLength(shingleInfo.getKey());
                len += shingleInfo.getValue().expectedStringLength();
            }
        }
        return len;
    }

    public GeneralShingles<S> setSpam(final boolean isSpam) {
        List<Object> one = new ArrayList<>();
        one.add(1L);
        for (Map.Entry<ShingleType, Map<Long, GeneralShingleInfo<S>>> entry : entrySet()) {
            for (Map.Entry<Long, GeneralShingleInfo<S>> shingleTypeInfo : get(entry.getKey()).entrySet()) {
                for (Map.Entry<S, Map<String, List<Object>>> shingleInfo : shingleTypeInfo.getValue().entrySet()) {
                    shingleInfo.getValue().put(isSpam ? "spam" : "ham", one);
                }
            }
        }
        return this;
    }
}
