package ru.yandex.client.so.shingler;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.errorprone.annotations.NoAllocation;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.function.GenericFunction;
import ru.yandex.function.StringBuilderable;

public class MassShingleInfo extends AbstractStringBuilderable {
    public static final String SHIDENT = "<SH: ";
    public static final int SHIDENT_LENGTH = SHIDENT.length();

    private static final String PREFIX = "MassShingleInfo(";
    private static final int PREFIX_LENGTH = PREFIX.length();

    @Nonnull
    private final String shingleHash;
    private final int shingleType;
    private final String shingleUniqHash;
    private final int shingleUniqType;

    @Nullable
    private final MassShingleStats todayStats;
    @Nullable
    private final MassShingleStats yesterdayStats;
    @Nullable
    private final Double ageCoefficient;
    @Nullable
    private final Long lastMidnight;

    public MassShingleInfo(@Nonnull final String value) throws ShingleException {
        shingleUniqHash = null;
        shingleUniqType = 0;
        int shingleEnd = value.indexOf(' ', SHIDENT_LENGTH);
        if (shingleEnd == -1) {
            throw new ShingleException("Shingle not found");
        }
        String shingle = value.substring(SHIDENT_LENGTH, shingleEnd);
        int dash = shingle.indexOf('-');
        if (dash == -1) {
            throw new ShingleException("No dash found in shingle '" + shingle + '\'');
        }
        shingleHash = shingle.substring(0, dash);
        try {
            shingleType = Integer.parseInt(shingle.substring(dash + 1));
        } catch (RuntimeException e) {
            throw new ShingleException("Failed to parse shingle type from '" + shingle + '\'');
        }
        todayStats = parseIdent(value, "ti='", MassShingleStats::new);
        yesterdayStats = parseIdent(value, "yi='", MassShingleStats::new);
        ageCoefficient = parseIdent(value, "ak='", Double::valueOf);
        lastMidnight = parseIdent(value, "lmt='", Long::valueOf);
    }

    public MassShingleInfo(@Nonnull final Shingle shingle, @Nonnull final ShingleType shingleType) {
        this(shingle, shingleType, null, ShingleType.fromId(shingleType.idUniq()));
    }

    public MassShingleInfo(
        @Nonnull final Shingle shingle,
        @Nonnull final ShingleType shingleType,
        final Shingle shingleUniq,
        @Nonnull final ShingleType shingleUniqType)
    {
        this.shingleType = shingleType.id();
        shingleHash = Long.toHexString(shingle.shingleHash());
        this.shingleUniqType = shingleUniqType.id();
        shingleUniqHash = shingleUniq == null ? null : Long.toHexString(shingleUniq.shingleHash());
        todayStats = new MassShingleStats();
        yesterdayStats = new MassShingleStats();
        ageCoefficient = 0.0;
        lastMidnight = 0L;
    }

    @Nullable
    private static String ident(
        @Nonnull final String value,
        @Nonnull final String ident)
    {
        int start = value.indexOf(ident);
        if (start != -1) {
            start += ident.length();
            int end = value.indexOf('\'', start);
            if (end > start) {
                return value.substring(start, end);
            }
        }
        return null;
    }

    @Nullable
    private static <T> T parseIdent(
        @Nonnull final String value,
        @Nonnull final String ident,
        @Nonnull final GenericFunction<String, T, ShingleException> parser)
        throws ShingleException
    {
        String identValue = ident(value, ident);
        if (identValue == null) {
            return null;
        } else {
            try {
                return parser.apply(identValue);
            } catch (RuntimeException e) {
                throw new ShingleException(
                    "Failed to parse ident " + ident + " with value '" + identValue + '\'',
                    e);
            } catch (ShingleException e) {
                e.addSuppressed(
                    new ShingleException(
                        "While parsing ident " + ident + " with value '" + identValue + '\''));
                throw e;
            }
        }
    }

    @NoAllocation
    @Nonnull
    public String shingleHash() {
        return shingleHash;
    }

    @NoAllocation
    public int shingleType() {
        return shingleType;
    }

    @NoAllocation
    @Nullable
    public MassShingleStats todayStats() {
        return todayStats;
    }

    @NoAllocation
    @Nullable
    public MassShingleStats yesterdayStats() {
        return yesterdayStats;
    }

    @NoAllocation
    @Nullable
    public Double ageCoefficient() {
        return ageCoefficient;
    }

    @NoAllocation
    @Nullable
    public Long lastMidnight() {
        return lastMidnight;
    }

    @NoAllocation
    public String bindedUniqueShingle() {
        return shingleUniqHash;
    }

    @NoAllocation
    public int bindedUniqueShingleType() {
        return shingleUniqType;
    }

    @NoAllocation
    @Override
    public int expectedStringLength() {
        return PREFIX_LENGTH
            + 6
            + shingleHash.length()
            + StringBuilderable.calcExpectedStringLength((long) shingleType)
            + StringBuilderable.calcExpectedStringLength(todayStats)
            + StringBuilderable.calcExpectedStringLength(yesterdayStats)
            + StringBuilderable.calcExpectedStringLength(ageCoefficient)
            + StringBuilderable.calcExpectedStringLength(lastMidnight)
            + (shingleUniqType > 0 ?
                (3 + (shingleUniqHash == null ? 4 : shingleUniqHash.length())
                    + StringBuilderable.calcExpectedStringLength((long) shingleUniqType))
                : 0);
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append(PREFIX);
        sb.append(shingleHash);
        sb.append('-');
        sb.append(shingleType);
        sb.append(',');
        StringBuilderable.toStringBuilder(sb, todayStats);
        sb.append(',');
        StringBuilderable.toStringBuilder(sb, yesterdayStats);
        sb.append(',');
        sb.append(ageCoefficient);
        sb.append(',');
        sb.append(lastMidnight);
        if (shingleUniqType > 0) {
            sb.append('<');
            sb.append(shingleUniqHash);
            sb.append('-');
            sb.append(shingleUniqType);
            sb.append('>');
        }
        sb.append(')');
    }
}

