package ru.yandex.client.so.shingler;

import java.util.regex.Pattern;

import javax.annotation.Nonnull;

import com.google.errorprone.annotations.NoAllocation;

import ru.yandex.function.AbstractStringBuilderable;

public class MassShingleStats extends AbstractStringBuilderable {
    private static final Pattern SPLIT_PATTERN = Pattern.compile(",");
    private static final String PREFIX = "MassShingleStats(";
    private static final int PREFIX_LENGTH = PREFIX.length();
    private static final int EXPECTED_STRING_LENGTH = 38;

    private long ham;
    private long malic;
    private long spam;
    private long personalHam;
    private long personalSpam;
    private double weight;

    public MassShingleStats() {
        ham = 0L;
        malic = 0L;
        spam = 0L;
        personalHam = 0L;
        personalSpam = 0L;
        weight = 0.0;
    }

    @SuppressWarnings("StringSplitter")
    public MassShingleStats(@Nonnull final String value) throws ShingleException {
        String[] parts = SPLIT_PATTERN.split(value);
        if (parts.length != 6) {
            throw new ShingleException("Invalid number of components");
        }
        ham = parse(parts[0], "ham");
        malic = parse(parts[1], "malic");
        spam = parse(parts[2], "spam");
        personalHam = parse(parts[3], "personal ham");
        personalSpam = parse(parts[4], "personal spam");
        try {
            weight = Double.parseDouble(parts[5]);
        } catch (RuntimeException e) {
            throw new ShingleException(
                "Failed to parse weight from <" + parts[5] + '>',
                e);
        }
    }

    private static long parse(
        @Nonnull final String value,
        @Nonnull final String name)
        throws ShingleException
    {
        try {
            return Long.parseLong(value);
        } catch (RuntimeException e) {
            throw new ShingleException(
                "Failed to parse " + name + " from <" + value + '>',
                e);
        }
    }

    @NoAllocation
    public long ham() {
        return ham;
    }

    @NoAllocation
    public void setHam(final long ham) {
        this.ham = ham;
    }

    @NoAllocation
    public long malic() {
        return malic;
    }

    @NoAllocation
    public void setMalic(final long malic) {
        this.malic = malic;
    }

    @NoAllocation
    public long spam() {
        return spam;
    }

    @NoAllocation
    public void setSpam(final long spam) {
        this.spam = spam;
    }

    @NoAllocation
    public long personalHam() {
        return personalHam;
    }

    @NoAllocation
    public void setPersonalHam(final long personalHam) {
        this.personalHam = personalHam;
    }

    @NoAllocation
    public long personalSpam() {
        return personalSpam;
    }

    @NoAllocation
    public void setPersonalSpam(final long personalSpam) {
        this.personalSpam = personalSpam;
    }

    @NoAllocation
    public double weight() {
        return weight;
    }

    @NoAllocation
    public void setWeight(final double weight) {
        this.weight = weight;
    }

    @NoAllocation
    public long allSpam() {
        return spam + malic;
    }

    @NoAllocation
    public double hamToAllSpam() {
        return ham / (double) Math.max(allSpam(), 1L);
    }

    @NoAllocation
    @Override
    public int expectedStringLength() {
        return PREFIX_LENGTH + EXPECTED_STRING_LENGTH + 1;
    }

    @Override
    public void toStringBuilder(@Nonnull final StringBuilder sb) {
        sb.append(PREFIX);
        sb.append(ham);
        sb.append(',');
        sb.append(malic);
        sb.append(',');
        sb.append(spam);
        sb.append(',');
        sb.append(personalHam);
        sb.append(',');
        sb.append(personalSpam);
        sb.append(',');
        sb.append(weight);
        sb.append(')');
    }
}

