package ru.yandex.client.so.shingler;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;

import ru.yandex.concurrent.CompletedFuture;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.CallbackFutureBase;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;

public class MassShinglerClient extends AbstractShinglerClient<MassShinglerClient, MassShinglerResult> {
    private static final CompletedFuture<MassShinglerResult> EMPTY_RESULT =
        new CompletedFuture<>(MassShinglerResult.EMPTY);
    public static final String URI = "/shinrqst?";
    public static final String GET_URI = URI + "shget=";
    public static final String PUT_URI = URI + "shput=";

    public MassShinglerClient(
        @Nonnull final SharedConnectingIOReactor reactor,
        @Nonnull final ImmutableHttpHostConfig config)
    {
        super(reactor, config);
    }

    private MassShinglerClient(
        @Nonnull final CloseableHttpAsyncClient client,
        @Nonnull final MassShinglerClient sample)
    {
        super(client, sample);
    }

    @Override
    public MassShinglerClient adjust(@Nonnull final CloseableHttpAsyncClient client)
    {
        return new MassShinglerClient(client, this);
    }

    public static String getGetQuery(@Nonnull final Map<ShingleType, Map<Long, Shingle>> shinglesStats) {
        StringBuilder sb = new StringBuilder(GET_URI);
        for (Map.Entry<ShingleType, Map<Long, Shingle>> entry : shinglesStats.entrySet()) {
            ShingleType type = entry.getKey();
            for (Long shingle: entry.getValue().keySet()) {
                sb.append(Long.toHexString(shingle));
                sb.append('-');
                sb.append(type.id());
                sb.append('=');
                sb.append('&');
            }
        }
        return sb.toString();
    }

    @Override
    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<MassShinglerResult> getShingles(
        @Nonnull final AbstractShinglesMap<?> shingles,
        @Nonnull final Supplier<? extends HttpClientContext> contextGenerator,
        @Nonnull final FutureCallback<? super MassShinglerResult> callback)
        throws ShinglerClientException, ShingleException
    {
        if (shingles.isEmpty()) {
            callback.completed(MassShinglerResult.EMPTY);
            return EMPTY_RESULT;
        }
        GetQueryCallbackProxy callbackProxy = new GetQueryCallbackProxy(callback, (Shingles) shingles);
        execute(
            host,
            new BasicAsyncRequestProducerGenerator(getGetQuery((Shingles) shingles)),
            AsyncStringConsumerFactory.OK,
            contextGenerator,
            callbackProxy);
        return callbackProxy;
    }

    public static String getPutQuery(@Nonnull final MassShinglerResult shinglesStats) {
        StringBuilder sb = new StringBuilder(PUT_URI);
        int spamType;
        int persType;
        MassShingleInfo shingleInfo;
        for (Map.Entry<ShingleType, List<MassShingleResult>> entry
                : shinglesStats.entrySet().stream()
                    .sorted(Comparator.comparing(el -> el.getKey().id())).collect(Collectors.toList())) {
            ShingleType type = entry.getKey();
            spamType = 3;
            persType = 0;
            for (final MassShingleResult shingleData : entry.getValue()) {
                if (shingleData == null || shingleData.shingleInfo().todayStats() == null) {
                    return null;
                }
                shingleInfo = shingleData.shingleInfo();
                sb.append(Long.toHexString(shingleData.shingle().shingleHash()));
                sb.append('-');
                sb.append(type.id());
                sb.append('=');
                sb.append(shingleData.shingle().count());
                sb.append('-');
                if (shingleInfo.todayStats().ham() > 0) {
                    spamType = 0;
                } else if (shingleInfo.todayStats().malic() > 0) {
                    spamType = 1;
                } else if (shingleInfo.todayStats().spam() > 0) {
                    spamType = 2;
                }
                sb.append(spamType);
                sb.append('-');
                if (shingleInfo.todayStats().personalHam() > 0) {
                    persType = 1;
                } else if (shingleInfo.todayStats().personalSpam() > 0) {
                    persType = 2;
                }
                sb.append(persType);
                if (shingleInfo.bindedUniqueShingle() != null && shingleInfo.bindedUniqueShingleType() != 0) {
                    sb.append(',');
                    sb.append(shingleInfo.bindedUniqueShingle());
                    sb.append('-');
                    sb.append(shingleInfo.bindedUniqueShingleType());
                }
                sb.append('&');
            }
        }
        return sb.toString();
    }

    @Override
    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(
        @Nonnull final MassShinglerResult shinglesStats,
        @Nonnull final Supplier<? extends HttpClientContext> contextGenerator,
        @Nonnull final FutureCallback<String> callback)
        throws ShinglerClientException, ShingleException
    {
        if (shinglesStats.isEmpty()) {
            callback.completed(OK);
            return SUCCESS_PUT_RESULT;
        }
        PutQueryCallbackProxy callbackProxy = new PutQueryCallbackProxy(callback);
        execute(
            host,
            new BasicAsyncRequestProducerGenerator(getPutQuery(shinglesStats)),
            EmptyAsyncConsumerFactory.OK,
            contextGenerator,
            callbackProxy);
        return callbackProxy;
    }

    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(@Nonnull final String query)
    {
        return putShingles(query, httpClientContextGenerator(), StringFutureCallback.INSTANCE);
    }

    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(
        @Nonnull final String query,
        @Nonnull final FutureCallback<String> callback)
    {
        return putShingles(query, httpClientContextGenerator(), callback);
    }

    @Nonnull
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<String> putShingles(
        @Nonnull final String query,
        @Nonnull final Supplier<? extends HttpClientContext> contextGenerator,
        @Nonnull final FutureCallback<String> callback)
    {
        PutQueryCallbackProxy callbackProxy = new PutQueryCallbackProxy(callback);
        execute(
            host,
            new BasicAsyncRequestProducerGenerator(query),
            EmptyAsyncConsumerFactory.OK,
            contextGenerator,
            callbackProxy);
        return callbackProxy;
    }

    private static class GetQueryCallbackProxy
        extends CallbackFutureBase<MassShinglerResult, String>
    {
        @Nonnull
        private final Shingles shingles;

        GetQueryCallbackProxy(
            @Nonnull final FutureCallback<? super MassShinglerResult> callback,
            @Nonnull final Shingles shingles)
        {
            super(callback);
            this.shingles = shingles;
        }

        @Override
        @Nonnull
        protected MassShinglerResult convertResult(@Nonnull final String response) {
            return new MassShinglerResult(shingles, response);
        }

        @Override
        protected void onFailure(@Nonnull final Exception e) {
            callback.completed(new MassShinglerResult(e));
        }
    }

    private static class PutQueryCallbackProxy extends CallbackFutureBase<String, Void>
    {
        PutQueryCallbackProxy(@Nonnull final FutureCallback<String> callback) {
            super(callback);
        }

        @Override
        @Nonnull
        protected String convertResult(@Nonnull final Void unused) {
            return OK;
        }

        @Override
        protected void onFailure(final Exception e) {
            //System.err.println("MassShinglerClient.PutQueryCallbackProxy failed: " + e.toString());
            //callback.failed(e);
            callback.completed(null);
        }
    }
}

