package ru.yandex.client.so.shingler;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nonnull;

import com.google.errorprone.annotations.NoAllocation;

import ru.yandex.function.StringBuilderable;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.mail.errors.ErrorInfo;

public class MassShinglerResult extends AbstractShinglesMap<List<MassShingleResult>> {
    public static final MassShinglerResult EMPTY = new MassShinglerResult();

    private static final long serialVersionUID = 0L;
    private static final String TYPE = "type";
    private static final String UNIQ = "uniq";
    private static final String SHINGLE = "shingle";
    private static final String PREFIX = "MassShinglerResult(";
    private static final int PREFIX_LENGTH = PREFIX.length();
    private static final List<MassShingleResult> EMPTY_SHINGLE_RESULT =
        Collections.emptyList();

    @Nonnull
    private final List<ErrorInfo> errors;

    public MassShinglerResult() {
        super();
        errors = Collections.emptyList();
    }

    public MassShinglerResult(@Nonnull final Exception e) {
        super();
        errors = new ArrayList<>(1);
        errors.add(
            new ErrorInfo(
                ErrorInfo.Scope.SHINGLER,
                ErrorInfo.Type.EXTERNAL_COMPONENT_ERROR,
                "Mass Shingler failed to respond",
                e));
    }

    public MassShinglerResult(@Nonnull final Shingles shingles) {
        this();
        for (Map.Entry<ShingleType, Map<Long, Shingle>> entry : shingles.entrySet()) {
            ShingleType type = entry.getKey();
            Map<Long, Shingle> hashToShingle = entry.getValue();
            List<MassShingleResult> shinglesInfo = new ArrayList<>(hashToShingle.size());
            for (Map.Entry<Long, Shingle> shingleInfo : entry.getValue().entrySet()) {
                shinglesInfo.add(
                    new MassShingleResult(
                        shingleInfo.getValue(),
                        new MassShingleInfo(shingleInfo.getValue(), entry.getKey())));
            }
            put(type, shinglesInfo);
        }
    }

    public MassShinglerResult(
        @Nonnull final Shingles shingles,
        @Nonnull final String response)
    {
        super();
        errors = new ArrayList<>();
        Map<Integer, ShinglesAndType> idToType =
            new HashMap<>(shingles.size() << 1);
        for (Map.Entry<ShingleType, Map<Long, Shingle>> entry
            : shingles.entrySet())
        {
            ShingleType type = entry.getKey();
            Map<Long, Shingle> hashToShingle = entry.getValue();
            idToType.put(
                type.id(),
                new ShinglesAndType(type, hashToShingle));
            put(type, new ArrayList<>(hashToShingle.size()));
        }
        boolean empty = true;
        int end = -1;
        while (true) {
            int start = response.indexOf('<', end + 1);
            end = response.indexOf('>', start + 1);
            if (start == -1 || end == -1) {
                break;
            }
            empty = false;
            String str = response.substring(start, end);
            if (str.startsWith(MassShingleInfo.SHIDENT)) {
                try {
                    MassShingleInfo info = new MassShingleInfo(str);
                    int id = info.shingleType();
                    ShinglesAndType shinglesAndType = idToType.get(id);
                    if (shinglesAndType == null) {
                        errors.add(
                            new ErrorInfo(
                                ErrorInfo.Scope.SHINGLER_RESPONSE,
                                ErrorInfo.Type.UNEXPECTED_ENTRY,
                                "Unexpected shingle type: " + str));
                    } else {
                        Shingle shingle =
                            shinglesAndType.shingles.get(info.shingleHash());
                        if (shingle == null) {
                            errors.add(
                                new ErrorInfo(
                                    ErrorInfo.Scope.SHINGLER_RESPONSE,
                                    ErrorInfo.Type.UNEXPECTED_ENTRY,
                                    "Unexpected shingle: " + str));
                        } else {
                            get(shinglesAndType.type)
                                .add(new MassShingleResult(shingle, info));
                        }
                    }
                } catch (ShingleException e) {
                    errors.add(
                        new ErrorInfo(
                            ErrorInfo.Scope.SHINGLER_RESPONSE,
                            ErrorInfo.Type.SYNTAX_ERROR,
                            "Failed to parse shingle from " + str,
                            e));
                }
            }
        }
        if (empty) {
            errors.add(
                new ErrorInfo(
                    ErrorInfo.Scope.SHINGLER_RESPONSE,
                    ErrorInfo.Type.FIELD_MISSING,
                    "No shingles found in '" + response + '\''));
        }
    }

    public MassShinglerResult(@Nonnull final JsonObject request) throws JsonException {
        super();
        errors = new ArrayList<>();
        ShingleType shingleType;
        ShingleType shingleUniqType;
        if (request instanceof JsonList) {
            final JsonList data = request.asList();
            for (final JsonObject entry : data) {
                if (entry instanceof JsonMap) {
                    final JsonMap counters = entry.asMap();
                    shingleType = ShingleType.UNKNOWN;
                    shingleUniqType = ShingleType.UNKNOWN;
                    long shingle = 0L;
                    long shingleUniq = 0L;
                    int count = 0;
                    for (final Entry<String, JsonObject> counter : counters.entrySet()) {
                        switch (counter.getKey()) {
                            case TYPE:
                                shingleType = ShingleType.fromId((int) counter.getValue().asLong());
                                break;
                            case SHINGLE:
                                if (counter.getValue().type() == JsonObject.Type.STRING) {
                                    shingle = Long.parseUnsignedLong(counter.getValue().asString());
                                } else if (counter.getValue().type() == JsonObject.Type.LONG) {
                                    shingle = counter.getValue().asLong();
                                }
                                break;
                            case "count":
                                if (counter.getValue().type() == JsonObject.Type.STRING) {
                                    count = Integer.parseUnsignedInt(counter.getValue().asString());
                                } else if (counter.getValue().type() == JsonObject.Type.LONG) {
                                    count = (int) counter.getValue().asLong();
                                }
                                break;
                            case UNIQ:
                                if (counter.getValue().type() == JsonObject.Type.MAP) {
                                    final JsonMap uniqMap = counter.getValue().asMap();
                                    for (final Entry<String, JsonObject> uniqInfo : uniqMap.entrySet()) {
                                        if (uniqInfo.getKey().equals(TYPE)) {
                                            shingleUniqType = ShingleType.fromId((int) uniqInfo.getValue().asLong());
                                        } else if (uniqInfo.getKey().equals(SHINGLE)) {
                                            if (uniqInfo.getValue().type() == JsonObject.Type.STRING) {
                                                shingleUniq = Long.parseUnsignedLong(uniqInfo.getValue().asString());
                                            } else if (uniqInfo.getValue().type() == JsonObject.Type.LONG) {
                                                shingleUniq = uniqInfo.getValue().asLong();
                                            }
                                        }
                                    }
                                }
                                break;
                        }
                    }
                    final Shingle shingleObj = new Shingle(shingle);
                    final Shingle shingleUniqObj = new Shingle(shingleUniq);
                    shingleObj.setCount(count);
                    computeIfAbsent(shingleType, x -> new ArrayList<>())
                        .add(new MassShingleResult(
                            shingleObj,
                            new MassShingleInfo(shingleObj, shingleType, shingleUniqObj, shingleUniqType)));
                }
            }
        }
    }

    public MassShinglerResult(final MassShinglerResult data) {
        errors = new ArrayList<>(data.errors());
        for (final Map.Entry<ShingleType, List<MassShingleResult>> entry : data.entrySet()) {
            for (final MassShingleResult result : entry.getValue()) {
                computeIfAbsent(entry.getKey(), x -> new ArrayList<>()).add(result);
            }
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public void addAll(final AbstractShinglesMap<?> data) throws ShingleException {
        try {
            for (final Map.Entry<ShingleType, List<MassShingleResult>> entry
                    : ((AbstractShinglesMap<List<MassShingleResult>>) data).entrySet()) {
                computeIfAbsent(entry.getKey(), x -> new ArrayList<>()).addAll(entry.getValue());
            }
        } catch (Exception e) {
            throw new ShingleException("MassShinglerResult.addAll failed: " + e.toString(), e);
        }
    }

    public MassShinglerResult setSpam(final boolean isSpam, final boolean isPersonal) {
        for (final Map.Entry<ShingleType, List<MassShingleResult>> entry : entrySet()) {
            for (final MassShingleResult info : entry.getValue()) {
                if (isSpam) {
                    Objects.requireNonNull(info.shingleInfo().todayStats()).setSpam(1);
                    if (isPersonal) {
                        Objects.requireNonNull(info.shingleInfo().todayStats()).setPersonalSpam(1);
                    }
                } else {
                    Objects.requireNonNull(info.shingleInfo().todayStats()).setHam(1);
                    if (isPersonal) {
                        Objects.requireNonNull(info.shingleInfo().todayStats()).setPersonalHam(1);
                    }
                }
            }
        }
        return this;
    }

    @Override
    public MassShinglerResult getEmpty() {
        return EMPTY;
    }

    @NoAllocation
    @Nonnull
    public List<MassShingleResult> shinglesFor(final ShingleType type) {
        return getOrDefault(type, EMPTY_SHINGLE_RESULT);
    }

    @NoAllocation
    @Nonnull
    public List<ErrorInfo> errors() {
        return errors;
    }

    @Override
    public int expectedStringLength() {
        int len = PREFIX_LENGTH + 4 - 1 + (size() << 1);
        for (Map.Entry<ShingleType, List<MassShingleResult>> entry : entrySet())
        {
            len += entry.getKey().toString().length();
            len += StringBuilderable.calcExpectedStringLength(entry.getValue());
        }
        return len;
    }

    @Override
    public void toStringBuilder(@Nonnull final StringBuilder sb) {
        sb.append(PREFIX);
        sb.append('{');
        boolean first = true;
        for (Map.Entry<ShingleType, List<MassShingleResult>> entry : entrySet())
        {
            if (first) {
                first = false;
            } else {
                sb.append(',');
            }
            sb.append(entry.getKey().toString());
            sb.append('=');
            StringBuilderable.toStringBuilder(sb, entry.getValue());
        }
        sb.append('}');
        sb.append(',');
        StringBuilderable.toStringBuilder(sb, errors);
        sb.append(')');
    }

    private static class ShinglesAndType {
        @Nonnull
        private final ShingleType type;
        @Nonnull
        private final Map<String, Shingle> shingles;

        ShinglesAndType(
            @Nonnull final ShingleType type,
            @Nonnull final Map<Long, Shingle> shingles)
        {
            this.type = type;
            this.shingles = new HashMap<>(shingles.size() << 1);
            for (Map.Entry<Long, Shingle> entry: shingles.entrySet()) {
                this.shingles.put(
                    Long.toHexString(entry.getKey()),
                    entry.getValue());
            }
        }
    }
}
