package ru.yandex.parser.mail.envelope;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Random;
import java.util.function.Consumer;
import java.util.regex.Pattern;

import com.google.protobuf.ByteString;
import com.google.protobuf.Int64Value;

import ru.yandex.function.NullConsumer;
import ru.yandex.mail.so.api.v1.ConnectInfo;
import ru.yandex.mail.so.api.v1.Email;
import ru.yandex.mail.so.api.v1.EmailInfo;
import ru.yandex.mail.so.api.v1.SmtpEnvelope;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.mail.errors.ErrorInfo;
import ru.yandex.parser.string.ValuesStorage;
import ru.yandex.util.ip.IpCompressor;
import ru.yandex.util.string.StringUtils;

public class SmtpEnvelopeHolder {
    private static final String QID = "qid=";
    private static final String SUID_START = " id=";
    private static final String UID_START = " uid=";
    private static final Pattern SEMICOLON_PATTERN = Pattern.compile(";");
    private static final int MIN_RCPTTOS = 5;
    private static final int MAX_RCPTTOS = 50;

    private final SmtpEnvelope envelope;

    public <E extends Exception> SmtpEnvelopeHolder(
        final ValuesStorage<E> params)
    {
        this(NullConsumer.instance(), params);
    }

    public <E extends Exception> SmtpEnvelopeHolder(
        final Consumer<ErrorInfo> errorsConsumer,
        final ValuesStorage<E> params)
    {
        SmtpEnvelope.Builder envelopeBuilder = SmtpEnvelope.newBuilder();
        parseConnect(errorsConsumer, envelopeBuilder, params);
        parseMailFrom(errorsConsumer, envelopeBuilder, params);
        parseRcptTos(errorsConsumer, envelopeBuilder, params);

        envelope = envelopeBuilder.build();
    }

    public SmtpEnvelopeHolder(final SmtpEnvelope envelope) {
        SmtpEnvelope.Builder builder = envelope.toBuilder();
        if (builder.hasMailFrom()) {
            enrichEmailInfo(builder.getMailFromBuilder());
        }
        List<EmailInfo.Builder> recipients =
            builder.getRecipientsBuilderList();
        int size = recipients.size();
        long hashCode = 0L;
        for (int i = 0; i < size; ++i) {
            EmailInfo.Builder recipient = recipients.get(i);
            enrichEmailInfo(recipient);
            hashCode *= 31L;
            hashCode += recipient.getAddressBuilder().getEmail().hashCode();
        }
        List<EmailInfo.Builder> recipientsCopy = new ArrayList<>(recipients);
        builder.clearRecipients();
        setRecipients(builder, recipientsCopy, hashCode);
        this.envelope = builder.build();
    }

    private static void enrichEmailInfo(final EmailInfo.Builder emailInfo) {
        if (emailInfo.hasAddress()) {
            Email.Builder address = emailInfo.getAddressBuilder();
            String email = address.getEmail();
            if (!email.isEmpty()) {
                emailInfo.setAddress(parseEmail(email));
            }
        }
    }

    private static String extractParam(
        final String haystack,
        final String needle)
    {
        return extractParam(haystack, needle, 0);
    }

    private static String extractParam(
        final String haystack,
        final String needle,
        final int start)
    {
        int idx = StringUtils.indexOfIgnoreCase(haystack, needle, start);
        if (idx != -1) {
            idx += needle.length();
            int end = haystack.indexOf(' ', idx);
            if (end == -1) {
                end = haystack.length();
            }
            if (end > idx) {
                return haystack.substring(idx, end);
            }
        }
        return null;
    }

    private static <E extends Exception> void parseConnect(
        final Consumer<ErrorInfo> errorsConsumer,
        final SmtpEnvelope.Builder envelopeBuilder,
        final ValuesStorage<E> params)
    {
        String connect = params.getString("CONNECT", null);
        if (connect == null) {
            String clientIp = params.getString("client_ip", null);
            if (clientIp != null) {
                ConnectInfo.Builder builder = ConnectInfo.newBuilder();
                try {
                    builder.setRemoteIp(
                        ByteString.copyFrom(
                            InetAddress.getByName(clientIp).getAddress()));
                    envelopeBuilder.setConnectInfo(builder);
                } catch (UnknownHostException e) {
                    errorsConsumer.accept(
                        new ErrorInfo(
                            ErrorInfo.Scope.IP,
                            ErrorInfo.Type.SYNTAX_ERROR,
                            "Failed to parse IP <" + clientIp + '>',
                            e));
                }
            }
            return;
        } else if (connect.isEmpty()) {
            return;
        }
        ConnectInfo.Builder builder = ConnectInfo.newBuilder();
        boolean empty = true;
        int ipStart;
        int end;
        if (connect.charAt(0) == '[') {
            ipStart = 0;
            end = 0;
        } else {
            end = connect.indexOf(' ');
            if (end > 0) {
                builder.setRemoteHost(
                    connect.substring(0, end).toLowerCase(Locale.ROOT));
                empty = false;
            }
            ipStart = connect.indexOf('[', end + 1);
        }
        if (ipStart != -1) {
            ++ipStart;
            end = connect.indexOf(']', ipStart);
            if (end > ipStart) {
                String ipString = connect.substring(ipStart, end);
                try {
                    builder.setRemoteIp(
                        ByteString.copyFrom(
                            InetAddress.getByName(ipString).getAddress()));
                    empty = false;
                } catch (UnknownHostException e) {
                    errorsConsumer.accept(
                        new ErrorInfo(
                            ErrorInfo.Scope.IP,
                            ErrorInfo.Type.SYNTAX_ERROR,
                            "Failed to parse IP <" + ipString + '>',
                            e));
                }
            }
        }
        String qid = extractParam(connect, QID, end + 1);
        if (qid != null) {
            builder.setSessionId(qid);
            empty = false;
        }

        String helo = params.getString("HELO", null);
        if (helo != null) {
            helo = helo.trim();
        }
        if (helo != null && !helo.isEmpty() && helo.indexOf(' ') == -1) {
            builder.setRemoteDomain(helo);
            empty = false;
        }

        if (!empty) {
            envelopeBuilder.setConnectInfo(builder);
        }
    }

    public static Email.Builder parseEmail(final String email) {
        return parseEmail(email, MailAliases.emailSeparatorPos(email));
    }

    public static Email.Builder parseEmail(
        final String email,
        final int sep)
    {
        Email.Builder builder = Email.newBuilder();
        builder.setEmail(email);
        if (sep == -1) {
            String domain = MailAliases.domain(email);
            builder.setNormalizedEmail(domain);
            builder.setNormalizedDomain(domain);
        } else {
            Map.Entry<String, String> normalized =
                MailAliases.INSTANCE.parseAndNormalize(email, sep);
            String normalizedLocal = normalized.getKey();
            String normalizedDomain = normalized.getValue();
            builder.setNormalizedLocal(normalizedLocal);
            builder.setNormalizedDomain(normalizedDomain);
            builder.setNormalizedEmail(
                StringUtils.concat(normalizedLocal, '@', normalizedDomain));
        }
        return builder;
    }

    private static <E extends Exception> void parseMailFrom(
        final Consumer<ErrorInfo> errorsConsumer,
        final SmtpEnvelope.Builder envelopeBuilder,
        final ValuesStorage<E> params)
    {
        String mailfrom = params.getString("MAILFROM", null);
        if (mailfrom == null) {
            EmailInfo.Builder builder = EmailInfo.newBuilder();
            boolean empty = true;
            mailfrom = params.getString("from", null);
            if (mailfrom != null) {
                mailfrom = mailfrom.trim();
                if (!mailfrom.isEmpty()) {
                    builder.setAddress(parseEmail(mailfrom));
                    empty = false;
                }
            }
            Long uid = null;
            try {
                uid = params.getLong("uid", null);
            } catch (Exception e) {
                errorsConsumer.accept(
                    new ErrorInfo(
                        ErrorInfo.Scope.UID,
                        ErrorInfo.Type.SYNTAX_ERROR,
                        "Failed to parse uid",
                        e));
            }
            if (uid != null) {
                builder.setUid(
                    Int64Value.newBuilder().setValue(uid.longValue()));
                empty = false;
            }
            if (!empty) {
                envelopeBuilder.setMailFrom(builder);
            }
        } else {
            mailfrom = mailfrom.trim();
            if (!mailfrom.isEmpty()) {
                EmailInfo.Builder emailInfo =
                    parseEmailInfo(errorsConsumer, mailfrom);
                if (emailInfo != null) {
                    envelopeBuilder.setMailFrom(emailInfo);
                }
            }
        }
    }

    private static EmailInfo.Builder parseEmailInfo(
        final Consumer<ErrorInfo> errorsConsumer,
        final String rcptto)
    {
        EmailInfo.Builder builder = EmailInfo.newBuilder();
        boolean empty = true;
        int space = rcptto.indexOf(' ');
        if (space == -1) {
            space = rcptto.length();
        }
        if (space > 0) {
            int equal = rcptto.indexOf('=');
            if (equal == -1 || equal > space) {
                builder.setAddress(parseEmail(rcptto.substring(0, space)));
                empty = false;
            } else {
                String maybeEmail = rcptto.substring(0, space);
                int sep = MailAliases.emailSeparatorPos(maybeEmail);
                if (sep > equal) {
                    builder.setAddress(parseEmail(maybeEmail, sep));
                    empty = false;
                }
            }
            String suid = extractParam(rcptto, SUID_START);
            if (suid != null) {
                try {
                    builder.setSuid(
                        Int64Value.newBuilder()
                            .setValue(Long.parseLong(suid)));
                    empty = false;
                } catch (RuntimeException e) {
                    errorsConsumer.accept(
                        new ErrorInfo(
                            ErrorInfo.Scope.SUID,
                            ErrorInfo.Type.SYNTAX_ERROR,
                            "Failed to parse suid <" + suid + '>',
                            e));
                }
            }
            String uid = extractParam(rcptto, UID_START);
            if (uid != null) {
                try {
                    builder.setUid(
                        Int64Value.newBuilder()
                            .setValue(Long.parseLong(uid)));
                    empty = false;
                } catch (RuntimeException e) {
                    errorsConsumer.accept(
                        new ErrorInfo(
                            ErrorInfo.Scope.UID,
                            ErrorInfo.Type.SYNTAX_ERROR,
                            "Failed to parse uid <" + uid + '>',
                            e));
                }
            }
        }
        if (empty) {
            return null;
        } else {
            return builder;
        }
    }

    private static void setRecipients(
        final SmtpEnvelope.Builder envelopeBuilder,
        final List<EmailInfo.Builder> recipients,
        final long hashCode)
    {
        int size = recipients.size();
        if (size <= MAX_RCPTTOS) {
            for (int i = 0; i < size; ++i) {
                envelopeBuilder.addRecipients(recipients.get(i));
            }
        } else {
            for (int i = 0; i < MIN_RCPTTOS; ++i) {
                envelopeBuilder.addRecipients(recipients.get(i));
            }
            Random random = new Random(hashCode);
            int added = MIN_RCPTTOS;
            int bound = size - MIN_RCPTTOS;
            while (added++ < MAX_RCPTTOS) {
                int i = MIN_RCPTTOS + random.nextInt(bound--);
                envelopeBuilder.addRecipients(recipients.remove(i));
            }
        }
    }

    private static <E extends Exception> void parseRcptTos(
        final Consumer<ErrorInfo> errorsConsumer,
        final SmtpEnvelope.Builder envelopeBuilder,
        final ValuesStorage<E> params)
    {
        List<EmailInfo.Builder> rcpttos = new ArrayList<>();
        Iterator<String> rcpttosIter = params.getAllOrNull("RCPTTO");
        if (rcpttosIter == null) {
            rcpttosIter = params.getAllOrNull("to");
        }
        long hashCode = 0L;
        if (rcpttosIter != null) {
            while (rcpttosIter.hasNext()) {
                String rcpttosStr = rcpttosIter.next();
                hashCode *= 31L;
                hashCode += rcpttosStr.hashCode();
                String[] splitted = SEMICOLON_PATTERN.split(rcpttosStr);
                for (String rcptto: splitted) {
                    EmailInfo.Builder recipient =
                        parseEmailInfo(errorsConsumer, rcptto.trim());
                    if (recipient != null) {
                        rcpttos.add(recipient);
                    }
                }
            }
        }
        setRecipients(envelopeBuilder, rcpttos, hashCode);
    }

    public String ipString() {
        InetAddress address = ip();
        if (address == null) {
            return null;
        } else {
            return IpCompressor.compress(address);
        }
    }

    public InetAddress ip() {
        if (envelope.hasConnectInfo()) {
            ByteString address = envelope.getConnectInfo().getRemoteIp();
            if (address != null) {
                try {
                    return InetAddress.getByAddress(address.toByteArray());
                } catch (UnknownHostException e) {
                    // impossible case
                    throw new RuntimeException(e);
                }
            }
        }
        return null;
    }

    // Tries to get frm with fallback to MAILFROM.
    public Email from() {
        if (envelope.hasMailFrom()) {
            EmailInfo mailFrom = envelope.getMailFrom();
            if (mailFrom.hasAddress()) {
                return mailFrom.getAddress();
            }
        }
        return null;
    }

    // Return 0 if uid is not known
    public long fromUid() {
        return envelope.getMailFrom().getUid().getValue();
    }

    public SmtpEnvelope envelope() {
        return envelope;
    }

    public List<EmailInfo> recipients() {
        return envelope.getRecipientsList();
    }
}

