package ru.yandex.mail.so.factors;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import ru.yandex.mail.so.factors.types.SoFactorType;

public class BasicSoFunctionInputs implements SoFunctionInputs {
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final Lock readLock = lock.readLock();
    private final Lock writeLock = lock.writeLock();
    private final FactorsAccessViolationHandler accessViolationHandler;
    private final int size;
    private final List<SoFactor<?>> factors;

    public BasicSoFunctionInputs(
        final FactorsAccessViolationHandler accessViolationHandler,
        final int size)
    {
        this.accessViolationHandler = accessViolationHandler;
        this.size = size;
        factors = new ArrayList<>(size);
        for (int i = 0; i < size; ++i) {
            factors.add(null);
        }
    }

    public BasicSoFunctionInputs(
        final FactorsAccessViolationHandler accessViolationHandler,
        final List<SoFactor<?>> factors)
    {
        this.accessViolationHandler = accessViolationHandler;
        size = factors.size();
        this.factors = new ArrayList<>(factors);
    }

    public BasicSoFunctionInputs(
        final FactorsAccessViolationHandler accessViolationHandler,
        final SoFactor<?> factor)
    {
        this.accessViolationHandler = accessViolationHandler;
        size = 1;
        factors = new ArrayList<>(1);
        factors.add(factor);
    }

    public BasicSoFunctionInputs(
        final FactorsAccessViolationHandler accessViolationHandler,
        final SoFactor<?> factor1,
        final SoFactor<?> factor2)
    {
        this.accessViolationHandler = accessViolationHandler;
        size = 2;
        factors = new ArrayList<>(2);
        factors.add(factor1);
        factors.add(factor2);
    }

    @Override
    public FactorsAccessViolationHandler accessViolationHandler() {
        return accessViolationHandler;
    }

    public void set(final int pos, final SoFactor<?> factor) {
        writeLock.lock();
        try {
            factors.set(pos, factor);
        } finally {
            writeLock.unlock();
        }
    }

    public int set(
        final List<SoFactor<?>> factors,
        final int[] remapping,
        final int offset)
    {
        int size = factors.size();
        int setCount = 0;
        writeLock.lock();
        try {
            for (int i = 0; i < size; ++i) {
                SoFactor<?> factor = factors.get(i);
                if (factor != null) {
                    ++setCount;
                    this.factors.set(remapping[i] - offset, factor);
                }
            }
        } finally {
            writeLock.unlock();
        }
        return setCount;
    }

    @Override
    public SoFactor<?> get(final int pos) {
        if (pos < 0 || pos >= size) {
            accessViolationHandler.handleFactorOutOfBounds(pos, size);
            return null;
        }
        readLock.lock();
        try {
            return factors.get(pos);
        } finally {
            readLock.unlock();
        }
    }

    @Override
    public <T> T get(int pos, SoFactorType<T> type) {
        if (pos < 0 || pos >= size) {
            accessViolationHandler.handleFactorOutOfBounds(pos, size, type);
            return null;
        }
        SoFactor<?> factor;
        readLock.lock();
        try {
            factor = factors.get(pos);
        } finally {
            readLock.unlock();
        }
        if (factor == null) {
            return null;
        }
        SoFactorType<?> factorType = factor.type();
        if (factorType != type) {
            accessViolationHandler.handleFactorTypeMismatch(pos, type, factor);
            return null;
        }
        return type.cast(factor.value());
    }

    @Override
    public int size() {
        return factors.size();
    }
}

