package ru.yandex.mail.so.factors;

import java.util.concurrent.atomic.LongAdder;
import java.util.logging.Level;
import java.util.logging.Logger;

import ru.yandex.mail.so.factors.types.SoFactorType;

public class LoggingFactorsAccessViolationHandler
    implements FactorsAccessViolationHandler
{
    private final LongAdder violationsCounter;
    private final Logger logger;

    public LoggingFactorsAccessViolationHandler(
        final LongAdder violationsCounter,
        final Logger logger)
    {
        this.violationsCounter = violationsCounter;
        this.logger = logger;
    }

    @Override
    public LongAdder violationsCounter() {
        return violationsCounter;
    }

    @Override
    public void handleFactorOutOfBounds(final int id, final int size) {
        violationsCounter.increment();
        StringBuilder sb = new StringBuilder("Requested factor #");
        sb.append(id);
        sb.append(" while factors storage size is ");
        sb.append(size);
        logger.log(
            Level.WARNING,
            new String(sb),
            new Exception("From here"));
    }

    @Override
    public void handleFactorOutOfBounds(
        final int id,
        final int size,
        final SoFactorType<?> expectedType)
    {
        violationsCounter.increment();
        StringBuilder sb = new StringBuilder("Requested factor #");
        sb.append(id);
        sb.append(" of type ");
        expectedType.toStringBuilder(sb);
        sb.append(" while factors storage size is ");
        sb.append(size);
        logger.log(
            Level.WARNING,
            new String(sb),
            new Exception("From here"));
    }

    @Override
    public void handleFactorTypeMismatch(
        final int id,
        final SoFactorType<?> expectedType,
        final SoFactor<?> actualFactor)
    {
        violationsCounter.increment();
        StringBuilder sb = new StringBuilder("Type mismatch for factor #");
        sb.append(id);
        sb.append(", expected type ");
        expectedType.toStringBuilder(sb);
        sb.append(", while stored factor is ");
        actualFactor.toStringBuilder(sb);
        logger.log(
            Level.WARNING,
            new String(sb),
            new Exception("From here"));
    }

    @Override
    public void handleBadFieldAccessorFieldType(
        final int id,
        final SoFactor<?> factor,
        final SoFactorFieldAccessor fieldAccessor,
        final Object value)
    {
        violationsCounter.increment();
        StringBuilder sb =
            new StringBuilder(
                "Bad field accessor field type for factor #" + id);
        sb.append(id);
        sb.append(": field accessor ");
        fieldAccessor.toStringBuilder(sb);
        sb.append(" extracted value ");
        sb.append(value.toString());
        sb.append(' ');
        sb.append('(');
        sb.append(value.getClass().getCanonicalName());
        sb.append(" from factor ");
        factor.toStringBuilder(sb);
        sb.append(" which cannot be cannot be used for factor type ");
        fieldAccessor.fieldType().toStringBuilder(sb);
        logger.log(
            Level.WARNING,
            new String(sb),
            new Exception("From here"));
    }

    @Override
    public void handleFieldConversionFailure(final String message) {
        violationsCounter.increment();
        logger.log(Level.WARNING, message, new Exception("From here"));
    }

    @Override
    public void handleFieldConversionFailure(
        final String message,
        final Exception e)
    {
        violationsCounter.increment();
        logger.log(Level.WARNING, message + ". From here", e);
    }
}

