package ru.yandex.mail.so.factors.actions;

import java.io.IOException;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.function.CorpUidPredicate;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorFactoryContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.JsonObjectSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class ActionsProcessorExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(ActionsSoFactorType.ACTIONS);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(JsonObjectSoFactorType.JSON_OBJECT);

    private final AsyncClient producerClient;
    private final HttpHost producerHost;

    public ActionsProcessorExtractor(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        ImmutableHttpHostConfig hostConfig =
            new HttpHostConfigBuilder(config.section("producer")).build();
        producerClient =
            context.asyncClientRegistrar().client(
                name + "-Producer",
                hostConfig);
        producerHost = hostConfig.host();
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        Actions actions = inputs.get(0, ActionsSoFactorType.ACTIONS);
        if (actions == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        AsyncClient producerClient =
            this.producerClient.adjust(context.httpContext());
        Supplier<? extends HttpClientContext> contextGenerator =
            context.requestsListener().createContextGeneratorFor(
                producerClient);
        MultiFutureCallback<Object> multiCallback =
            new MultiFutureCallback<>(new Callback(callback));
        for (Map.Entry<Long, EnumMap<IndexActionType, JsonList>> entry
            : actions.indexActions().uidToActions().entrySet())
        {
            long uid = entry.getKey().longValue();
            String service;
            if (CorpUidPredicate.INSTANCE.test(uid)) {
                service = "corp_change_log";
            } else {
                service = "change_log";
            }
            for (Map.Entry<IndexActionType, JsonList> actionEntry
                : entry.getValue().entrySet())
            {
                IndexActionType actionType = actionEntry.getKey();
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = JsonType.DOLLAR.create(sbw)){
                    writer.startObject();
                    writer.key("prefix");
                    writer.value(uid);
                    actionType.additionalJsonData(writer);
                    writer.key("docs");
                    actionEntry.getValue().writeValue(writer);
                    writer.endObject();
                } catch (IOException e) {
                    callback.failed(e);
                    return;
                }
                producerClient.execute(
                    producerHost,
                    new BasicAsyncRequestProducerGenerator(
                        actionType.uri()
                        + "&actions-processor&service=" + service
                        + "&prefix=" + uid,
                        sbw.toString(),
                        ContentType.APPLICATION_JSON.withCharset(
                            producerClient.requestCharset())),
                    EmptyAsyncConsumerFactory.OK,
                    contextGenerator,
                    multiCallback.newCallback());
            }
        }
        multiCallback.done();
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        ActionsProcessorExtractorFactory.INSTANCE.registerInternals(registry);
    }

    private static class Callback
        extends AbstractFilterFutureCallback<List<Object>, List<SoFactor<?>>>
    {
        Callback(final FutureCallback<? super List<SoFactor<?>>> callback) {
            super(callback);
        }

        @Override
        public void completed(final List<Object> response) {
            callback.completed(
                Collections.singletonList(
                    JsonObjectSoFactorType.JSON_OBJECT.createFactor(
                        JsonNull.INSTANCE)));
        }
    }
}

