package ru.yandex.mail.so.factors.bigb;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import yabs.proto.Profile;

import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.ByteBufferFactory;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.ByteArrayProcessableAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorFactoryContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.BooleanSoFactorType;
import ru.yandex.mail.so.factors.types.LongSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.yabs.server.proto.keywords.EKeyword;

public class BigBExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(LongSoFactorType.LONG);
    private static final List<SoFactorType<?>> OUTPUTS =
        Arrays.asList(
            BooleanSoFactorType.BOOLEAN,
            CryptaUserVectorSoFactorType.CRYPTA_USER_VECTOR);
    private static final SoFactor<?> TRUE =
        BooleanSoFactorType.BOOLEAN.createFactor(true);
    private static final List<SoFactor<?>> EMPTY_RESULT =
        Arrays.asList(
            BooleanSoFactorType.BOOLEAN.createFactor(false),
            null);

    private static final int CRYPTA_USER_VECTOR_ID =
        EKeyword.KW_CRYPTA_YANDEXUID_VECTOR_1.getNumber();

    private final AsyncClient client;
    private final HttpHost host;

    public BigBExtractor(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        ImmutableHttpHostConfig hostConfig =
            new HttpHostConfigBuilder(config).build();
        client = context.asyncClientRegistrar().client(name, hostConfig);
        host = hostConfig.host();
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        Long uid = inputs.get(0, LongSoFactorType.LONG);
        if (uid == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        AsyncClient client = this.client.adjust(context.httpContext());
        client.execute(
            host,
            new BasicAsyncRequestProducerGenerator(
                "/bigb?client=so&format=protobuf&puid=" + uid),
            ByteArrayProcessableAsyncConsumerFactory.OK,
            context.requestsListener().createContextGeneratorFor(client),
            new Callback(callback));
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        BigBExtractorFactory.INSTANCE.registerInternals(registry);
    }

    private static class Callback
        extends AbstractFilterFutureCallback<
            ByteArrayProcessable,
            List<SoFactor<?>>>
    {
        Callback(final FutureCallback<? super List<SoFactor<?>>> callback) {
            super(callback);
        }

        @Override
        public void completed(final ByteArrayProcessable result) {
            try {
                Profile profile = Profile.parseFrom(
                    result.processWith(ByteBufferFactory.INSTANCE));
                for (Profile.ProfileItem item: profile.getItemsList()) {
                    if (item.getKeywordId() == CRYPTA_USER_VECTOR_ID) {
                        SoFactor<?> vector = null;
                        if (item.hasStringValue()) {
                            ByteString value = item.getStringValue();
                            int size = value.size();
                            byte[] bytes = new byte[size];
                            value.copyTo(bytes, 0);
                            vector =
                                CryptaUserVectorSoFactorType.CRYPTA_USER_VECTOR
                                    .createFactor(bytes);
                        }
                        List<SoFactor<?>> factors = new ArrayList<>(2);
                        factors.add(TRUE);
                        factors.add(vector);
                        callback.completed(factors);
                        return;
                    }
                }
            } catch (InvalidProtocolBufferException | RuntimeException e) {
                failed(e);
            }
            callback.completed(EMPTY_RESULT);
        }
    }
}

