package ru.yandex.mail.so.factors.blackbox;

import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;

import ru.yandex.blackbox.BlackboxAliasType;
import ru.yandex.blackbox.BlackboxAttributeType;
import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxDbfield;
import ru.yandex.blackbox.BlackboxEmailsType;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorFactoryContext;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.EnumParser;

public class BlackboxExtractorBase {
    protected final BlackboxClient blackboxClient;
    protected final BlackboxClient corpBlackboxClient;
    protected final String sid;
    protected final BlackboxEmailsType emailsType;
    protected final Set<BlackboxDbfield> dbfields;
    protected final Set<BlackboxDbfield> requiredDbfields;
    protected final Set<BlackboxAliasType> aliases;
    protected final Set<BlackboxAttributeType> attributes;
    protected final boolean getFamilyInfo;

    public BlackboxExtractorBase(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        IniConfig blackboxSection = config.section("blackbox");
        ImmutableHttpHostConfig blackboxConfig =
            new HttpHostConfigBuilder(blackboxSection).build();
        blackboxClient =
            context.asyncClientRegistrar().registerClient(
                name,
                new BlackboxClient(
                    context.asyncClientRegistrar().reactor(),
                    blackboxConfig),
                blackboxConfig);

        IniConfig corpBlackboxSection = config.section("corp-blackbox");
        ImmutableHttpHostConfig corpBlackboxConfig =
            new HttpHostConfigBuilder(corpBlackboxSection).build();
        corpBlackboxClient =
            context.asyncClientRegistrar().registerClient(
                "corp-" + name,
                new BlackboxClient(
                    context.asyncClientRegistrar().reactor(),
                    corpBlackboxConfig),
                corpBlackboxConfig);

        sid = config.getString("sid", "smtp");
        emailsType = config.getEnum(
            BlackboxEmailsType.class,
            "emails-type",
            BlackboxEmailsType.NONE);
        dbfields = parseSet(config, BlackboxDbfield.class, "dbfields");
        requiredDbfields =
            parseSet(config, BlackboxDbfield.class, "required-dbfields");
        aliases = parseSet(config, BlackboxAliasType.class, "aliases");
        attributes =
            parseSet(config, BlackboxAttributeType.class, "attributes");
        getFamilyInfo = config.getBoolean("get-family-info", false);
    }

    private static <E extends Enum<E>> Set<E> parseSet(
        final IniConfig config,
        final Class<E> clazz,
        final String name)
        throws ConfigException
    {
        Set<E> set = config.get(
            name,
            Collections.emptySet(),
            new CollectionParser<>(new EnumParser<>(clazz), HashSet::new));
        if (set.isEmpty()) {
            return Collections.emptySet();
        } else {
            return EnumSet.copyOf(set);
        }
    }

    public BlackboxUserinfoRequest adjustRequest(
        final BlackboxUserinfoRequest request)
    {
        return request
            .sid(sid)
            .emailsType(emailsType)
            .dbfields(dbfields)
            .requiredDbfields(requiredDbfields)
            .aliases(aliases)
            .attributes(attributes)
            .getFamilyInfo(getFamilyInfo);
    }
}

