package ru.yandex.mail.so.factors.blackbox;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.logging.Level;

import com.google.protobuf.Int64Value;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.blackbox.BlackboxAttributeType;
import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxDbfield;
import ru.yandex.blackbox.BlackboxNotFoundException;
import ru.yandex.blackbox.BlackboxUserIdType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.blackbox.BlackboxUserinfos;
import ru.yandex.collection.IntList;
import ru.yandex.collection.LongList;
import ru.yandex.detect.locale.LocaleDetector;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.so.api.v1.Email;
import ru.yandex.mail.so.api.v1.EmailInfo;
import ru.yandex.mail.so.api.v1.SmtpEnvelope;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorFactoryContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.BooleanSoFactorType;
import ru.yandex.mail.so.factors.types.LongSoFactorType;
import ru.yandex.mail.so.factors.types.SmtpEnvelopeSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.StringSoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.mail.envelope.SmtpEnvelopeHolder;
import ru.yandex.stater.GolovanPanelConfig;

public class BlackboxUserinfosExtractor
    extends BlackboxExtractorBase
    implements SoFactorsExtractor
{
    private static final Function<Object, IntList> INT_LIST_FACTORY =
        x -> new IntList(1);

    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
    private static final List<SoFactorType<?>> OUTPUTS =
        Arrays.asList(
            SmtpEnvelopeSoFactorType.SMTP_ENVELOPE,
            BlackboxUserinfosMapSoFactorType.USERINFOS_MAP,
            BlackboxUserinfoSoFactorType.USERINFO,
            StringSoFactorType.STRING,
            BooleanSoFactorType.BOOLEAN,
            LongSoFactorType.LONG);
    private static final List<SoFactor<?>> NULL_RESULT =
        Arrays.asList(null, null, null, null, null, null);

    private static final BlackboxUserinfos RCPTTO_NOT_FOUND =
        new BlackboxUserinfos(
            Collections.singletonList(new BlackboxUserinfo(0L)));
    private static final BlackboxUserinfos BLACKBOX_ERROR =
        new BlackboxUserinfos(
            Collections.singletonList(new BlackboxUserinfo(0L)));

    private final Consumer<BlackboxStat> statsConsumer;
    private final Consumer<BlackboxStat> mailfromStatsConsumer;

    public BlackboxUserinfosExtractor(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        super(name, context, config);

        BlackboxStater blackboxStater = new BlackboxStater(
            GolovanPanelConfig.CATEGORY_UPSTREAMS,
            null,
            name + "-blackbox-userinfo-");
        context.registry().statersRegistrar().registerStater(blackboxStater);
        statsConsumer = blackboxStater.consumer();

        BlackboxStater blackboxMailfromStater = new BlackboxStater(
            GolovanPanelConfig.CATEGORY_UPSTREAMS,
            null,
            "blackbox-mailfrom-");
        context.registry().statersRegistrar().registerStater(
            blackboxMailfromStater);
        mailfromStatsConsumer = blackboxMailfromStater.consumer();
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    private void spawnLoginRequest(
        final BlackboxClient client,
        final String login,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<Map.Entry<Object, BlackboxUserinfos>> callback,
        final PrefixedLogger logger)
    {
        client.userinfo(
            adjustRequest(
                new BlackboxUserinfoRequest(BlackboxUserIdType.LOGIN, login)),
            contextGenerator,
            new ErrorClassifyingCallback(callback, login, logger));
    }

    private void spawnUidRequests(
        final Map<Long, IntList> uids,
        final BlackboxClient client,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final MultiFutureCallback<Map.Entry<Object, BlackboxUserinfos>>
            callback,
        final PrefixedLogger logger)
    {
        int uidsSize = uids.size();
        if (uidsSize > 0) {
            LongList uidsList = new LongList(uidsSize);
            Set<Long> uidsSet = uids.keySet();
            uidsList.addAll(uidsSet);
            BlackboxUserinfoRequest request;
            if (uidsSize == 1) {
                request = new BlackboxUserinfoRequest(uidsList.get(0));
            } else {
                long first = uidsList.remove(uidsSize - 1);
                request = new BlackboxUserinfoRequest(
                    first,
                    uidsList.toLongArray());
            }
            client.userinfo(
                adjustRequest(request).allowNotFound(true),
                contextGenerator,
                new ErrorClassifyingCallback(
                    callback.newCallback(),
                    uidsSet,
                    logger));
        }
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        SmtpEnvelopeHolder envelope =
            inputs.get(0, SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
        if (envelope == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        List<EmailInfo> recipients = envelope.recipients();
        int size = recipients.size();
        long incompleteCount = 0L;
        // Maps logins to their pos in recipients list
        Map<String, IntList> logins = new HashMap<>(size << 1);
        Map<String, IntList> corpLogins = new HashMap<>(size << 1);
        // Maps uids to their pos in recipients list
        Map<Long, IntList> uids = new LinkedHashMap<>(size << 1);
        Map<Long, IntList> corpUids = new LinkedHashMap<>(size << 1);
        for (int i = 0; i < size; ++i) {
            EmailInfo recipient = recipients.get(i);
            if (recipient.hasUid()) {
                long uid = recipient.getUid().getValue();
                if (BlackboxUserinfo.corp(uid)) {
                    corpUids.computeIfAbsent(uid, INT_LIST_FACTORY).add(i);
                } else {
                    uids.computeIfAbsent(uid, INT_LIST_FACTORY).add(i);
                }
            } else if (recipient.hasAddress()) {
                Email email = recipient.getAddress();
                String local = email.getNormalizedLocal();
                String domain = email.getNormalizedDomain();
                if (local.isEmpty() || domain.isEmpty()) {
                    ++incompleteCount;
                    context.logger().warning(
                        "Incomplete RCPTTO: " + recipient);
                } else {
                    String login = email.getNormalizedEmail();
                    if (domain.equals("yandex-team.ru")) {
                        corpLogins.computeIfAbsent(login, INT_LIST_FACTORY)
                            .add(i);
                    } else {
                        logins.computeIfAbsent(login, INT_LIST_FACTORY).add(i);
                    }

                }
            } else {
                ++incompleteCount;
                context.logger().warning("Empty RCPTTO: " + recipient);
            }
        }
        long incompleteFrom;
        String fromString;
        boolean corpFrom;
        long fromUid = envelope.fromUid();
        if (fromUid == 0) {
            Email from = envelope.from();
            if (from == null) {
                corpFrom = false;
                incompleteFrom = 1L;
                fromString = null;
                context.logger().warning("MAILFROM is missing in envelope");
            } else {
                String local = from.getNormalizedLocal();
                String domain = from.getNormalizedDomain();
                if (local.isEmpty() || domain.isEmpty()) {
                    incompleteFrom = 1L;
                    context.logger().warning("Incomplete MAILFROM: " + from);
                } else {
                    incompleteFrom = 0L;
                }
                fromString = from.getNormalizedEmail();
                corpFrom = domain.equals("yandex-team.ru");
            }
        } else {
            incompleteFrom = 0L;
            fromString = null;
            corpFrom = BlackboxUserinfo.corp(fromUid);
            if (corpFrom) {
                corpUids.computeIfAbsent(fromUid, INT_LIST_FACTORY).add(-1);
            } else {
                uids.computeIfAbsent(fromUid, INT_LIST_FACTORY).add(-1);
            }
        }

        PrefixedLogger logger = context.logger();
        MultiFutureCallback<Map.Entry<Object, BlackboxUserinfos>>
            multiCallback =
                new MultiFutureCallback<>(
                    new Callback(
                        callback,
                        context,
                        envelope,
                        logins,
                        corpLogins,
                        uids,
                        corpUids,
                        fromString,
                        statsConsumer,
                        mailfromStatsConsumer,
                        incompleteCount,
                        incompleteFrom));

        if ((fromString != null && !corpFrom)
            || logins.size() + uids.size() > 0)
        {
            BlackboxClient client =
                this.blackboxClient.adjust(context.httpContext());
            Supplier<? extends HttpClientContext> contextGenerator =
                context.requestsListener().createContextGeneratorFor(client);
            for (String login: logins.keySet()) {
                spawnLoginRequest(
                    client,
                    login,
                    contextGenerator,
                    multiCallback.newCallback(),
                    logger);
            }

            spawnUidRequests(
                uids,
                client,
                contextGenerator,
                multiCallback,
                logger);

            if (fromString != null && !corpFrom) {
                spawnLoginRequest(
                    client,
                    fromString,
                    contextGenerator,
                    multiCallback.newCallback(),
                    logger);
            }
        }

        if (corpFrom || corpLogins.size() + corpUids.size() > 0) {
            BlackboxClient client =
                this.corpBlackboxClient.adjust(context.httpContext());
            Supplier<? extends HttpClientContext> contextGenerator =
                context.requestsListener().createContextGeneratorFor(client);
            for (String login: corpLogins.keySet()) {
                spawnLoginRequest(
                    client,
                    login,
                    contextGenerator,
                    multiCallback.newCallback(),
                    logger);
            }

            spawnUidRequests(
                corpUids,
                client,
                contextGenerator,
                multiCallback,
                logger);

            if (fromString != null && corpFrom) {
                spawnLoginRequest(
                    client,
                    fromString,
                    contextGenerator,
                    multiCallback.newCallback(),
                    logger);
            }
        }

        multiCallback.done();
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        BlackboxUserinfosExtractorFactory.INSTANCE.registerInternals(registry);
    }

    private static class ErrorClassifyingCallback
        extends AbstractFilterFutureCallback<
            BlackboxUserinfos,
            Map.Entry<Object, BlackboxUserinfos>>
    {
        private final Object login;
        private final PrefixedLogger logger;

        ErrorClassifyingCallback(
            final FutureCallback<Map.Entry<Object, BlackboxUserinfos>>
                callback,
            final Object login,
            final PrefixedLogger logger)
        {
            super(callback);
            this.login = login;
            this.logger = logger;
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof BlackboxNotFoundException) {
                logger.warning("Login not found <" + login + '>');
                callback.completed(Map.entry(login, RCPTTO_NOT_FOUND));
            } else {
                logger.log(
                    Level.WARNING,
                    "Failed to resolve login <" + login + '>',
                    e);
                callback.completed(Map.entry(login, BLACKBOX_ERROR));
            }
        }

        @Override
        public void completed(final BlackboxUserinfos userinfos) {
            if (userinfos.isEmpty()) {
                logger.warning("Empty userinfos for <" + login + '>');
            }
            callback.completed(Map.entry(login, userinfos));
        }
    }

    private static LongList extractOrgIds(
        final BlackboxUserinfo userinfo,
        final PrefixedLogger logger)
    {
        try {
            JsonObject obj =
                userinfo.attributes().get(
                    BlackboxAttributeType.ACCOUNT_CONNECT_ORGANIZATION_IDS);
            if (obj == null) {
                return null;
            }
            JsonList orgIds = obj.asList();
            int size = orgIds.size();
            if (size == 0) {
                return null;
            }
            LongList result = new LongList(size);
            for (int i = 0; i < size; ++i) {
                result.addLong(orgIds.get(i).asLong());
            }
            return result;
        } catch (JsonBadCastException e) {
            logger.log(
                Level.WARNING,
                "Failed to extract org ids from " + userinfo.toString(),
                e);
            return null;
        }
    }

    private static class Callback extends AbstractFilterFutureCallback<
        List<Map.Entry<Object, BlackboxUserinfos>>,
        List<SoFactor<?>>>
    {
        private final SoFactorsExtractorContext context;
        private final SmtpEnvelopeHolder envelope;
        private final Map<String, IntList> logins;
        private final Map<String, IntList> corpLogins;
        private final Map<Long, IntList> uids;
        private final Map<Long, IntList> corpUids;
        private final String fromLogin;
        private final Consumer<BlackboxStat> statsConsumer;
        private final Consumer<BlackboxStat> mailfromStatsConsumer;
        private final long incompleteCount;
        private final long incompleteFrom;

        Callback(
            final FutureCallback<? super List<SoFactor<?>>> callback,
            final SoFactorsExtractorContext context,
            final SmtpEnvelopeHolder envelope,
            final Map<String, IntList> logins,
            final Map<String, IntList> corpLogins,
            final Map<Long, IntList> uids,
            final Map<Long, IntList> corpUids,
            final String fromLogin,
            final Consumer<BlackboxStat> statsConsumer,
            final Consumer<BlackboxStat> mailfromStatsConsumer,
            final long incompleteCount,
            final long incompleteFrom)
        {
            super(callback);
            this.context = context;
            this.envelope = envelope;
            this.logins = logins;
            this.corpLogins = corpLogins;
            this.uids = uids;
            this.corpUids = corpUids;
            this.fromLogin = fromLogin;
            this.statsConsumer = statsConsumer;
            this.mailfromStatsConsumer = mailfromStatsConsumer;
            this.incompleteCount = incompleteCount;
            this.incompleteFrom = incompleteFrom;
        }

        @SuppressWarnings("ReferenceEquality")
        @Override
        public void completed(
            final List<Map.Entry<Object, BlackboxUserinfos>> results)
        {
            List<EmailInfo> recipients = envelope.recipients();
            SmtpEnvelope.Builder envelopeBuilder =
                envelope.envelope().toBuilder();
            envelopeBuilder.clearRecipients();
            long empty = 0L;
            long notFound = 0L;
            long errors = 0L;
            long success = 0L;
            long fromEmpty = 0L;
            long fromNotFound = 0L;
            long fromErrors = 0L;
            int size = recipients.size();
            BlackboxUserinfos userinfos = new BlackboxUserinfos(size);
            BlackboxUserinfosMap userinfosMap = new BlackboxUserinfosMap(size);
            for (int i = 0; i < size; ++i) {
                userinfos.add(null);
                EmailInfo recipient = recipients.get(i);
                if (recipient.hasAddress()) {
                    userinfosMap.put(recipient.getAddress().getEmail(), null);
                }
            }
            BlackboxUserinfo fromUserinfo = null;
            for (Map.Entry<Object, BlackboxUserinfos> entry: results) {
                Object key = entry.getKey();
                BlackboxUserinfos resultUserinfos = entry.getValue();
                int resultsSize = resultUserinfos.size();
                boolean isFrom = key.equals(fromLogin);
                if (isFrom) {
                    if (resultsSize == 0) {
                        ++fromEmpty;
                    } else if (resultUserinfos == RCPTTO_NOT_FOUND) {
                        ++fromNotFound;
                    } else if (resultUserinfos == BLACKBOX_ERROR) {
                        ++fromErrors;
                    } else {
                        fromUserinfo = resultUserinfos.get(0);
                    }
                }
                if (resultsSize == 0) {
                    ++empty;
                } else if (resultUserinfos == RCPTTO_NOT_FOUND) {
                    ++notFound;
                } else if (resultUserinfos == BLACKBOX_ERROR) {
                    ++errors;
                } else {
                    for (int i = 0; i < resultsSize; ++i) {
                        ++success;
                        BlackboxUserinfo userinfo = resultUserinfos.get(i);
                        Object login = entry.getKey();
                        IntList positions = logins.get(login);
                        if (positions == null) {
                            long uid = userinfo.uid();
                            positions = uids.get(uid);
                            if (positions == null) {
                                positions = corpLogins.get(login);
                                if (positions == null) {
                                    positions = corpUids.get(uid);
                                }
                            }
                        }
                        if (positions == null) {
                            if (!isFrom) {
                                context.logger().warning(
                                    "Unexpected userinfo found: " + userinfo);
                            }
                        } else {
                            int positionsSize = positions.size();
                            for (int j = 0; j < positionsSize; ++j) {
                                int pos = positions.getInt(j);
                                if (pos == -1) {
                                    fromUserinfo = userinfo;
                                } else {
                                    userinfos.set(pos, userinfo);
                                    EmailInfo recipient = recipients.get(pos);
                                    if (recipient.hasAddress()) {
                                        userinfosMap.put(
                                            recipient.getAddress().getEmail(),
                                            userinfo);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            boolean recipientsFromSameOrgId = true;
            Long commonOrgId = null;
            for (int i = 0; i < size; ++i) {
                EmailInfo.Builder recipientBuilder =
                    recipients.get(i).toBuilder();
                BlackboxUserinfo userinfo = userinfos.get(i);
                if (userinfo == null) {
                    recipientBuilder.clearUid();
                    recipientsFromSameOrgId = false;
                } else {
                    recipientBuilder.setUid(
                        Int64Value.newBuilder()
                            .setValue(userinfo.uid())
                            .build());
                    if (recipientsFromSameOrgId) {
                        try {
                            Long orgId =
                                userinfo.attributes().getOrDefault(
                                    BlackboxAttributeType.ACCOUNT_ORG_ID,
                                    JsonNull.INSTANCE)
                                    .asLongOrNull();
                            if (orgId == null) {
                                recipientsFromSameOrgId = false;
                            } else if (commonOrgId == null) {
                                commonOrgId = orgId;
                            } else {
                                long common = commonOrgId.longValue();
                                recipientsFromSameOrgId =
                                    orgId.longValue() == common;
                            }
                        } catch (JsonBadCastException e) {
                            context.logger().log(
                                Level.WARNING,
                                "Failed to extract org id from "
                                + userinfo.toString(),
                                e);
                        }
                    }
                    String suidString =
                        userinfo.dbfields().get(BlackboxDbfield.SUID);
                    if (suidString != null && !suidString.isEmpty()) {
                        try {
                            recipientBuilder.setSuid(
                                Int64Value.newBuilder()
                                    .setValue(Long.parseLong(suidString))
                                    .build());
                        } catch (RuntimeException e) {
                            context.logger().log(
                                Level.WARNING,
                                "Failed to parse suid <" + suidString + '>',
                                e);
                        }
                    }
                }
                envelopeBuilder.addRecipients(recipientBuilder.build());
            }
            statsConsumer.accept(
                new BlackboxStat(
                    notFound,
                    incompleteCount,
                    empty,
                    errors,
                    success));
            mailfromStatsConsumer.accept(
                new BlackboxStat(
                    fromNotFound,
                    incompleteFrom,
                    fromEmpty,
                    fromErrors,
                    fromUserinfo == null ? 0L : 1L));
            List<SoFactor<?>> factors = new ArrayList<>(5);
            factors.add(
                SmtpEnvelopeSoFactorType.SMTP_ENVELOPE.createFactor(
                    new SmtpEnvelopeHolder(envelopeBuilder.build())));
            factors.add(
                BlackboxUserinfosMapSoFactorType.USERINFOS_MAP.createFactor(
                    userinfosMap));
            if (fromUserinfo == null) {
                factors.add(null);
            } else {
                factors.add(
                    BlackboxUserinfoSoFactorType.USERINFO.createFactor(
                        fromUserinfo));
            }
            Email from = envelope.from();
            String fromString = null;
            if (from == null) {
                if (fromUserinfo != null) {
                    fromString =
                        LocaleDetector.INSTANCE.toLowerCase(
                            fromUserinfo.login());
                    if (fromString.indexOf('@') == -1) {
                        if (fromUserinfo.corp()) {
                            fromString += "@yandex-team.ru";
                        } else {
                            fromString += "@yandex.ru";
                        }
                    }
                }
            } else {
                fromString = from.getEmail();
            }

            if (fromString == null) {
                factors.add(null);
            } else {
                factors.add(
                    StringSoFactorType.STRING.createFactor(fromString));
            }

            boolean allFromSameOrgId =
                fromUserinfo != null
                && incompleteCount == 0
                && empty == 0
                && notFound == 0
                && errors == 0;
            if (allFromSameOrgId) {
                LongList fromOrgIds =
                    extractOrgIds(fromUserinfo, context.logger());
                allFromSameOrgId = fromOrgIds != null;
                if (allFromSameOrgId) {
                    for (BlackboxUserinfo userinfo: userinfosMap.values()) {
                        LongList orgIds =
                            extractOrgIds(userinfo, context.logger());
                        if (orgIds == null) {
                            allFromSameOrgId = false;
                            break;
                        }
                        size = orgIds.size();
                        boolean hasCommonOrg = false;
                        for (int i = 0; i < size && !hasCommonOrg; ++i) {
                            hasCommonOrg =
                                fromOrgIds.containsLong(orgIds.getLong(i));
                        }
                        if (!hasCommonOrg) {
                            allFromSameOrgId = false;
                            break;
                        }
                    }
                }
            }
            factors.add(
                BooleanSoFactorType.BOOLEAN.createFactor(allFromSameOrgId));
            if (recipientsFromSameOrgId && commonOrgId != null) {
                factors.add(LongSoFactorType.LONG.createFactor(commonOrgId));
            } else {
                factors.add(null);
            }
            callback.completed(factors);
        }
    }
}

