package ru.yandex.mail.so.factors.dsl;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import core.org.luaj.vm2.Globals;
import core.org.luaj.vm2.LuaClosure;

import ru.yandex.mail.so.factors.SoFunctionArgumentInfo;
import ru.yandex.mail.so.factors.extractors.ChainExtractor;
import ru.yandex.mail.so.factors.extractors.LuaExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorFactory;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorFactoryContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.predicates.LuaPredicate;
import ru.yandex.mail.so.factors.predicates.SoPredicate;
import ru.yandex.mail.so.factors.predicates.SoPredicateType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.util.timesource.TimeSource;

public abstract class DslParserBase {
    private final List<ChainExtractorContext> contexts = new ArrayList<>();
    protected final StringBuilder buffer = new StringBuilder();
    protected final List<String> variables = new ArrayList<>();
    protected final List<SoFunctionArgumentInfo> arguments = new ArrayList<>();
    protected final List<SoCallArgument> callArguments =
        new ArrayList<>();
    protected boolean criticalDefault = false;
    protected SoFactorType<?> argumentType;

    protected SoPredicateType predicateType;
    protected String predicateName;

    protected SoFactorsExtractorFactory extractorType;
    protected String extractorName;

    protected String callName;
    protected String alias;
    protected boolean trace;
    protected boolean async;
    protected boolean critical;

    protected List<SoCallArgument> returns;
    protected List<SoCallArgument> predicateArguments = new ArrayList<>();
    protected List<SoCallArgument> elses = Collections.emptyList();
    protected boolean invertPredicate;
    protected String returnPredicate;

    protected DslParserBase(final SoFactorsExtractorFactoryContext context) {
        contexts.add(new ChainExtractorContext(null, null, null, context));
    }

    private ChainExtractorContext chainContext() {
        return contexts.get(contexts.size() - 1);
    }

    protected SoFactorsExtractorFactoryContext context() {
        return chainContext().factoryContext;
    }

    public void parse(final BufferedReader reader)
        throws ConfigException, IOException
    {
        int lineNumber = 0;
        while (true) {
            String line = reader.readLine();
            if (line == null) {
                break;
            }
            ++lineNumber;
            try {
                processLine(line + '\n');
            } catch (ConfigException e) {
                throw new ConfigException(
                    "Failed to parse line #" + lineNumber + ": " + line,
                    e);
            }
        }
    }

    protected SoFactorType<?> parseType(
        final String typeName,
        final String typeType,
        final int pos)
        throws ConfigException
    {
        SoFactorType<?> type =
            context().registry().typesRegistry().getFactorType(typeName);
        if (type == null) {
            throw new ConfigException(
                "Unknown function " + typeType + " type <" + typeName + '>');
        }
        return type;
    }

    private List<SoFactorType<?>> parseOutputs(final List<String> outputNames)
        throws ConfigException
    {
        int size = outputNames.size();
        List<SoFactorType<?>> outputs = new ArrayList<>(size);
        for (int i = 0; i < size; ++i) {
            outputs.add(parseType(outputNames.get(i), "output", i));
        }
        return outputs;
    }

    protected abstract void processLine(String line)
        throws ConfigException, IOException;

    protected void commitPredicateType() throws ConfigException {
        String typeName = buffer.toString();
        predicateType = context().registry().getPredicateType(typeName);
        if (predicateType == null) {
            throw new ConfigException(
                "Unknown predicate type <" + typeName + '>');
        }
    }

    protected void commitPredicate() throws ConfigException, IOException {
        SoFactorsExtractorFactoryContext context = context();
        context.logger().info(
            "Creating predicate " + predicateName
            + " with " + predicateType.getClass().getSimpleName());
        long start = TimeSource.INSTANCE.currentTimeMillis();
        SoPredicate predicate = predicateType.create(
            arguments,
            context,
            buffer.toString());
        context.registry().registerPredicate(predicateName, predicate);
        long end = TimeSource.INSTANCE.currentTimeMillis();
        context.logger().info(
            "Predicate " + predicateName
            + " created in " + (end - start) + " ms");
    }

    protected void commitExtractorType() throws ConfigException {
        String typeName = buffer.toString();
        if (typeName.equals("chain")) {
            // chain extractors handled separately
            extractorType = null;
        } else {
            extractorType = context().registry().getExtractorFactory(typeName);
            if (extractorType == null) {
                throw new ConfigException(
                    "Unknown extractor type <" + typeName + '>');
            }
        }
    }

    protected void commitSimpleExtractor()
        throws ConfigException, IOException
    {
        SoFactorsExtractorFactoryContext context = context();
        context.logger().info(
            "Creating extractor " + extractorName
            + " with " + extractorType.getClass().getSimpleName());
        long start = TimeSource.INSTANCE.currentTimeMillis();
        SoFactorsExtractor extractor = extractorType.createExtractor(
            extractorName,
            arguments,
            parseOutputs(variables),
            context,
            buffer.toString());
        context.registry().registerExtractor(extractorName, extractor);
        long end = TimeSource.INSTANCE.currentTimeMillis();
        context.logger().info(
            "Extractor " + extractorName
            + " created in " + (end - start) + " ms");
    }

    protected void pushChainExtractor() throws ConfigException {
        SoFactorsExtractorFactoryContext context = context();
        contexts.add(
            new ChainExtractorContext(
                extractorName,
                new ArrayList<>(arguments),
                parseOutputs(variables),
                new SoFactorsExtractorFactoryContext(
                    context.path(),
                    new SoFactorsExtractorsRegistry(context.registry()),
                    context.statsConsumerFactory(),
                    context.violationsCounter(),
                    context.luaErrorsCounter(),
                    context.threadGroup(),
                    context.asyncClientRegistrar(),
                    context.externalDataProvider(),
                    context.modulesCache(),
                    context.luaModulesCache(),
                    context.logger().addPrefix(extractorName),
                    context.samplesNotifier(),
                    context.metricsTimeFrame())));
    }

    protected void popChainExtractor() throws ConfigException {
        ChainExtractorContext context = contexts.remove(contexts.size() - 1);
        context().registry().registerExtractor(
            context.name,
            new ChainExtractor(
                context.name,
                context.arguments,
                context.outputs,
                context.calls,
                context.returns,
                context.factoryContext));
    }

    protected void commitCall() {
        ChainExtractorContext context = chainContext();
        context.calls.add(
            new SoCallDeclaration(
                callName,
                alias,
                trace,
                async,
                critical,
                new ArrayList<>(callArguments),
                new ArrayList<>(variables)));
        alias = null;
        trace = false;
        async = false;
        critical = criticalDefault;
    }

    protected void commitReturnPredicate() throws ConfigException {
        returnPredicate = buffer.toString();
    }

    protected void commitReturn() throws ConfigException {
        if (returnPredicate != null
            && elses.size() != 0
            && returns.size() != elses.size())
        {
            throw new ConfigException(
                "Returns mismatch on predicate <" + returnPredicate
                + ">, returns " + returns + " doesn't match elses " + elses);
        }
        ChainExtractorContext context = chainContext();
        context.returns.add(
            new SoReturnDeclaration(
                returns,
                elses,
                returnPredicate,
                invertPredicate,
                predicateArguments));
        returns = new ArrayList<>();
        elses = Collections.emptyList();
        predicateArguments = new ArrayList<>();
        returnPredicate = null;
        invertPredicate = false;
    }

    protected void commitImport(final String fileName)
        throws ConfigException, IOException
    {
        Path path = Paths.get(fileName);
        Path parentPath = context().path();
        if (parentPath != null) {
            path = parentPath.resolveSibling(path);
        }
        SoFactorsExtractorFactoryContext context = context();
        if (fileName.endsWith(".lua")) {
            Globals globals = context.loadLuaModule(path);
            for (String functionName: variables) {
                LuaClosure closure;
                try {
                    closure = globals.get(functionName).checkclosure();
                } catch (RuntimeException e) {
                    throw new ConfigException(
                        "Can't load function <" + functionName
                        + "> from lua module " + path,
                        e);
                }
                context.registry().registerExtractor(
                    functionName,
                    new LuaExtractor(closure, context.luaErrorsCounter()));
                context.registry().registerPredicate(
                    functionName,
                    new LuaPredicate(closure, context.luaErrorsCounter()));
            }
        } else {
            SoFactorsExtractorsRegistry localRegistry =
                context.loadModule(path);
            for (String extractorName: variables) {
                SoFactorsExtractor extractor =
                    localRegistry.getExtractor(extractorName);
                if (extractor == null) {
                    throw new ConfigException(
                        "Extractor <" + extractorName
                        + "> was not found in module " + path);
                }
                context.registry().registerExtractor(extractorName, extractor);
            }
        }
    }

    protected void reportError(final char c, final int p)
        throws ConfigException
    {
        throw new ConfigException(
            "Unexpected character <" + c + "> at pos: " + p);
    }

    private static class ChainExtractorContext {
        private final List<SoCallDeclaration> calls = new ArrayList<>();
        private final List<SoReturnDeclaration> returns = new ArrayList<>();
        private final String name;
        private final List<SoFunctionArgumentInfo> arguments;
        private final List<SoFactorType<?>> outputs;
        private final SoFactorsExtractorFactoryContext factoryContext;

        ChainExtractorContext(
            final String name,
            final List<SoFunctionArgumentInfo> arguments,
            final List<SoFactorType<?>> outputs,
            final SoFactorsExtractorFactoryContext factoryContext)
        {
            this.name = name;
            this.arguments = arguments;
            this.outputs = outputs;
            this.factoryContext = factoryContext;
        }
    }
}

