package ru.yandex.mail.so.factors.dsl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.mail.so.factors.IdentitySoFactorFieldAccessor;
import ru.yandex.mail.so.factors.Remapping;
import ru.yandex.mail.so.factors.SoFactorFieldAccessor;
import ru.yandex.mail.so.factors.types.NullSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.xpath.XPathParser;

public class SoVariables {
    private final SoConsts consts;
    private final Map<String, SoVariableInfo> variables;

    public SoVariables(final SoConsts consts) {
        this.consts = consts;
        variables = new HashMap<>();
    }

    public SoVariables(final SoVariables other) {
        consts = other.consts;
        variables = new HashMap<>(other.variables);
    }

    public int size() {
        return consts.size() + variables.size();
    }

    public SoVariableInfo getVariableInfo(
        final String variableName,
        final String description)
        throws ConfigException
    {
        SoVariableInfo info = variables.get(variableName);
        if (info == null) {
            throw new ConfigException(
                description + " variable <"
                + variableName + "> not found");
        }
        return info;
    }

    public SoFactorType<?> getVariableType(
        final String variableName,
        final String description)
        throws ConfigException
    {
        return getVariableInfo(variableName, description).type();
    }

    public int getVariableId(
        final String variableName,
        final String description)
        throws ConfigException
    {
        return getVariableInfo(variableName, description).id();
    }

    public Remapping createRemapping(
        final List<SoCallArgument> arguments,
        final String description)
        throws ConfigException
    {
        int size = arguments.size();
        int[] remapping = new int[size];
        List<SoFactorFieldAccessor> accessors = new ArrayList<>(size);
        for (int i = 0; i < size; ++i) {
            SoCallArgument argument = arguments.get(i);
            if (argument.type() == SoCallArgument.Type.TOKEN) {
                ValueInfo info = new ValueInfo(this, description);
                XPathParser.parse('.' + argument.image(), info);
                remapping[i] = info.variableInfo().id();
                SoFactorFieldAccessor fieldAccessor = info.fieldAccessor();
                if (fieldAccessor instanceof IdentitySoFactorFieldAccessor) {
                    accessors.add(null);
                } else {
                    accessors.add(fieldAccessor);
                }
            } else {
                remapping[i] = consts.getConst(argument).id();
                accessors.add(null);
            }
        }
        return new Remapping(remapping, accessors);
    }

    public String[] variableIdToName() {
        String[] variableIdToName = new String[size()];
        for (Map.Entry<String, SoVariableInfo> entry: variables.entrySet()) {
            variableIdToName[entry.getValue().id()] = entry.getKey();
        }
        return variableIdToName;
    }

    public void addVariable(
        final String variableName,
        final SoFactorType<?> variableType,
        final String description)
        throws ConfigException
    {
        SoVariableInfo oldInfo =
            variables.putIfAbsent(
                variableName,
                new SoVariableInfo(size(), variableType));
        if (oldInfo != null) {
            throw new ConfigException(
                description + " variable <" + variableName
                + "> can't be registered with type " + variableType
                + " as it already registered with type " + oldInfo.type());
        }
    }

    public SoFactorType<?> getArgumentType(
        final SoCallArgument argument,
        final String description)
        throws ConfigException
    {
        if (argument.type() == SoCallArgument.Type.TOKEN) {
            ValueInfo info = new ValueInfo(this, description);
            XPathParser.parse('.' + argument.image(), info);
            return info.fieldAccessor().fieldType();
        } else {
            return consts.getConst(argument).value().type();
        }
    }

    public void checkArguments(
        final List<SoCallArgument> arguments,
        final List<SoFactorType<?>> expectedTypes,
        final String description)
        throws ConfigException
    {
        int size = expectedTypes.size();
        if (size != arguments.size()) {
            throw new ConfigException(
                description + " has invalid arguments count. Expected "
                + size + " arguments with types " + expectedTypes
                + ", found " + arguments.size() + " with names "
                + arguments);
        }
        for (int i = 0; i < size; ++i) {
            SoCallArgument argument = arguments.get(i);
            SoFactorType<?> inputType = getArgumentType(argument, description);
            SoFactorType<?> expectedType = expectedTypes.get(i);
            if (inputType != NullSoFactorType.NULL
                && inputType != expectedType)
            {
                throw new ConfigException(
                    description + " argument " + argument
                    + " type mismatch, expected type: " + expectedType
                    + ", actual type: " + inputType);
            }
        }
    }
}

