package ru.yandex.mail.so.factors.dssm;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.PositionSavingContainerFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.TikaiteDocSoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.so.dssm.applier.DssmApplier;
import ru.yandex.so.dssm.applier.DssmApplierException;

public class DssmEmbeddingExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(TikaiteDocSoFactorType.TIKAITE_DOC);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(DssmEmbeddingSoFactorType.DSSM_EMBEDDING);

    private final DssmApplier dssmApplier;
    // Maps from TikaiteDoc fields to dssm fields
    // Values may be not unique, order is from fallback field to primary field
    private final Map<String, String> fieldsMapping;
    private final Set<String> dssmFields;

    public DssmEmbeddingExtractor(final String name, final IniConfig config)
        throws ConfigException
    {
        ImmutableDssmExtractorConfig dssmExtractorConfig =
            new DssmExtractorConfigBuilder(
                new DssmExtractorConfigBuilder(config).build())
                .build();

        try {
            dssmApplier = new DssmApplier(
                dssmExtractorConfig.model().getAbsolutePath(),
                dssmExtractorConfig.layer(),
                dssmExtractorConfig.preprocessFields(),
                false);
        } catch (DssmApplierException e) {
            throw new ConfigException(
                "Failed to construct dssm applier for <" + name + '>',
                e);
        }
        fieldsMapping = dssmExtractorConfig.fieldsMapping();
        dssmFields = new LinkedHashSet<>(fieldsMapping.values());
    }

    @Override
    public void close() {
        dssmApplier.close();
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        JsonMap doc = inputs.get(0, TikaiteDocSoFactorType.TIKAITE_DOC);
        if (doc == null || doc.isEmpty()) {
            callback.completed(NULL_RESULT);
            return;
        }
        JsonMap dssmData =
            new JsonMap(PositionSavingContainerFactory.INSTANCE);
        for (Map.Entry<String, String> entry: fieldsMapping.entrySet()) {
            String field = entry.getKey();
            JsonObject value = doc.get(field);
            if (value.type() == JsonObject.Type.STRING) {
                dssmData.put(entry.getValue(), value);
            } else if (context.logger().isLoggable(Level.INFO)) {
                StringBuilder sb = new StringBuilder("Ignoring field <");
                sb.append(field);
                sb.append("> with value: ");
                JsonType.NORMAL.toStringBuilder(sb, value);
                context.logger().info(new String(sb));
            }
        }

        for (String field: dssmFields) {
            dssmData.putIfAbsent(field, JsonNull.INSTANCE);
        }

        String jsonString = JsonType.NORMAL.toString(dssmData);
        try {
            callback.completed(
                Collections.singletonList(
                    DssmEmbeddingSoFactorType.DSSM_EMBEDDING.createFactor(
                        dssmApplier.apply(jsonString))));
        } catch (DssmApplierException e) {
            callback.failed(e);
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        DssmEmbeddingExtractorFactory.INSTANCE.registerInternals(registry);
    }
}

